package com.digiwin.athena.dao.mongodao.assembly.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import com.digiwin.athena.bo.assembly.SolutionPlan;
import com.digiwin.athena.constants.enums.assembly.AssemblyType;
import com.digiwin.athena.constants.enums.assembly.OnlineState;
import com.digiwin.athena.convertor.assembly.SolutionConvertor;
import com.digiwin.athena.dao.mongodao.assembly.DDesignerMongoDao;
import com.digiwin.athena.dao.mongodao.assembly.PublishableContext;
import com.digiwin.athena.dao.mongodao.assembly.RSolutionMongoDao;
import com.digiwin.athena.dto.PageReqCondition;
import com.digiwin.athena.dto.Pagination;
import com.digiwin.athena.dto.assembly.solution.RSolutionPlanPageQo;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanListDTO;
import com.digiwin.athena.mongodb.domain.application.ApplicationConfig;
import com.digiwin.athena.mongodb.repository.MongoRuntimeRepositoryDecorator;
import com.digiwin.athena.mongodb.repository.MongoUserRepositoryDecorator;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Repository;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

@RequiredArgsConstructor
@Repository
public class RSolutionMongoDaoImpl implements RSolutionMongoDao {

    protected static final SolutionConvertor CONVERTOR = SolutionConvertor.INSTANCE;
    protected final MongoRuntimeRepositoryDecorator runtimeDecorator;
    protected DSolutionMongoDaoImpl dSolutionMongoDao;

    @Override
    public SolutionPlan selectById(String objectId) {
        return PublishableContext.withRuntime(() -> dSolutionMongoDao.selectById(objectId));
    }

    @Override
    public Pagination<SolutionPlanListDTO> selectOnlineSolutionPage(PageReqCondition<RSolutionPlanPageQo> pageParam) {
        Query query = new Query();

        // 未整合已有解决方案设计器前 先进行筛选，用来区分
        commonCondition(query, true);

        // 添加keyword查询条件
        RSolutionPlanPageQo condition = ObjectUtil.defaultIfNull(pageParam.getCondition(), RSolutionPlanPageQo::new);
        if (condition.getIsDynamic() != null) {
            Criteria dynamic = Criteria.where(ApplicationConfig.Fields.assemblyType);
            if (condition.getIsDynamic()) {
                dynamic.is(AssemblyType.Dynamic.getCode());
            } else {
                dynamic.is(AssemblyType.Static.getCode());
            }

            query.addCriteria(dynamic);
        }

        DDesignerMongoDao.addSearchNameOrType(query, condition.getSearchContent());

        return getDecorator()
                .findWithPage(query, ApplicationConfig.class, pageParam)
                .convert(SolutionConvertor.INSTANCE::entityToListDTO);
    }

    @Override
    public void save(SolutionPlan solution) {
        PublishableContext.withRuntime(() -> dSolutionMongoDao.save(solution));
    }

    @Override
    public boolean deleteById(String objectId) {
        return PublishableContext.withRuntime(() -> dSolutionMongoDao.deleteById(objectId));
    }

    @Override
    public List<SolutionPlanListDTO> selectByAppType(Collection<Integer> appTypeList, boolean onlyOnline) {
        Query query = new Query();

        if (CollUtil.isNotEmpty(appTypeList)) {
            query.addCriteria(Criteria.where(ApplicationConfig.Fields.appType).in(appTypeList));
        }

        commonCondition(query, onlyOnline);

        return getDecorator().find(query, ApplicationConfig.class).stream()
                .map(CONVERTOR::entityToListDTO)
                .collect(Collectors.toList());
    }

    @Override
    public SolutionPlan selectByAppType(Integer appType) {
        Query query = new Query(Criteria.where(ApplicationConfig.Fields.appType).is(appType));
        ApplicationConfig applicationConfig = getDecorator().findOne(query, ApplicationConfig.class);
        return CONVERTOR.toBO(applicationConfig);
    }

    protected void commonCondition(Query query, boolean onlyOnline) {
        // 未整合已有解决方案设计器前 先进行筛选，用来区分
        query.addCriteria(Criteria.where(ApplicationConfig.Fields.assemblyType).exists(true).ne(null));
        if (onlyOnline) {
            query.addCriteria(Criteria.where(ApplicationConfig.Fields.status).is(OnlineState.Online.getCode()));
        }
    }

    public MongoUserRepositoryDecorator getDecorator() {
        return runtimeDecorator;
    }

    @Autowired
    @Lazy
    public void setDSolutionMongoDao(DSolutionMongoDaoImpl dSolutionMongoDao) {
        this.dSolutionMongoDao = dSolutionMongoDao;
    }
}
