package com.digiwin.athena.service.assembly.impl;

import cn.hutool.core.net.url.UrlBuilder;
import com.digiwin.athena.bo.assembly.SolutionPlan;
import com.digiwin.athena.dao.mongodao.assembly.PublishableContext;
import com.digiwin.athena.dao.mongodao.assembly.RSolutionMongoDao;
import com.digiwin.athena.utils.expression.ExpressionResolver;
import com.digiwin.athena.dto.PageReqCondition;
import com.digiwin.athena.dto.Pagination;
import com.digiwin.athena.dto.assembly.solution.RSolutionPlanPageQo;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanDetailDTO;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanListDTO;
import com.digiwin.athena.service.assembly.RSolutionPlanService;
import com.digiwin.athena.service.assembly.RVariableService;
import com.digiwin.athena.utils.RequestShareUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import java.util.HashSet;
import java.util.List;
import java.util.Map;

/**
 * 运行时SolutionPlan列表服务实现
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class RSolutionPlanServiceImpl implements RSolutionPlanService {

    protected final RSolutionMongoDao rSolutionMongoDao;
    protected final RVariableService rVariableService;
    protected final ExpressionResolver expressionResolver;
    protected DSolutionPlanServiceImpl dSolutionPlanService;


    @Override
    public Pagination<SolutionPlanListDTO> list(PageReqCondition<RSolutionPlanPageQo> pageReqCondition) {
        return rSolutionMongoDao.selectOnlineSolutionPage(pageReqCondition);
    }

    @Override
    public List<SolutionPlanListDTO> findByAppTypeOrAll(List<Integer> appType) {
        return findByAppTypeOrAll(appType, true);
    }

    @Override
    public List<SolutionPlanListDTO> findByAppTypeOrAll(List<Integer> appType, boolean onlyOnline) {
        return rSolutionMongoDao.selectByAppType(new HashSet<>(appType), onlyOnline);
    }

    @Override
    public SolutionPlan findByAppType(Integer appType) {
        return rSolutionMongoDao.selectByAppType(appType);
    }

    @Override
    public Map<String, List<SolutionPlanListDTO>> findBaseListByDesignerId(List<String> designerIds) {
        return PublishableContext.withRuntime(() -> dSolutionPlanService.findBaseListByDesignerId(designerIds));
    }

    @Override
    public SolutionPlanDetailDTO findDetailById(String id) {
        return PublishableContext.withRuntime(() -> dSolutionPlanService.findDetailById(id));
    }

    @Override
    public String resolveSsoUrl(SolutionPlanDetailDTO.SsoConfigDTO ssoConfig) {
        if (ssoConfig == null) {
            return null;
        }

        Map<String, Object> contextMap = RequestShareUtil.computeIfAbsent("SolutionPlan_Sso_Context", k ->
                rVariableService.getRuntimeVarMap(DVariableServiceImpl.VarKey.SolutionPlan_Sso.key()));
        Map<String, Object> paramMap = ssoConfig.toParamMap(expressionResolver::wrapVarName);

        Map<String, Object> resolvedMap = expressionResolver.resolveMap(paramMap, contextMap);
        resolvedMap.forEach(contextMap::putIfAbsent);

        String url = expressionResolver.resolveToStr(ssoConfig.getUrl(), contextMap);

        // 使用hutool构建请求url，url为ssoConfig的url，param为resolvedMap
        UrlBuilder urlBuilder = UrlBuilder.ofHttp(url);
        resolvedMap.forEach(urlBuilder::addQuery);

        return urlBuilder.build();
    }

    @Autowired
    @Lazy
    public void setDSolutionPlanService(DSolutionPlanServiceImpl dSolutionPlanService) {
        this.dSolutionPlanService = dSolutionPlanService;
    }
}
