package com.digiwin.athena.datacollect.collector.impl;

import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.StrUtil;
import com.digiwin.athena.base.BusinessException;
import com.digiwin.athena.convertor.BaseConverter;
import com.digiwin.athena.datacollect.api.dto.AssetCollectApiRequest;
import com.digiwin.athena.datacollect.api.dto.AssetCollectApiResponse;
import com.digiwin.athena.datacollect.collector.BaseAssetDataCollector;
import com.digiwin.athena.datacollect.context.CollectContext;
import com.digiwin.athena.datacollect.context.HttpRequestContext;
import com.digiwin.athena.datacollect.model.CollectExecutorProp;
import com.digiwin.athena.datacollect.model.PageResult;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.retry.RetryCallback;
import org.springframework.retry.backoff.ExponentialBackOffPolicy;
import org.springframework.retry.backoff.FixedBackOffPolicy;
import org.springframework.retry.policy.SimpleRetryPolicy;
import org.springframework.retry.support.RetryTemplate;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.web.client.RestTemplate;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * TODO pzz refactor 解耦HttpCollector和Asset的关系，使用泛型或接口方式增强Collector的泛用性
 * TODO pzz refactor 使用工厂模式将组件与逻辑实现抽离，创建内部类，封装fetchPage逻辑，减少临时变量增强可读性
 * HTTP方式的资产数据采集器实现
 * 只负责HTTP请求的组装和发送，分页循环由父类控制
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class HttpAssetDataCollector extends BaseAssetDataCollector {

    protected final RestTemplate restTemplate;

    @Override
    public String getType() {
        return "http";
    }

    /**
     * 分页查询（只查询一页）
     */
    @Override
    protected PageResult<AssetCollectApiResponse.AssetDataItem> fetchPage(CollectContext collectContext) {
        CollectExecutorProp executor = collectContext.getConfig().getExecutor();

        try {
            HttpRequestContext requestContext = buildRequestContext(collectContext, executor);

            // 3. 执行请求前置脚本处理
            processRequestScript(executor, requestContext);

            // 4. 构建最终的HTTP Headers
            execute(executor, requestContext);

            // 7. 执行响应后置脚本处理并反序列化
            processResponseScript(executor, requestContext);

            return processResult(collectContext, requestContext, executor);
        } catch (Exception e) {
            throw new BusinessException(StrUtil.format("{}执行器失败!msg:{}", getType(), e.getMessage()), e);
        }
    }

    protected HttpRequestContext buildRequestContext(CollectContext collectContext, CollectExecutorProp executor) {
        HttpRequestContext requestContext = new HttpRequestContext(executor.getUrl(), collectContext);
        // 1. 构建HTTP Headers
        requestContext.setRequestHeader(new HashMap<>(MapUtil.emptyIfNull(executor.getHeaders())));
        // 2. 构建请求体
        requestContext.setRequestBody(AssetCollectApiRequest.buildRequestMap(collectContext));

        return requestContext;
    }

    /**
     * 处理请求前置脚本（使用HttpScriptContext）
     */
    protected void processRequestScript(CollectExecutorProp executor, HttpRequestContext requestContext) {
        if (StrUtil.isBlank(executor.getRequestScript())) {
            return;
        }
        // TODO pzz refactor clone requestContext防止被修改？
        // TODO pzz fix trace 打印执行前后数据？或者使用方法拦截，用户自定义log点？
        // 执行脚本
        HttpRequestContext resultContext = scriptExecutor.execute(executor.getRequestScript(),
                requestContext,
                HttpRequestContext.class);

        if (resultContext == null || requestContext.getRequestBody() == null) {
            log.warn("http请求脚本执行结果为null,忽略脚本处理！");
            return;
        }

        requestContext.setUrl(resultContext.getUrl());
        requestContext.setResponseHeader(resultContext.getResponseHeader());
        requestContext.setRequestBody(resultContext.getRequestBody());
    }

    protected void execute(CollectExecutorProp executor, HttpRequestContext requestContext) {
        HttpHeaders headers = buildHeaders(requestContext.getRequestHeader());
        HttpEntity<Object> entity = new HttpEntity<>(requestContext.getRequestBody(), headers);

        if (log.isDebugEnabled()) {
            log.debug("HTTP请求参数：url:{},header:{},body:{}", executor.getUrl(),
                    BaseConverter.INSTANCE.serialize(headers), BaseConverter.INSTANCE.serialize(requestContext.getRequestBody()));
        }

        // 5. 发送HTTP请求（使用动态创建的RetryTemplate）
        RetryTemplate dynamicRetryTemplate = createRetryTemplate(executor);

        ResponseEntity<String> resp = dynamicRetryTemplate.execute(request(executor.getUrl(), entity));

        requestContext.setResponseSuccess(resp.getStatusCode().is2xxSuccessful());
        requestContext.setResponseStatusCode(resp.getStatusCode().value());
        requestContext.setResponseBodyStr(resp.getBody());
        requestContext.setResponseBody(resp.getBody());
        requestContext.setResponseHeader(resp.getHeaders().asSingleValueMap());

        String responseBodyStr = (String) requestContext.getResponseBody();
        if (log.isDebugEnabled()) {
            log.debug("HTTP请求响应：url:{},status:{},body:{}", executor.getUrl(), requestContext.getResponseStatusCode(), responseBodyStr);
        }
    }


    protected RetryCallback<ResponseEntity<String>, RuntimeException> request(String url, HttpEntity<?> entity) {
        return retryContext -> {
            log.debug("发送HTTP请求: url={}, attempt={}", url, retryContext.getRetryCount() + 1);
            return restTemplate.postForEntity(url, entity, String.class);
        };
    }

    /**
     * 处理响应后置脚本并反序列化（使用HttpScriptContext）
     * 如果存在脚本，则将responseBody反序列化为Map传入脚本执行，脚本执行后再反序列化为目标类型
     * 否则直接使用scriptExecutor.convertResult进行反序列化
     */
    protected void processResponseScript(CollectExecutorProp executor, HttpRequestContext requestContext) {
        String responseBodyStr = requestContext.getResponseBodyStr();
        if (!requestContext.isResponseSuccess() || StrUtil.isBlank(responseBodyStr)) {
            log.warn("HTTP接口返回非成功状态: status={},body:{}", requestContext.getResponseStatusCode(), responseBodyStr);
            return;
        }

        if (StrUtil.isBlank(executor.getResponseScript())) {
            // 无脚本或脚本执行失败，直接反序列化
            AssetCollectApiResponse respBody = scriptExecutor.convertResult(responseBodyStr, AssetCollectApiResponse.class);
            requestContext.setResponseBody(respBody);

            return;
        }

        // 将响应体反序列化为Map
        //noinspection unchecked
        Map<String, Object> responseBodyMap = scriptExecutor.convertResult(responseBodyStr, Map.class);
        requestContext.setResponseBody(responseBodyMap);

        // 执行脚本
        HttpRequestContext result = scriptExecutor.execute(
                executor.getResponseScript(),
                requestContext,
                HttpRequestContext.class);

        if (result == null || result.getResponseBody() == null) {
            log.warn("http响应脚本执行结果为null,忽略脚本处理！");
            return;
        }

        requestContext.setResponseBody(scriptExecutor.convertResult(
                result.getResponseBody(), AssetCollectApiResponse.class));
    }

    protected PageResult<AssetCollectApiResponse.AssetDataItem> processResult(CollectContext collectContext,
                                                                              HttpRequestContext requestContext,
                                                                              CollectExecutorProp executor) {
        AssetCollectApiResponse responseBody = (AssetCollectApiResponse) requestContext.getResponseBody();
        if (responseBody != null) {
            // 8. 转换为PageResult（返回原始数据项）
            return convertToPageResult(responseBody, collectContext.getPageNo(), collectContext.getPageSize());
        } else {
            log.error("HTTP请求失败: status={}, url={},body:{}",
                    requestContext.getResponseStatusCode(), executor.getUrl(), requestContext.getResponseBodyStr());
            return new PageResult<>(new ArrayList<>(), collectContext.getPageNo(), collectContext.getPageSize());
        }
    }

    /**
     * 根据executor配置创建RetryTemplate
     * 支持配置重试次数、间隔和退避策略
     */
    protected RetryTemplate createRetryTemplate(CollectExecutorProp executor) {
        RetryTemplate template = new RetryTemplate();

        // 获取重试配置
        CollectExecutorProp.Retry retry = executor.getRetry();

        if (retry == null || !retry.isEnabled()) {
            // 未配置重试或禁用重试，使用默认配置（不重试）
            SimpleRetryPolicy retryPolicy = new SimpleRetryPolicy();
            retryPolicy.setMaxAttempts(1);
            template.setRetryPolicy(retryPolicy);
            return template;
        }

        // 配置重试策略
        SimpleRetryPolicy retryPolicy = new SimpleRetryPolicy();
        retryPolicy.setMaxAttempts(retry.getMaxAttempts());
        template.setRetryPolicy(retryPolicy);

        // 配置退避策略
        if (retry.getMultiplier() > 1.0) {
            // 使用指数退避策略
            ExponentialBackOffPolicy backOffPolicy = new ExponentialBackOffPolicy();
            backOffPolicy.setInitialInterval(retry.getInterval());
            backOffPolicy.setMaxInterval(retry.getMaxInterval());
            backOffPolicy.setMultiplier(retry.getMultiplier());
            template.setBackOffPolicy(backOffPolicy);

            log.debug("配置指数退避策略: initialInterval={}ms, maxInterval={}ms, multiplier={}",
                    backOffPolicy.getInitialInterval(), backOffPolicy.getMaxInterval(), backOffPolicy.getMultiplier());
        } else {
            // 使用固定间隔退避策略
            FixedBackOffPolicy backOffPolicy = new FixedBackOffPolicy();
            backOffPolicy.setBackOffPeriod(retry.getInterval());
            template.setBackOffPolicy(backOffPolicy);

            log.debug("配置固定退避策略: interval={}ms", backOffPolicy.getBackOffPeriod());
        }

        return template;
    }

    /**
     * 构建HTTP Headers
     */
    protected HttpHeaders buildHeaders(Map<String, Object> headersConfig) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        if (headersConfig != null && !headersConfig.isEmpty()) {
            for (Map.Entry<String, Object> entry : headersConfig.entrySet()) {
                headers.set(entry.getKey(), String.valueOf(entry.getValue()));
            }
        }

        return headers;
    }


    /**
     * 转换API响应为PageResult
     */
    protected PageResult<AssetCollectApiResponse.AssetDataItem> convertToPageResult(AssetCollectApiResponse apiResponse,
                                                                                    Integer pageNo, int pageSize) {
        if (apiResponse == null || !apiResponse.success()) {
            // TODO pzz 非成功抛出异常打断循环
            log.warn("HTTP接口返回非成功状态: code={}", apiResponse != null ? apiResponse.getCode() : null);
            return new PageResult<>(new ArrayList<>(), pageNo, pageSize);
        }

        List<AssetCollectApiResponse.AssetDataItem> dataItems = apiResponse.getData();
        if (CollectionUtils.isEmpty(dataItems)) {
            return new PageResult<>(new ArrayList<>(), pageNo, pageSize);
        }

        // 返回原始数据项，由上层负责转换
        boolean hasMore = dataItems.size() >= pageSize;

        PageResult<AssetCollectApiResponse.AssetDataItem> pageResult = new PageResult<>();
        pageResult.setData(dataItems);
        pageResult.setHasMore(hasMore);
        pageResult.setCurrentPage(pageNo);
        pageResult.setCurrentPageSize(dataItems.size());

        return pageResult;
    }
}
