package com.digiwin.athena.common.dict;

import cn.hutool.core.annotation.AnnotationUtil;
import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.util.ReflectUtil;
import com.digiwin.athena.constants.enums.BaseLangEnum;
import com.digiwin.athena.dto.action.MultiLanguageDTO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.beans.PropertyDescriptor;
import java.lang.annotation.*;
import java.lang.reflect.Field;
import java.util.Map;

@Target({ElementType.FIELD, ElementType.METHOD})
@Retention(RetentionPolicy.RUNTIME)
@Documented
@Component
public @interface Dict {

    Class<?> enumClass() default Void.class;

    @Slf4j
    class Translator {
        /**
         * 扫描对象中标记了@Dict注解的字段，根据字段值从枚举类中获取多语言文本
         * 并添加到对象的lang字段中
         *
         * @param obj 要处理的对象
         */
        public static void translate(Object obj) {
            if (obj == null) {
                return;
            }

            // 使用BeanProperty访问对象属性
            Class<?> clazz = obj.getClass();
            // TODO pzz performance 缓存注解查找
            Map<String, PropertyDescriptor> beanProperties = BeanUtil.getPropertyDescriptorMap(clazz, false);
            if (beanProperties == null || beanProperties.isEmpty()) {
                return;
            }

            // 扫描所有字段
            Field[] fields = ReflectUtil.getFields(clazz);
            for (Field field : fields) {
                try {
                    translate(obj, field, beanProperties);
                } catch (Exception e) {
                    if (log.isTraceEnabled()) {
                        log.trace("字典翻译异常！", e);
                    } else {
                        log.warn("字典翻译异常！msg:{}", e.getMessage());
                    }
                }
            }
        }

        protected static void translate(Object obj, Field field, Map<String, PropertyDescriptor> beanProperties) {
            // 检查是否标记了@Dict注解
            Dict dictAnnotation = AnnotationUtil.getAnnotation(field, Dict.class);
            if (dictAnnotation == null) {
                return;
            }

            // 获取字段值
            PropertyDescriptor prop = beanProperties.get(field.getName());
            if (prop == null) {
                return;
            }

            Object fieldVal = ReflectUtil.invoke(obj, prop.getReadMethod());
            // 获取枚举类
            Class<?> enumClass = dictAnnotation.enumClass();

            // 检查是否是BaseLangEnum的实现
            if (BaseLangEnum.class.isAssignableFrom(enumClass)) {
                Object langObj = ReflectUtil.invoke(obj, beanProperties.get("lang").getReadMethod());
                // noinspection unchecked
                translateLangEnum(langObj, (Class<? extends BaseLangEnum<?>>) enumClass, prop, fieldVal);
            } else {
                log.warn("暂不支持的字典注解转义方式！annotation:{}", dictAnnotation);
            }
        }

        protected static <K> void translateLangEnum(Object langVal, Class<? extends BaseLangEnum<?>> enumClass,
                                                    PropertyDescriptor prop, Object fieldVal) {
            // 检查lang字段是否存在且为Map类型
            if (!(langVal instanceof Map)) {
                return;
            }

            //noinspection unchecked
            Map<String, MultiLanguageDTO<?>> langMap = (Map<String, MultiLanguageDTO<?>>)  langVal;
            if (langMap.containsKey(prop.getName())) {
                return;
            }


            // 根据字段值从枚举类中获取匹配的枚举
            //noinspection unchecked
            BaseLangEnum<K> enumValue = BaseLangEnum.byCodeObj((Class<? extends BaseLangEnum<K>>) enumClass, fieldVal);
            if (enumValue == null) {
                return;
            }

            langMap.put(prop.getName(), enumValue.buildMultiLangDto());
        }
    }
}
