package com.digiwin.athena.dao.mongodao.asset;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.config.BranchConfig;
import com.digiwin.athena.dto.PageReqCondition;
import com.digiwin.athena.dto.asset.DesignerAssetListReqDto;
import com.digiwin.athena.enums.VersionStatusEnum;
import com.digiwin.athena.mongodb.domain.application.Asset;
import com.digiwin.athena.mongodb.domain.application.AssetDefinition;
import com.digiwin.athena.mongodb.repository.MongoPrimaryRepositoryDecorator;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Field;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Repository;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * 资产定义交互层
 */
@Slf4j
@Repository
public class AssetMongoDao {

    @Autowired
    private MongoPrimaryRepositoryDecorator mongoPrimaryRepositoryDecorator;

    public void save(Asset asset){
        mongoPrimaryRepositoryDecorator.save(asset);
    }

    public Asset findOne(String assetId){
        Query query = new Query(Criteria.where("assetId").is(assetId));
        return mongoPrimaryRepositoryDecorator.findOne(query, Asset.class);
    }

    public List<Asset> findByApplication(String application){
        Query query = new Query(Criteria.where("manageContext.application").is(application));
        return mongoPrimaryRepositoryDecorator.find(query, Asset.class);
    }

    public List<Asset> findByAssetIds(List<String> assetIds){
        Query query = new Query(Criteria.where("assetId").in(assetIds));
        return mongoPrimaryRepositoryDecorator.find(query, Asset.class);
    }

    public void delete(String assetId){
        Query query = new Query(Criteria.where("assetId").is(assetId));
        mongoPrimaryRepositoryDecorator.delete(query, Asset.class);
    }

    public void insert(Asset asset) {
        mongoPrimaryRepositoryDecorator.insert(asset);
    }

    public void delete(Asset asset) {
        mongoPrimaryRepositoryDecorator.delete(asset);
    }

    public List<Asset> selectByCodeAndApplication(String code, String application) {
        Criteria criteria = Criteria.where("manageContext.application").is(application)
                .and("code").is(code);
        return mongoPrimaryRepositoryDecorator.find(new Query(criteria),Asset.class);
    }

    public List<Asset> selectByAssetId(String assetId) {
        Criteria criteria = Criteria.where("assetId").is(assetId);
        return mongoPrimaryRepositoryDecorator.find(new Query(criteria),Asset.class);
    }

    public Asset selectByAssetIdAndSourceBranchAndAdpVersion(String assetId, String sourceBranch, String adpVersion) {
        Criteria criteria = Criteria.where("assetId").is(assetId)
                .and("manageContext.branch").is(sourceBranch)
                .and("manageContext.adpVersion").is(adpVersion);
        return mongoPrimaryRepositoryDecorator.findOne(new Query(criteria),Asset.class);
    }

    /**
     * 将生效版本置为失效
     * @param editBy
     * @param assetId
     */
    public void updateDraftAdpStatusByAssetId(String editBy, String assetId){
        Update update = new Update();
        update.set("manageContext.adpStatus","draft");
        update.set("editDate",new Date());
        update.set("editBy",editBy);

        Criteria criteria = Criteria.where("assetId").is(assetId).and("manageContext.adpStatus").is("effect");
        mongoPrimaryRepositoryDecorator.updateMulti(new Query(criteria),update,Asset.class);
    }

    public Asset selectEffectByAssetId(String assetId) {
        Criteria criteria = Criteria.where("assetId").is(assetId).and("manageContext.adpStatus").is("effect");
        return mongoPrimaryRepositoryDecorator.findOne(new Query(criteria),Asset.class);
    }

    public List<Asset> selectEffectByAssetIds(Collection<String> assetIds) {
        Criteria criteria = Criteria.where("assetId").in(assetIds).and("manageContext.adpStatus").is("effect");
        return mongoPrimaryRepositoryDecorator.find(new Query(criteria),Asset.class);
    }

    public List<Asset> selectBasicInfoByObjectId(List<String> objIds) {
        Criteria criteria = Criteria.where("_id").in(objIds);
        Query query = new Query(criteria);
        query.fields().exclude("businessData","content");
        return mongoPrimaryRepositoryDecorator.find(query,Asset.class);
    }

    public Asset selectByObjectId(String objId) {
        Criteria criteria = Criteria.where("_id").is(objId);
        Query query = new Query(criteria);
        return mongoPrimaryRepositoryDecorator.findOne(query,Asset.class);
    }

    public List<Asset> selectByCodeAndApplicationAndSourceBranchAndAdpVersion(Asset asset) {
        Criteria criteria = Criteria.where("code").is(asset.getCode())
                .and("manageContext.application").is(asset.getManageContext().getApplication())
                .and("manageContext.branch").is(asset.getManageContext().getBranch());

        if (StringUtils.isNotEmpty(asset.getManageContext().getAdpVersion())){
            criteria.and("manageContext.adpVersion").is(asset.getManageContext().getAdpVersion());
        }
        return mongoPrimaryRepositoryDecorator.find(new Query(criteria),Asset.class);
    }

    public void deleteByCodeAndApplicationAndSourceBranchAndAdpVersion(Asset asset) {
        Criteria criteria = Criteria.where("code").is(asset.getCode())
                .and("manageContext.application").is(asset.getManageContext().getApplication())
                .and("manageContext.branch").is(asset.getManageContext().getBranch());

        if (StringUtils.isNotEmpty(asset.getManageContext().getAdpVersion())){
            criteria.and("manageContext.adpVersion").is(asset.getManageContext().getAdpVersion());
        }
        mongoPrimaryRepositoryDecorator.delete(new Query(criteria),Asset.class);
    }

    /**
     * 查询指定分支下的资产最新的版本
     * @param assetId
     * @param branch
     * @param application
     * @return
     */
    public Asset selectLatestVersionByAssetIdAndSourceBranchAndApplication(String assetId, String branch, String application) {
        Criteria criteria = Criteria.where("assetId").is(assetId)
                .and("manageContext.application").is(application)
                .and("manageContext.branch").is(branch);
        Query query = new Query(criteria);
        query.with(Sort.by(Sort.Direction.DESC,"manageContext.adpVersion"));
        query.limit(1);

        return mongoPrimaryRepositoryDecorator.findOne(query,Asset.class);
    }

    public Asset selectAssetIdByCodeAndApplication(String code, String application) {
        Criteria criteria = Criteria.where("manageContext.application").is(application)
                .and("code").is(code);
        Query query = new Query(criteria);
        query.fields().include("assetId");
        query.limit(1);
        return mongoPrimaryRepositoryDecorator.findOne(query,Asset.class);
    }

    public List<Asset> selectByApplicationAndAdpVersionAndBranch(String application, String adpVersion, String branch) {
        Criteria criteria = Criteria.where("manageContext.application").is(application);
        if (StringUtils.isNotEmpty(adpVersion)){
            criteria.and("manageContext.adpVersion").is(adpVersion);
        }
        if (StringUtils.isNotEmpty(branch)){
            criteria.and("manageContext.branch").is(branch);
        }
        return mongoPrimaryRepositoryDecorator.find(new Query(criteria),Asset.class);
    }

    public List<String> selectAllId() {
        Query query = new Query();
        query.fields().include("_id");
        return mongoPrimaryRepositoryDecorator.find(query,Asset.class).stream().map(Asset::getObjectId).collect(Collectors.toList());
    }

    /**
     * 查询指定资产id下，有没有生效的资产版本
     * @param assetIds
     * @return
     */
    public Long selectEffectCount(List<String> assetIds) {
        Criteria criteria = Criteria.where("assetId").in(assetIds).and("manageContext.adpStatus").is(VersionStatusEnum.RUN_STATE.getCode());
        return mongoPrimaryRepositoryDecorator.count(new Query(criteria),Asset.class);
    }

    public Set<String> selectExistAssetId(List<String> assetIds) {
        Criteria criteria = Criteria.where("assetId").in(assetIds);
        Query query = new Query(criteria);
        query.fields().include("assetId");
        return mongoPrimaryRepositoryDecorator.find(query,Asset.class).stream().map(Asset::getAssetId).collect(Collectors.toSet());
    }

    public Asset selectLastOneNew(String assetId) {
        Criteria criteria = Criteria.where("assetId").is(assetId).and("manageContext.branch").is(BranchConfig.DEV_BRANCH_NAME);
        return mongoPrimaryRepositoryDecorator.findOne(new Query(criteria),Asset.class);
    }

    public List<Asset> selectByApplicationAndBranch(String application, String branch) {
        Criteria criteria = Criteria.where("manageContext.application").is(application).and("manageContext.branch").is(branch);
        return mongoPrimaryRepositoryDecorator.find(new Query(criteria),Asset.class);
    }

    public void insertAll(List<Asset> assetList) {
        mongoPrimaryRepositoryDecorator.insertAll(assetList);
    }

    public void deleteByApplicationAndBranch(String application, String branch) {
        Criteria criteria = Criteria.where("manageContext.application").is(application).and("manageContext.branch").is(branch);
        mongoPrimaryRepositoryDecorator.remove(new Query(criteria),Asset.class);
    }

    public void updateDraftByApplicationAndType(String application, AssetDefinition.AssetTypeEnum assetTypeEnum) {
        Criteria criteria = Criteria.where("manageContext.application").is(application).and("type").is(assetTypeEnum.getCode());
        Update update = new Update();
        update.set("manageContext.adpStatus","draft");
        mongoPrimaryRepositoryDecorator.updateMulti(new Query(criteria),update,Asset.class);
    }

    public List<Asset> selectEffectVersionByAssetIds(List<String> assetIds) {
        Criteria criteria = Criteria.where("assetId").in(assetIds).and("manageContext.adpStatus").is("effect");
        Query query = new Query(criteria);
        query.fields().include("assetId","manageContext.adpVersion","showFields.adpRemarkNumber");
        return mongoPrimaryRepositoryDecorator.find(query,Asset.class);
    }

    public List<Asset> selectByIds(Collection<String> objIds) {
        Criteria criteria = Criteria.where("_id").in(objIds);
        Query query = new Query(criteria);
        return mongoPrimaryRepositoryDecorator.find(query,Asset.class);
    }

    public List<Asset> selectBasicInfoByAssetId(String assetId) {
        Criteria criteria = Criteria.where("assetId").is(assetId);
        Query query = new Query(criteria);
        query.fields().exclude("businessData","content");
        return mongoPrimaryRepositoryDecorator.find(query,Asset.class);
    }

    /**
     * 查询最新的资产版本
     * @param assetIds
     * @return
     */
    public List<Asset> selectVersionInfo(List<String> assetIds) {
        Criteria criteria = Criteria.where("assetId").in(assetIds);
        Query query = new Query(criteria);
        Field fields = query.fields();
        fields.include("assetId");
        fields.include("manageContext.adpVersion");
        fields.include("manageContext.branch");
        fields.include("showFields.adpRemarkNumber");
        query.with(Sort.by(Sort.Direction.DESC,"_id"));
        return mongoPrimaryRepositoryDecorator.find(query,Asset.class);
    }

    public List<Asset> findByAssetIdsAndBranch(List<String> assetIds,String branch){
        Query query = new Query(Criteria.where("assetId").in(assetIds).and("manageContext.branch").is(branch).
                and("manageContext.adpStatus").is(VersionStatusEnum.RUN_STATE.getCode()));
        return mongoPrimaryRepositoryDecorator.find(query, Asset.class);
    }

    public Asset selectByAssetIdAndMCDataId(String assetId, Asset.ManageContext manageContext) {
        Criteria criteria = Criteria.where("assetId").is(assetId);

        if (manageContext != null && manageContext.containsKey(Asset.ManageContext.Fields.dataId)) {
            criteria.and("manageContext.dataId").is(manageContext.get(Asset.ManageContext.Fields.dataId));
        }

        return mongoPrimaryRepositoryDecorator.findOne(new Query(criteria),Asset.class);
    }

    public Long countDesignerAssetPagination(PageReqCondition<DesignerAssetListReqDto> pageReqCondition) {
        DesignerAssetListReqDto condition = pageReqCondition.getCondition();

        PageReqCondition<String> newPageReqCondition = new PageReqCondition<>();
        newPageReqCondition.setCondition(condition.getSearchContent());
        Criteria criteria = buildAdvancedCondition(newPageReqCondition);

        Asset.ManageContext manageContext = condition.getManageContext();
        List<String> types = condition.getTypes();
        Criteria manageContextCriteria = new Criteria();

        if (StringUtils.isEmpty(manageContext.getApplication())){
            manageContextCriteria.and("manageContext.application").isNull();
        }else{
            manageContextCriteria.and("manageContext.application").is(manageContext.getApplication());
        }

        if (StringUtils.isEmpty(manageContext.getBranch())){
            manageContextCriteria.and("manageContext.branch").isNull();
        }else{
            manageContextCriteria.and("manageContext.branch").is(manageContext.getBranch());
        }

        if (StringUtils.isEmpty(manageContext.getAdpVersion())){
            manageContextCriteria.and("manageContext.adpVersion").isNull();
        }else{
            manageContextCriteria.and("manageContext.adpVersion").is(manageContext.getAdpVersion());
        }

        if (criteria == null){
            criteria= new Criteria();
            criteria.andOperator(manageContextCriteria);
            if (!CollectionUtils.isEmpty(types)){
                criteria = criteria.and("type").in(types);
            }
        }else{
            List<Criteria> criteriaList = new ArrayList<>();
            criteriaList.add(criteria);
            criteriaList.add(manageContextCriteria);
            if (!CollectionUtils.isEmpty(types)){
                criteriaList.add(Criteria.where("type").in(types));
            }

            criteria = new Criteria().andOperator(criteriaList);
        }
        Query query = new Query(criteria);

        return mongoPrimaryRepositoryDecorator.count(query, Asset.class);
    }

    public List<Asset> selectDesignerAssetPagination(PageReqCondition<DesignerAssetListReqDto> pageReqCondition) {
        DesignerAssetListReqDto condition = pageReqCondition.getCondition();

        PageReqCondition<String> newPageReqCondition = new PageReqCondition<>();
        newPageReqCondition.setCondition(condition.getSearchContent());
        Criteria criteria = buildAdvancedCondition(newPageReqCondition);

        Asset.ManageContext manageContext = condition.getManageContext();
        List<String> types = condition.getTypes();
        Criteria manageContextCriteria = new Criteria();

        if (StringUtils.isEmpty(manageContext.getApplication())){
            manageContextCriteria.and("manageContext.application").isNull();
        }else{
            manageContextCriteria.and("manageContext.application").is(manageContext.getApplication());
        }

        if (StringUtils.isEmpty(manageContext.getBranch())){
            manageContextCriteria.and("manageContext.branch").isNull();
        }else{
            manageContextCriteria.and("manageContext.branch").is(manageContext.getBranch());
        }

        if (StringUtils.isEmpty(manageContext.getAdpVersion())){
            manageContextCriteria.and("manageContext.adpVersion").isNull();
        }else{
            manageContextCriteria.and("manageContext.adpVersion").is(manageContext.getAdpVersion());
        }

        if (criteria == null){
            criteria= new Criteria();
            criteria.andOperator(manageContextCriteria);
            if (!CollectionUtils.isEmpty(types)){
                criteria = criteria.and("type").in(types);
            }
        }else{
            List<Criteria> criteriaList = new ArrayList<>();
            criteriaList.add(criteria);
            criteriaList.add(manageContextCriteria);
            if (!CollectionUtils.isEmpty(types)){
                criteriaList.add(Criteria.where("type").in(types));
            }

            criteria = new Criteria().andOperator(criteriaList);
        }
        Query query = new Query(criteria);
        query.limit(pageReqCondition.getPageSize());
        query.skip((pageReqCondition.getPageNum() - 1) * pageReqCondition.getPageSize());
        query.with(Sort.by(Sort.Direction.DESC,"_id"));

        return mongoPrimaryRepositoryDecorator.find(query, Asset.class);
    }

    private Criteria buildAdvancedCondition(PageReqCondition<String> pageReqCondition) {
        Criteria criteria = null;
        if (pageReqCondition.getAdvancedQueryParams() != null) {
            Criteria adVancedCriteria = pageReqCondition.getAdvancedQueryParams().buildCriteria();
            if (adVancedCriteria != null) {
                criteria = adVancedCriteria;
            }
        }

        if (criteria == null && StringUtils.isNotEmpty(pageReqCondition.getCondition())) {
            Pattern pattern = Pattern.compile(".*" + Pattern.quote(pageReqCondition.getCondition()) + ".*", Pattern.CASE_INSENSITIVE);

            criteria = new Criteria();
            criteria.orOperator(Criteria.where("assetId").regex(pattern),
                    Criteria.where("code").regex(pattern),
                    Criteria.where("assetName").regex(pattern),
                    Criteria.where("assetDesc").regex(pattern));
        }
        return criteria;
    }

    public Asset selectByCodeAndTypeAndMCDataId(String type, String code, JSONObject manageContext) {
        Criteria criteria = Criteria.where("type").is(type).and("code").is(code);

        if (!CollectionUtils.isEmpty(manageContext) && manageContext.containsKey("dataId")){
            criteria.and("manageContext.dataId").is(manageContext.getString("dataId"));
        }

        return mongoPrimaryRepositoryDecorator.findOne(new Query(criteria),Asset.class);
    }

    public List<String> selectIdByType(List<String> types){
        Query query = new Query(Criteria.where("type").in(types));
        query.fields().include("_id");
        return mongoPrimaryRepositoryDecorator.find(query,Asset.class).stream().map(Asset::getObjectId).collect(Collectors.toList());
    }

    public List<Asset> selectByManageContext(Asset.ManageContext manageContext) {
        Criteria criteria = new Criteria();
        for (Map.Entry<String, Object> entry : manageContext.entrySet()) {
            criteria.and("manageContext."+entry.getKey()).is(entry.getValue());
        }
        Query query = new Query(criteria);
        query.with(Sort.by(Sort.Direction.DESC,"_id"));
        return mongoPrimaryRepositoryDecorator.find(query,Asset.class);
    }

    public List<Asset> selectByAssetIdAndManageContext(String assetId, Asset.ManageContext manageContext) {
        Criteria criteria = Criteria.where("assetId").is(assetId);

        if (manageContext!=null){
            for (Map.Entry<String, Object> entry : manageContext.entrySet()) {
                criteria.and("manageContext."+entry.getKey()).is(entry.getValue());
            }
        }

        return mongoPrimaryRepositoryDecorator.find(new Query(criteria),Asset.class);
    }
}
