package com.digiwin.athena.dao.mongodao.datacollect;

import com.digiwin.athena.mongodb.domain.datacollect.JobExecutionRecord;
import com.digiwin.athena.mongodb.repository.MongoPrimaryRepositoryDecorator;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Repository;

import java.util.Date;
import java.util.List;

/**
 * 任务执行记录MongoDB DAO
 */
@Repository
public class JobExecutionRecordMongoDao {

    @Autowired
    private MongoPrimaryRepositoryDecorator mongoPrimaryRepositoryDecorator;

    /**
     * 保存任务执行记录
     */
    public void save(JobExecutionRecord record) {
        mongoPrimaryRepositoryDecorator.save(record);
    }

    /**
     * 根据ID查询
     */
    public JobExecutionRecord findById(String id) {
        return mongoPrimaryRepositoryDecorator.findById(id, JobExecutionRecord.class);
    }

    /**
     * 根据任务名称查询最新记录
     */
    public JobExecutionRecord findLatestByJobName(String jobName) {
        Query query = new Query();
        query.addCriteria(Criteria.where("jobName").is(jobName));
        query.with(Sort.by(Sort.Direction.DESC, "fireTime"));
        query.limit(1);

        List<JobExecutionRecord> records = mongoPrimaryRepositoryDecorator.find(
                query, JobExecutionRecord.class);

        return records.isEmpty() ? null : records.get(0);
    }

    /**
     * 根据业务唯一标识查询最新记录
     */
    public JobExecutionRecord findLatestByBizKey(String bizKey) {
        Query query = new Query();
        query.addCriteria(Criteria.where("bizKey").is(bizKey));
        query.with(Sort.by(Sort.Direction.DESC, "fireTime"));
        query.limit(1);

        List<JobExecutionRecord> records = mongoPrimaryRepositoryDecorator.find(
                query, JobExecutionRecord.class);

        return records.isEmpty() ? null : records.get(0);
    }

    /**
     * 根据任务名称查询记录列表
     */
    public List<JobExecutionRecord> findByJobName(String jobName, int limit) {
        Query query = new Query();
        query.addCriteria(Criteria.where("jobName").is(jobName));
        query.with(Sort.by(Sort.Direction.DESC, "fireTime"));
        query.limit(limit);

        return mongoPrimaryRepositoryDecorator.find(query, JobExecutionRecord.class);
    }

    /**
     * 根据业务唯一标识查询记录列表
     */
    public List<JobExecutionRecord> findByBizKey(String bizKey, int limit) {
        Query query = new Query();
        query.addCriteria(Criteria.where("bizKey").is(bizKey));
        query.with(Sort.by(Sort.Direction.DESC, "fireTime"));
        query.limit(limit);

        return mongoPrimaryRepositoryDecorator.find(query, JobExecutionRecord.class);
    }

    /**
     * 根据时间范围查询
     */
    public List<JobExecutionRecord> findByTimeRange(Date startTime, Date endTime) {
        Query query = new Query();
        Criteria criteria = Criteria.where("fireTime").gte(startTime).lte(endTime);
        query.addCriteria(criteria);
        query.with(Sort.by(Sort.Direction.DESC, "fireTime"));

        return mongoPrimaryRepositoryDecorator.find(query, JobExecutionRecord.class);
    }

    /**
     * 根据状态查询
     */
    public List<JobExecutionRecord> findByStatus(String status, int limit) {
        Query query = new Query();
        query.addCriteria(Criteria.where("status").is(status));
        query.with(Sort.by(Sort.Direction.DESC, "fireTime"));
        query.limit(limit);

        return mongoPrimaryRepositoryDecorator.find(query, JobExecutionRecord.class);
    }

    /**
     * 删除过期记录（保留最近N天）
     */
    public long deleteOldRecords(int daysToKeep) {
        Date cutoffDate = new Date(System.currentTimeMillis() - daysToKeep * 24L * 60 * 60 * 1000);
        Query query = new Query();
        query.addCriteria(Criteria.where("createdAt").lt(cutoffDate));

        return mongoPrimaryRepositoryDecorator.remove(query, JobExecutionRecord.class);
    }
}
