package com.digiwin.athena.datacollect.model;

import com.digiwin.athena.config.AssemblyProp;
import lombok.Data;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;

/**
 * 数据采集执行器配置
 */
@Data
public class CollectExecutorProp {

    /**
     * 执行器类型（如：http、database等）
     */
    private String type;

    /**
     * 请求URL（支持表达式）
     */
    private String url;

    /**
     * 请求头配置（支持表达式）
     */
    private Map<String, Object> headers;

    /**
     * 循环参数列表
     * 当配置该值时，任务执行时需要遍历该数组调用任务实现
     * 每个元素支持表达式计算
     */
    private List<Map<String, Object>> loopParams;

    /**
     * 开始时间
     */
    private LocalDateTime startTime;

    /**
     * 查询时间步长
     * 用于步进式查询，避免一次查询时间范围过大
     */
    private Duration stepDuration;

    /**
     * 请求前置脚本
     * 用于在发送HTTP请求前对url、requestBody、requestHeader进行处理
     * 脚本输入：url, requestBody, requestHeader
     * 脚本输出：处理后的requestBody和requestHeader
     */
    private String requestScript;

    /**
     * 响应后置脚本
     * 用于在接收HTTP响应后对响应数据进行处理
     * 脚本输入：responseBody
     * 脚本输出：处理后的responseBody
     */
    private String responseScript;

    /**
     * 分页大小
     * 默认为100
     */
    private Integer pageSize;

    /**
     * 请求超时时间（毫秒）
     * 默认为30000（30秒）
     */
    private Integer timeout;

    /**
     * 重试配置
     */
    private Retry retry;

    public Integer getPageSize(AssemblyProp.DataCollector defaultProp) {
        if (pageSize == null) {
            return defaultProp.getPageSize();
        }

        return pageSize;
    }

    public Duration getStepDuration(AssemblyProp.DataCollector defaultProp) {
        if (stepDuration == null) {
            return defaultProp.getStepSpan();
        }

        return stepDuration;
    }

    public LocalDateTime getStartTime(AssemblyProp.DataCollector collector) {
        if (startTime != null) {
            return startTime;
        }

        return collector.getStartTime();
    }

    /**
     * 重试配置类
     */
    @Data
    public static class Retry {
        /**
         * 是否启用重试
         */
        private boolean enabled = false;

        /**
         * 最大重试次数
         */
        private int maxAttempts = 3;

        /**
         * 重试间隔（毫秒）
         */
        private long interval = 2000L;

        /**
         * 最大重试间隔（毫秒）
         * 用于指数退避策略
         */
        private long maxInterval = 10000L;

        /**
         * 退避倍数
         * 用于指数退避策略
         */
        private double multiplier = 2.0;
    }
}
