package com.digiwin.athena.dto.assembly.designer;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.date.DatePattern;
import com.digiwin.athena.common.dict.Dict;
import com.digiwin.athena.constants.enums.assembly.IntegrationType;
import com.digiwin.athena.constants.enums.assembly.OnlineState;
import com.digiwin.athena.dto.action.MultiLanguageDTO;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanBaseDTO;
import com.digiwin.athena.dto.assetType.AssetTypeBaseDTO;
import com.fasterxml.jackson.annotation.JsonFormat;
import lombok.Data;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 设计器列表DTO
 */
@Data
public class DesignerListDTO {

    /**
     * 设计器ID
     */
    private String id;

    /**
     * 设计器编码
     */
    private String type;

    /**
     * 归属方
     */
    private String owner;

    /**
     * 设计器名称
     */
    private String name;

    /**
     * 版本号
     */
    private String version;

    /**
     * 状态
     * @see OnlineState#getCode()
     */
    @Dict(enumClass = OnlineState.class)
    private String status;

    /**
     * 描述
     */
    private String description;

    /**
     * 可读资产类型
     */
    private List<AssetTypeBaseDTO> readAssetType;

    /**
     * 可产生资产类型
     */
    private List<AssetTypeBaseDTO> writeAssetType;

    /**
     * 关联的解决方案
     */
    private List<SolutionPlanBaseDTO> relatedSolutionPlan;

    /**
     * 集成方式
     * @see IntegrationType#getCode()
     */
    @Dict(enumClass = IntegrationType.class)
    private String integrationType;

    /**
     * 多语言配置
     */
    private Map<String, MultiLanguageDTO<String>> lang;


    /**
     * 发布时间
     */
    @JsonFormat(pattern = DatePattern.NORM_DATETIME_PATTERN)
    private LocalDateTime publishTime;

    /**
     * 发布人
     */
    private String publishBy;


    /**
     * 更新时间
     */
    @JsonFormat(pattern = DatePattern.NORM_DATETIME_PATTERN)
    private LocalDateTime editTime;

    /**
     * 更新人
     */
    private String editBy;

    public static List<String> obtainIdList(List<DesignerListDTO> list) {
        return list.stream()
                .map(DesignerListDTO::getId)
                .collect(Collectors.toList());
    }

    public static List<String> obtainAssetTypeIdList(List<DesignerListDTO> designer) {
        return designer.stream()
                .flatMap(v -> Stream.concat(
                        CollUtil.emptyIfNull(v.getReadAssetType()).stream(),
                        CollUtil.emptyIfNull(v.getWriteAssetType()).stream()
                ))
                .map(AssetTypeBaseDTO::getObjId)
                .collect(Collectors.toList());
    }
}
