package com.digiwin.athena.dto.assembly.solution;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.date.DatePattern;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.http.Method;
import com.digiwin.athena.constants.enums.assembly.AssemblyType;
import com.digiwin.athena.constants.enums.assembly.OnlineState;
import com.digiwin.athena.dto.action.MultiLanguageDTO;
import com.fasterxml.jackson.annotation.JsonFormat;
import jakarta.validation.constraints.NotBlank;
import lombok.Data;

import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 可组装解决方案配置
 */
@Data
public class SolutionPlanDetailDTO implements SolutionPlanAdapter {

    /**
     * 主键ID
     */
    private String id;

    /**
     * 解决方案编码，全局唯一
     */
    @NotBlank
    private String type;

    /**
     * 解决方案名称
     */
    @NotBlank
    private String name;

    /**
     * 版本号，service层暂时写死1.0
     */
    private String version;

    /**
     * 描述
     */
    private String description;

    /**
     * 组装方式：dynamic(可组装)/static(不可组装)
     * @see AssemblyType#getCode()
     */
    private String assemblyType;

    /**
     * 解决方案状态：Draft(草稿)/Online(已上架)/Offline(未上架)
     * @see OnlineState#getCode()
     */
    private String status;

    /**
     * 归属方
     */
    private String owner;

    private Integer category;

    /**
     * 应用类型，新增时自动分配
     */
    private Integer appType;

    /**
     * 关联设计器列表
     */
    private List<DesignerBaseInfoDTO> designerList;

    /**
     * 可产生资产的关联设计器列表（relatedAssetType.write不为空的设计器）
     */
    private List<DesignerBaseInfoDTO> writableDesignerList;

    /**
     * 布局配置
     */
    private Map<String, Object> canvas;

    /**
     * 布局方式
     */
    private String layoutMode;

    /**
     * SSO配置(不可组装页面配置)
     */
    private SsoConfigDTO ssoConfig;

    /**
     * 扩展配置字段 - 应用创建回调
     */
    private List<AppCallBackDTO> appCreateCallback;

    /**
     * 扩展配置字段 - 应用更新回调
     */
    private List<AppCallBackDTO> appUpdateCallback;

    /**
     * 扩展配置字段 - 编译回调
     */
    private List<AppCallBackDTO> compile;

    /**
     * 扩展配置字段 - 发布回调
     */
    private List<AppCallBackDTO> publish;

    /**
     * 前端变量
     */
    private Integer frontBizSetupPageIdx = 0;

    /**
     * true走标准发布，对应的应用类型会在运维中心体现
     */
    private Boolean isStdPublish;

    /**
     * 合并JSON字段，用于动态添加额外数据
     *
     * @see com.digiwin.athena.bo.assembly.SolutionPlan#getMergeJson
     */
    private Map<String, Object> mergeJson;

    /**
     * 多语言配置
     */
    private Map<String, MultiLanguageDTO<String>> lang;

    /**
     * 发布时间
     */
    @JsonFormat(pattern = DatePattern.NORM_DATETIME_PATTERN)
    private LocalDateTime publishTime;

    /**
     * 发布人
     */
    private String publishBy;

    /**
     * 创建时间
     */
    @JsonFormat(pattern = DatePattern.NORM_DATETIME_PATTERN)
    private LocalDateTime createTime;

    /**
     * 更新时间
     */
    @JsonFormat(pattern = DatePattern.NORM_DATETIME_PATTERN)
    private LocalDateTime editTime;

    /**
     * 创建人
     */
    private String createBy;

    /**
     * 更新人
     */
    private String editBy;

    /**
     * 设计器基础信息DTO
     */
    @Data
    public static class DesignerBaseInfoDTO {

        /**
         * ID
         */
        private String id;

        /**
         * 编码
         */
        private String type;

        /**
         * 名称
         */
        private String name;

        /**
         * 多语言配置
         */
        private Map<String, MultiLanguageDTO<String>> lang;
    }

    public List<String> obtainDesignerIds() {
        // 提取设计器ID列表
        return CollUtil.emptyIfNull(getDesignerList()).stream()
                .map(SolutionPlanDetailDTO.DesignerBaseInfoDTO::getId)
                .distinct().collect(Collectors.toList());
    }

    /**
     * 应用回调DTO
     */
    @Data
    public static class AppCallBackDTO {

        /**
         * api类型，例如：http，groovy
         */
        private String type;

        /**
         * url
         */
        private String url;

        /**
         * 当type是bean时，name是bean的name
         */
        private String name;

        /**
         * 请求类型：GET/POST
         */
        private Method method;

        /**
         * 头信息，json 字符串 k-v 格式
         */
        private String header;

        /**
         * groovy脚本Id
         */
        private String scriptId;

        /**
         * 请求脚本
         */
        private String requestScript;

        /**
         * 响应脚本
         */
        private String responseScript;

        /**
         * 请求方法名称
         */
        private String methodName;
    }

    /**
     * SSO配置DTO
     */
    @Data
    public static class SsoConfigDTO {

        /**
         * 表达式key
         */
        public static final String KEY_EXPRESSION = "value";
        public static final String KEY_TYPE = "type";
        public static final String EXPRESSION = "system";
        public static final String KEY_VAR_NAME = "key";


        /**
         * url
         */
        private String url;

        /**
         * 配置信息
         */
        private List<Map<String, Object>> params;

        public Map<String, Object> toParamMap(Function<String, String> wrapVarNameFunc) {
            return CollUtil.emptyIfNull(params).stream()
                    // 对用户自定义变量做转义？ 没必要，context中只有用户可访问变量
                    // .filter(v -> v != null && EXPRESSION.equals(v.get(KEY_TYPE)))
                    .reduce(
                            new HashMap<>(),
                            (v1, v2) -> {
                                String val = StrUtil.toStringOrNull(v2.get(KEY_EXPRESSION));
                                if (EXPRESSION.equals(v2.get(KEY_TYPE))) {
                                    val = wrapVarNameFunc.apply(val);
                                }

                                v1.put(MapUtil.getStr(v2, KEY_VAR_NAME), val);
                                return v1;
                            }
                    );
        }
    }
}
