package com.digiwin.athena.dto.assembly.solution;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.date.DatePattern;
import cn.hutool.core.date.LocalDateTimeUtil;
import com.digiwin.athena.common.dict.Dict;
import com.digiwin.athena.constants.enums.assembly.AssemblyType;
import com.digiwin.athena.constants.enums.assembly.OnlineState;
import com.digiwin.athena.dto.action.MultiLanguageDTO;
import com.fasterxml.jackson.annotation.JsonFormat;
import lombok.Data;

import java.time.LocalDateTime;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 解决方案列表DTO
 */
@Data
public class SolutionPlanListDTO implements SolutionPlanAdapter {

    /**
     * ID
     */
    private String id;

    /**
     * 名称
     */
    private String name;

    /**
     * 编码
     */
    private String type;

    private Integer category;
    private Integer appType;

    /**
     * 组装方式
     * @see AssemblyType#getCode()
     */
    @Dict(enumClass = AssemblyType.class)
    private String assemblyType;

    private SolutionPlanDetailDTO.SsoConfigDTO ssoConfig;              // SSO配置(不可组装页面配置)

    /**
     * 关联设计器
     */
    private List<SolutionPlanDetailDTO.DesignerBaseInfoDTO> designerList;

    /**
     * 归属方
     */
    private String owner;

    /**
     * 状态
     * @see OnlineState#getCode()
     */
    @Dict(enumClass = OnlineState.class)
    private String status;

    /**
     * 版本
     */
    private String version;

    /**
     * true走标准发布，对应的应用类型会在运维中心体现
     */
    private Boolean isStdPublish;

    /**
     * @see com.digiwin.athena.mongodb.domain.application.ApplicationConfig#getPopularSort
     */
    private Integer popularSort;

    /**
     * 前端变量
     */
    private Integer frontBizSetupPageIdx;

    /**
     * 多语言配置
     */
    private Map<String, MultiLanguageDTO<String>> lang;

    /**
     * 发布时间
     */
    @JsonFormat(pattern = DatePattern.NORM_DATETIME_PATTERN)
    private LocalDateTime publishTime;

    /**
     * 发布人
     */
    private String publishBy;

    /**
     * 创建时间
     */
    @JsonFormat(pattern = DatePattern.NORM_DATETIME_PATTERN)
    private LocalDateTime createTime;

    /**
     * 创建人
     */
    private String createBy;

    /**
     * 更新时间
     */
    @JsonFormat(pattern = DatePattern.NORM_DATETIME_PATTERN)
    private LocalDateTime editTime;

    /**
     * 更新人
     */
    private String editBy;


    public static List<String> obtainDesignerIds(List<SolutionPlanListDTO> list) {
        // 提取设计器ID列表
        return CollUtil.emptyIfNull(list).stream()
                .flatMap(v -> v.obtainDesignerIds().stream())
                .distinct().toList();
    }

    public static void sort(List<SolutionPlanListDTO> list) {
        list.sort(Comparator.comparingLong(SolutionPlanListDTO::obtainShowSort));
    }

    // 前端序列化字段
    @SuppressWarnings("unused")
    public Boolean getIsAssemblable() {
        return AssemblyType.isAssemblable(assemblyType);
    }

    public List<String> obtainDesignerIds() {
        // 提取设计器ID列表
        return CollUtil.emptyIfNull(getDesignerList()).stream()
                .map(SolutionPlanDetailDTO.DesignerBaseInfoDTO::getId)
                .distinct().collect(Collectors.toList());
    }

    /**
     * @return 在默认的添加顺序基础上，调低已下架的优先级
     */
    public long obtainShowSort() {
        return OnlineState.isOff(getStatus())
                ? Long.MAX_VALUE : Long.MAX_VALUE - LocalDateTimeUtil.toEpochMilli(getCreateTime());
    }


}
