package com.digiwin.athena.mongodb.domain.datacollect;

import com.digiwin.athena.datacollect.context.CollectContext;
import com.digiwin.athena.datacollect.model.CollectResult;
import com.digiwin.athena.datacollect.model.JobBizData;
import com.digiwin.athena.datacollect.model.JobExecData;
import com.digiwin.athena.mongodb.domain.DataCollectConfig;
import com.digiwin.athena.mongodb.domain.assetType.AssetType;
import lombok.Data;
import org.springframework.data.annotation.Id;
import org.springframework.data.mongodb.core.mapping.Document;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Optional;

/**
 * 任务执行记录日志
 */
@Data
@Document(collection = "job_execution_record")
public class JobExecutionRecord {

    /**
     * 主键
     */
    @Id
    private String id;

    /**
     * 任务名称
     */
    private String jobName;

    /**
     * 触发时间
     */
    private LocalDateTime fireTime;

    /**
     * 结束时间
     */
    private LocalDateTime finishTime;

    /**
     * 执行结果: SUCCESS/FAILURE/RUNNING
     */
    private String status;

    /**
     * 业务唯一标识（资产类型ID - AssetType.objId）
     */
    private String bizKey;

    /**
     * 业务数据
     */
    private JobBizData bizData;

    /**
     * 下次执行需要的数据
     */
    private JobExecData nextExecData;

    /**
     * 异常信息
     */
    private String errorMessage;

    /**
     * 执行耗时（毫秒）
     */
    private Long durationMs;

    /**
     * 创建时间
     */
    private LocalDateTime createdAt;

    /**
     * 执行状态常量
     */
    public static class Status {
        public static final String SUCCESS = "SUCCESS";
        public static final String FAILURE = "FAILURE";
        public static final String RUNNING = "RUNNING";
    }

    public static JobExecutionRecord initExecRecord(String jobName, LocalDateTime fireTime, String assetTypeKey) {
        JobExecutionRecord record = new JobExecutionRecord();

        record.setJobName(jobName);
        record.setBizKey(assetTypeKey);

        record.setFireTime(fireTime);
        record.setCreatedAt(LocalDateTime.now());
        record.setStatus(JobExecutionRecord.Status.RUNNING);

        return record;
    }

    public LocalDateTime obtainConfigUpdateTime() {
        return Optional.ofNullable(getBizData())
                .map(JobBizData::getConfigUpdateTime).orElse(null);
    }

    public void fail(CollectContext collectContext, String errorMessage) {
        setStatus(JobExecutionRecord.Status.FAILURE);

        setFinishTime(LocalDateTime.now());
        setDurationMs(ChronoUnit.MILLIS.between(getFireTime(), getFinishTime()));

        setErrorMessage(errorMessage);

        // 失败时保存Context以支持续查
        if (collectContext != null) {
            JobExecData failedNextData = buildNextData(collectContext);
            setNextExecData(failedNextData);
        }
    }


    public void complete(DataCollectConfig config, AssetType assetType, CollectContext context, CollectResult result) {
        // 8. 记录执行结果
        setStatus(result.isSuccess() ?
                JobExecutionRecord.Status.SUCCESS : JobExecutionRecord.Status.FAILURE);
        setFinishTime(result.getEndTime());
        setDurationMs(ChronoUnit.MILLIS.between(getFireTime(), getFinishTime()));

        // 9. 设置业务数据
        JobBizData bizData = buildBizData(assetType, config, context);
        setBizData(bizData);

        // 10. 设置下次执行数据
        JobExecData nextData = buildNextData(result, context);
        setNextExecData(nextData);

        if (!result.isSuccess()) {
            setErrorMessage(result.getErrorMessage());
        }
    }

    /**
     * 构建业务数据
     */
    protected JobBizData buildBizData(AssetType assetType, DataCollectConfig config, CollectContext context) {
        JobBizData bizData = new JobBizData();

        bizData.setAssetTypeId(assetType.getKey());
        bizData.setAssetType(assetType.getType());
        bizData.setAssetTypeName(assetType.getName());

        bizData.setConfigUpdateTime(config.getConfigUpdateTime());
        bizData.setStatus(assetType.getStatus());

        bizData.setTotalCount(context.getTotalRecords());
        bizData.setConsumedCount(context.getConsumedRecords());
        bizData.setTotalPages(context.getTotalPages());
        bizData.setTotalSteps(context.getTotalSteps());

        return bizData;
    }

    /**
     * 构建下次执行数据
     */
    protected JobExecData buildNextData(CollectResult result, CollectContext context) {
        JobExecData nextData = new JobExecData();

        if (result.isSuccess()) {
            // 成功时保存最后采集时间和startTime
            nextData.setLastCollectTime(result.getEndTime());
            nextData.setStartTime(context.getCurrentStepEndTime());
        } else {
            // 失败时保存完整Context信息以支持续查
            nextData = buildNextData(context);
        }

        return nextData;
    }

    /**
     * 构建失败续查数据（异常情况）
     */
    protected JobExecData buildNextData(CollectContext context) {
        JobExecData nextData = new JobExecData();

        nextData.setLastCollectTime(context.getFireTime());

        nextData.setStartTime(context.getStartTime());
        nextData.setEndTime(context.getEndTime());

        nextData.setPageNo(context.getPageNo());
        nextData.setLoopIndex(context.getLoopIndex());

        nextData.setStepStartTime(context.getCurrentStepStartTime());
        nextData.setStepEndTime(context.getCurrentStepEndTime());

        return nextData;
    }


}
