package com.digiwin.athena.service.assembly.impl;

import cn.hutool.core.collection.CollStreamUtil;
import cn.hutool.core.collection.CollUtil;
import com.digiwin.athena.base.BusinessException;
import com.digiwin.athena.bo.assembly.Designer;
import com.digiwin.athena.constants.AssemblyErrorCode;
import com.digiwin.athena.constants.enums.assembly.OnlineState;
import com.digiwin.athena.convertor.assembly.DesignerConvertor;
import com.digiwin.athena.convertor.assembly.SolutionConvertor;
import com.digiwin.athena.dao.mongodao.assembly.DDesignerMongoDao;
import com.digiwin.athena.dao.mongodao.assembly.PublishableContext;
import com.digiwin.athena.dao.mongodao.assembly.RDesignerMongoDao;
import com.digiwin.athena.dto.PageReqCondition;
import com.digiwin.athena.dto.Pagination;
import com.digiwin.athena.dto.assembly.designer.DesignerDetailDTO;
import com.digiwin.athena.dto.assembly.designer.DesignerListDTO;
import com.digiwin.athena.dto.assembly.designer.DesignerPageQo;
import com.digiwin.athena.dto.assembly.designer.DesignerSaveDTO;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanListDTO;
import com.digiwin.athena.dto.assetType.AssetTypeBaseDTO;
import com.digiwin.athena.dto.assetType.AssetTypeResDto;
import com.digiwin.athena.service.SyncRuntime;
import com.digiwin.athena.service.assembly.DDesignerService;
import com.digiwin.athena.service.assembly.RSolutionPlanService;
import com.digiwin.athena.service.assetType.RAssetTypeService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Map;
import java.util.Optional;


/**
 * 设计器管理服务实现
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class DDesignerServiceImpl extends SyncRuntime<Designer> implements DDesignerService {

    protected final DDesignerMongoDao dDesignerMongoDao;
    protected final RDesignerMongoDao rDesignerMongoDao;
    protected RSolutionPlanService rSolutionPlanService;
    protected RAssetTypeService rAssetTypeService;

    protected DDesignerServiceImpl proxyThis;

    @Override
    public Pagination<DesignerListDTO> findList(PageReqCondition<DesignerPageQo> param) {
        Pagination<DesignerListDTO> pagination = dDesignerMongoDao.selectPage(param);

        fillNewestSolutionPlan(pagination.getData());
        fillNewestAssetType(pagination.getData());

        return pagination;
    }

    @Transactional(rollbackFor = Throwable.class)
    @Override
    public void save(Designer updated) {
        log.info("开始创建设计时态设计器: {}", updated.getName());

        // 验证type字段格式
        updated.validateTypeFormat();

        checkUnique(updated);

        String objectId = updated.getId();
        if (objectId != null) {
            Designer existingBo = findExist(objectId);

            existingBo.overwriteUnmodifiable(updated);
        } else {
            updated.prepareNew();
        }

        dDesignerMongoDao.save(updated);
        log.info("成功创建设计时态设计器: {}", updated.getId());
    }

    /**
     * 单纯保存，不做任何业务处理
     */
    protected void pureSave(Designer designer) {
        dDesignerMongoDao.save(designer);
    }

    @Transactional(rollbackFor = Throwable.class)
    @Override
    public Designer save(DesignerSaveDTO designer) {
        Designer bo = DesignerConvertor.INSTANCE.toBo(designer);

        save(bo);

        return bo;
    }


    @Transactional(rollbackFor = Throwable.class)
    @Override
    public void deleteById(String objectId) {
        log.info("开始删除设计时态设计器: {}", objectId);

        // 检查设计器是否存在
        Designer existingBo = findExist(objectId);
        // 检查状态是否为草稿
        existingBo.checkDelete();

        dDesignerMongoDao.deleteById(objectId);
        log.info("成功删除设计时态设计器: {}", objectId);
    }

    protected void checkUnique(Designer designer) {
        if (!validateUnique(designer.getType(), null, designer.getId())) {
            throw new BusinessException(AssemblyErrorCode.NAME_CODE_NOT_UNIQUE, "类型已存在！");
        }

        if (!validateUnique(null, designer.getName(), designer.getId())) {
            throw new BusinessException(AssemblyErrorCode.NAME_CODE_NOT_UNIQUE, "名称已存在！");
        }
    }

    protected Designer findExist(String objectId) {
        return Optional
                .ofNullable(dDesignerMongoDao.selectById(objectId))
                .orElseThrow(() -> new BusinessException(AssemblyErrorCode.DESIGNER_NOT_EXIST, "设计器不存在！", objectId));
    }

    @Override
    public DesignerDetailDTO findDetailById(String designerId) {
        Map<String, List<SolutionPlanListDTO>> solutionMap = rSolutionPlanService.findBaseListByDesignerId(CollUtil.toList(designerId));

        DesignerDetailDTO dto = DesignerConvertor.INSTANCE.toDto(findById(designerId));
        dto.setRelatedSolutionPlan(SolutionConvertor.INSTANCE.toBaseDTO(solutionMap.get(designerId)));
        fillNewestAssetType(dto);

        return dto;
    }

    @Override
    public Designer findById(String objectId) {
        log.debug("查询设计时态设计器详情: {}", objectId);
        return dDesignerMongoDao.selectById(objectId);

    }

    protected void fillNewestAssetType(DesignerDetailDTO designer) {
        DesignerDetailDTO.RelatedAssetTypeDTO relatedAssetType = designer.getRelatedAssetType();
        List<String> asstTypeIdList = designer.obtainAssetTypeIdList();
        List<AssetTypeResDto> assetTypes = rAssetTypeService.getAssetTypes(asstTypeIdList);
        Map<String, AssetTypeResDto> assetTypeIdMap = CollStreamUtil.toIdentityMap(assetTypes, AssetTypeResDto::getObjId);
        relatedAssetType.setWrite(AssetTypeBaseDTO.sort(relatedAssetType.getWrite(), assetTypeIdMap));
        relatedAssetType.setRead(AssetTypeBaseDTO.sort(relatedAssetType.getRead(), assetTypeIdMap));
    }

    protected void fillNewestAssetType(List<DesignerListDTO> designers) {
        List<String> asstTypeIdList = DesignerListDTO.obtainAssetTypeIdList(designers);
        List<AssetTypeResDto> assetTypes = rAssetTypeService.getAssetTypes(asstTypeIdList);
        Map<String, AssetTypeResDto> assetTypeIdMap = CollStreamUtil.toIdentityMap(assetTypes, AssetTypeResDto::getObjId);
        designers.forEach(designer -> {
            designer.setWriteAssetType(AssetTypeBaseDTO.sort(designer.getWriteAssetType(), assetTypeIdMap));
            designer.setReadAssetType(AssetTypeBaseDTO.sort(designer.getReadAssetType(), assetTypeIdMap));
        });
    }

    protected void fillNewestSolutionPlan(List<DesignerListDTO> list) {
        List<String> designerIds = DesignerListDTO.obtainIdList(list);

        Map<String, List<SolutionPlanListDTO>> solutionMap = rSolutionPlanService.findBaseListByDesignerId(designerIds);
        // 将解决方案信息添加到设计器DTO中
        for (DesignerListDTO designer : list) {
            List<SolutionPlanListDTO> solutionPlan = solutionMap.get(designer.getId());
            designer.setRelatedSolutionPlan(SolutionConvertor.INSTANCE.toBaseDTO(solutionPlan));
        }
    }

    @Override
    public boolean validateUnique(String code, String name, String excludeObjectId) {
        return !dDesignerMongoDao.existsByKey(code, name, excludeObjectId);
    }

    @Override
    public List<DesignerListDTO> findPublishedList() {
        PageReqCondition<DesignerPageQo> param = new PageReqCondition<>();
        param.setPageNum(1);
        param.setPageSize(9999);
        param.setCondition(new DesignerPageQo());

        // 设置状态列表为已上架和已下架
        param.getCondition().setStatusList(OnlineState.publishedCode());

        return PublishableContext.withRuntime(() -> findList(param).getData());
    }

    @Override
    public List<Designer> findWritableDesignersByIds(List<String> ids) {
        return dDesignerMongoDao.selectWritableDesignersByIds(ids);
    }

    @Override
    protected void saveDesignerTimeData(Designer syncData) {
        pureSave(syncData);
    }

    @Override
    protected void insertRunTimeData(Designer syncData) {
        rDesignerMongoDao.save(syncData);
    }

    @Override
    protected void saveRunTimeData(Designer syncData) {
        rDesignerMongoDao.save(syncData);
    }

    @Override
    protected Designer getDesignerTimeData(String objId) {
        return proxyThis.findById(objId);
    }

    @Override
    protected Designer getRunTimeData(String objId) {
        return rDesignerMongoDao.selectById(objId);
    }

    @Override
    protected void deleteRuntimeData(String objId) {
        rDesignerMongoDao.deleteById(objId);
    }

    @Autowired
    @Lazy
    public void setProxyThis(DDesignerServiceImpl proxyThis) {
        this.proxyThis = proxyThis;
    }

    @Autowired
    @Lazy
    public void seRSolutionPlanService(RSolutionPlanService rSolutionPlanService) {
        this.rSolutionPlanService = rSolutionPlanService;
    }

    @Autowired
    @Lazy
    public void setRAssetTypeService(RAssetTypeService rAssetTypeService) {
        this.rAssetTypeService = rAssetTypeService;
    }
}