package com.digiwin.athena.dao.mongodao.asset;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.digiwin.athena.dto.PageReqCondition;
import com.digiwin.athena.dto.TenantInfo;
import com.digiwin.athena.dto.asset.UseAssetListReqDto;
import com.digiwin.athena.http.asset.PermissionConditionResDto;
import com.digiwin.athena.mongodb.domain.application.Asset;
import com.digiwin.athena.mongodb.domain.application.AssetDefinition;
import com.digiwin.athena.mongodb.repository.MongoPrimaryRepositoryDecorator;
import com.digiwin.athena.utils.CurThreadInfoUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Repository;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

@Slf4j
@Repository
public class AssetDefinitionMongoDao {
    private MongoPrimaryRepositoryDecorator mongoPrimaryRepositoryDecorator;

    @Autowired
    public void setMongoPrimaryRepositoryDecorator(MongoPrimaryRepositoryDecorator mongoPrimaryRepositoryDecorator) {
        this.mongoPrimaryRepositoryDecorator = mongoPrimaryRepositoryDecorator;
    }

    public AssetDefinition selectByApplicationAndCode(String code, String application) {
        Criteria criteria = Criteria.where("code").is(code).and("manageContext.application").is(application);
        Query query = new Query(criteria);
        return mongoPrimaryRepositoryDecorator.findOne(query, AssetDefinition.class);
    }

    public void insert(AssetDefinition assetDefinition) {
        mongoPrimaryRepositoryDecorator.insert(assetDefinition);
    }

    public AssetDefinition selectByAssetId(String assetId) {
        Criteria criteria = Criteria.where("assetId").is(assetId);
        Query query = new Query(criteria);
        return mongoPrimaryRepositoryDecorator.findOne(query, AssetDefinition.class);
    }

    public void updateStatus(Integer status, String editBy, String assetId) {

        Update update = new Update();
        update.set("status", status);
        update.set("editDate", new Date());
        update.set("editBy", editBy);

        Criteria criteria = Criteria.where("assetId").is(assetId);
        mongoPrimaryRepositoryDecorator.updateMulti(new Query(criteria), update, AssetDefinition.class);
    }

    public void save(AssetDefinition assetDefinition) {
        mongoPrimaryRepositoryDecorator.save(assetDefinition);
    }

    public List<AssetDefinition> selectByAssetIds(Collection<String> assetIds) {
        Criteria criteria = Criteria.where("assetId").in(assetIds);
        Query query = new Query(criteria);
        return mongoPrimaryRepositoryDecorator.find(query, AssetDefinition.class);
    }

    public List<AssetDefinition> selectPagination(PageReqCondition<String> pageReqCondition,String tenantId,List<String> types,PermissionConditionResDto permissionConditionResDto) {

        Criteria criteria = buildPageCondition(pageReqCondition,tenantId,types);

        if (permissionConditionResDto!=null){
            Criteria perCriteria = buildPermissionConditionResDto(permissionConditionResDto);
            if (perCriteria != null) {
                Criteria newCriteria = new Criteria();
                newCriteria.andOperator(criteria,perCriteria);
                criteria = newCriteria;
            }
        }

        Query query = new Query(criteria);
        query.limit(pageReqCondition.getPageSize());
        query.skip((pageReqCondition.getPageNum() - 1) * pageReqCondition.getPageSize());
        query.with(Sort.by(Sort.Direction.DESC,"_id"));

        return mongoPrimaryRepositoryDecorator.find(query, AssetDefinition.class);
    }

    public Long countPagination(PageReqCondition<String> pageReqCondition, String tenantId, List<String> types, PermissionConditionResDto permissionConditionResDto){
        Criteria criteria = buildPageCondition(pageReqCondition,tenantId,types);
        if (permissionConditionResDto!=null){
            Criteria perCriteria = buildPermissionConditionResDto(permissionConditionResDto);
            if (perCriteria != null) {
                Criteria newCriteria = new Criteria();
                newCriteria.andOperator(criteria,perCriteria);
                criteria = newCriteria;
            }
        }

        Query query = new Query(criteria);

        return mongoPrimaryRepositoryDecorator.count(query, AssetDefinition.class);
    }

    private Criteria buildPermissionConditionResDto(PermissionConditionResDto permissionConditionResDto){
        List<Criteria> conditions = new ArrayList<>();

        List<PermissionConditionResDto.Condition> fieldConditions = permissionConditionResDto.getChildren();
        if (!CollectionUtils.isEmpty(fieldConditions)){
            for (PermissionConditionResDto.Condition fieldCondition : fieldConditions) {
                if (fieldCondition.getOperator().equals("in")){
                    List values = JSONObject.parseObject(JSONObject.toJSONString(fieldCondition.getValue()), new TypeReference<List>() {
                    });

                    conditions.add(Criteria.where("manageContext."+fieldCondition.getField()).in(values));
                }else if (fieldCondition.getOperator().equals("equal")){
                    conditions.add(Criteria.where("manageContext."+fieldCondition.getField()).is(fieldCondition.getValue()));
                }else if (fieldCondition.getOperator().equals("nin")){
                    List values = JSONObject.parseObject(JSONObject.toJSONString(fieldCondition.getValue()), new TypeReference<List>() {
                    });
                    conditions.add(Criteria.where("manageContext."+fieldCondition.getField()).nin(values));
                }
            }
        }else{
            return null;
        }
        if (!conditions.isEmpty()){
            Criteria criteria = new Criteria();
            if ("and".equals(permissionConditionResDto.getLogic())){
                return criteria.andOperator(conditions);
            }else if ("or".equals(permissionConditionResDto.getLogic())){
                return criteria.orOperator(conditions);
            }
        }
        return null;
    }

    public Long countApplicationPagination(PageReqCondition<String> pageReqCondition,List<String> appCodes,List<String> types){
        Criteria criteria = buildAdvancedCondition(pageReqCondition);

        if (criteria == null){
            criteria = Criteria.where("manageContext.application").in(appCodes);
            if (!CollectionUtils.isEmpty(types)){
                criteria = criteria.and("type").in(types);
            }
        }else{
            List<Criteria> criteriaList = new ArrayList<>();
            criteriaList.add(criteria);
            criteriaList.add(Criteria.where("manageContext.application").in(appCodes));
            if (!CollectionUtils.isEmpty(types)){
                criteriaList.add(Criteria.where("type").in(types));
            }

            criteria = new Criteria().andOperator(criteriaList);
        }
        Query query = new Query(criteria);

        return mongoPrimaryRepositoryDecorator.count(query, AssetDefinition.class);
    }

    public List<AssetDefinition> selectApplicationPagination(PageReqCondition<String> pageReqCondition,List<String> appCodes,List<String> types) {

        Criteria criteria = buildAdvancedCondition(pageReqCondition);

        if (criteria == null){
            criteria = Criteria.where("manageContext.application").in(appCodes);
            if (!CollectionUtils.isEmpty(types)){
                criteria = criteria.and("type").in(types);
            }
        }else{
            Criteria criteria1 = Criteria.where("manageContext.application").in(appCodes);
            if (!CollectionUtils.isEmpty(types)){
                criteria1.and("type").in(types);
            }

            criteria = new Criteria().andOperator(criteria1,criteria);
        }

        Query query = new Query(criteria);
        query.limit(pageReqCondition.getPageSize());
        query.skip((pageReqCondition.getPageNum() - 1) * pageReqCondition.getPageSize());
        query.with(Sort.by(Sort.Direction.DESC,"_id"));

        return mongoPrimaryRepositoryDecorator.find(query, AssetDefinition.class);
    }

    private Criteria buildPageCondition(PageReqCondition<String> pageReqCondition,String tenantId,List<String> types) {
        Criteria criteria = buildAdvancedCondition(pageReqCondition);

        Criteria rangeCriteria = new Criteria();
        Criteria criteria1 = Criteria.where("openRange").is(AssetDefinition.ALL_RANGE);
        Criteria criteria2 = Criteria.where("openRange").is(AssetDefinition.CUR_TENANT_RANGE).and("adpTenantId").is(tenantId);
        Criteria criteria3 = Criteria.where("openRange").is(AssetDefinition.TENANT_RANGE).and("permission.tenantInfos.tenantId").in(tenantId);
        rangeCriteria.orOperator(criteria3,criteria1,criteria2);

        List<Criteria> criteriaList = new ArrayList<>();
        criteriaList.add(rangeCriteria);
        criteriaList.add(Criteria.where("status").is(1));
        if (!CollectionUtils.isEmpty(types)){
            criteriaList.add(Criteria.where("type").in(types));
        }

        if (criteria == null){
            criteria = new Criteria();
            criteria.andOperator(criteriaList);
        }else{
            criteriaList.add(criteria);
            criteria = new Criteria().andOperator(criteriaList);
        }

        return criteria;
    }

    private Criteria buildAdvancedCondition(PageReqCondition<String> pageReqCondition) {
        Criteria criteria = null;
        if (pageReqCondition.getAdvancedQueryParams() != null) {
            Criteria adVancedCriteria = pageReqCondition.getAdvancedQueryParams().buildCriteria();
            if (adVancedCriteria != null) {
                criteria = adVancedCriteria;
            }
        }

        if (criteria == null && StringUtils.isNotEmpty(pageReqCondition.getCondition())) {
            Pattern pattern = Pattern.compile(".*" + Pattern.quote(pageReqCondition.getCondition()) + ".*", Pattern.CASE_INSENSITIVE);

            criteria = new Criteria();
            criteria.orOperator(Criteria.where("assetId").regex(pattern),
                    Criteria.where("code").regex(pattern),
                    Criteria.where("assetName").regex(pattern),
                    Criteria.where("assetDesc").regex(pattern));
        }
        return criteria;
    }

    public List<String> selectAssetIdAvailableStatusByApplication(String appCode, String type) {

        Criteria criteria = Criteria.where("manageContext.application").is(appCode).and("status").is(AssetDefinition.AVAILABLE_STATUS).and("type").is(type);
        Query query = new Query(criteria);
        query.fields().include("assetId");
        return mongoPrimaryRepositoryDecorator.find(query, AssetDefinition.class).stream().map(AssetDefinition::getAssetId).collect(Collectors.toList());
    }

    public Long countSameTenantDtdAsset(PageReqCondition<String> pageReqCondition, String tenantId, String type) {
        Criteria criteria = buildAdvancedCondition(pageReqCondition);
        if (criteria == null){
            criteria = Criteria.where("adpTenantId").is(tenantId).and("type").is(type);
        }else{
            criteria = new Criteria().andOperator(Criteria.where("adpTenantId").is(tenantId), Criteria.where("type").is(type), criteria);
        }

        Query query = new Query(criteria);
        query.limit(pageReqCondition.getPageSize());
        query.skip((pageReqCondition.getPageNum() - 1) * pageReqCondition.getPageSize());
        query.with(Sort.by(Sort.Direction.ASC,"_id"));

        return mongoPrimaryRepositoryDecorator.count(query,AssetDefinition.class);
    }

    public List<AssetDefinition> selectSameTenantDtdAsset(PageReqCondition<String> pageReqCondition, String tenantId, String type) {
        Criteria criteria = buildAdvancedCondition(pageReqCondition);
        if (criteria == null){
            criteria = Criteria.where("adpTenantId").is(tenantId).and("type").is(type);
        }else{
            criteria = new Criteria().andOperator(Criteria.where("adpTenantId").is(tenantId), Criteria.where("type").is(type), criteria);
        }

        Query query = new Query(criteria);
        query.limit(pageReqCondition.getPageSize());
        query.skip((pageReqCondition.getPageNum() - 1) * pageReqCondition.getPageSize());
        query.with(Sort.by(Sort.Direction.ASC,"_id"));

        return mongoPrimaryRepositoryDecorator.find(query,AssetDefinition.class);
    }

    public Long countPlatformDtdAsset(PageReqCondition<String> pageReqCondition, String tenantId, String type) {
        Criteria criteria = buildAdvancedCondition(pageReqCondition);
        if (criteria == null){
            criteria = new Criteria().orOperator(Criteria.where("adpTenantId").ne(tenantId)
                    .and("type").is(type).and("openRange").is("all"),
                    Criteria.where("adpTenantId").ne(tenantId)
                    .and("type").is(type).and("openRange").is("tenant").and("permission.tenantInfos.tenantId").in(tenantId));
        }else{
            Criteria criteria2 = new Criteria().orOperator(Criteria.where("adpTenantId").ne(tenantId)
                            .and("type").is(type).and("openRange").is("all"),
                    Criteria.where("adpTenantId").ne(tenantId)
                            .and("type").is(type).and("openRange").is("tenant").and("permission.tenantInfos.tenantId").in(tenantId));

            criteria = new Criteria().andOperator(criteria2, criteria);
        }

        Query query = new Query(criteria);
        query.limit(pageReqCondition.getPageSize());
        query.skip((pageReqCondition.getPageNum() - 1) * pageReqCondition.getPageSize());
        query.with(Sort.by(Sort.Direction.ASC,"_id"));

        return mongoPrimaryRepositoryDecorator.count(query,AssetDefinition.class);
    }

    public List<AssetDefinition> selectPlatformDtdAsset(PageReqCondition<String> pageReqCondition, String tenantId, String type) {
        Criteria criteria = buildAdvancedCondition(pageReqCondition);
        if (criteria == null){
            criteria = new Criteria().orOperator(Criteria.where("adpTenantId").ne(tenantId)
                            .and("type").is(type).and("openRange").is("all"),
                    Criteria.where("adpTenantId").ne(tenantId)
                            .and("type").is(type).and("openRange").is("tenant").and("permission.tenantInfos.tenantId").in(tenantId));
        }else{
            Criteria criteria2 = new Criteria().orOperator(Criteria.where("adpTenantId").ne(tenantId)
                            .and("type").is(type).and("openRange").is("all"),
                    Criteria.where("adpTenantId").ne(tenantId)
                            .and("type").is(type).and("openRange").is("tenant").and("permission.tenantInfos.tenantId").in(tenantId));

            criteria = new Criteria().andOperator(criteria2, criteria);
        }

        Query query = new Query(criteria);
        query.limit(pageReqCondition.getPageSize());
        query.skip((pageReqCondition.getPageNum() - 1) * pageReqCondition.getPageSize());
        query.with(Sort.by(Sort.Direction.ASC,"_id"));

        return mongoPrimaryRepositoryDecorator.find(query,AssetDefinition.class);
    }

    public List<AssetDefinition> selectBasicByApplicationAndCodes(String application, List<String> codes) {

        Criteria criteria = Criteria.where("code").in(codes).and("manageContext.application").is(application);
        Query query = new Query(criteria);
        query.fields().include("assetId","code","type","assetName","manageContext.application");
        return mongoPrimaryRepositoryDecorator.find(query, AssetDefinition.class);
    }

    public List<AssetDefinition> selectByAssetIdsAndStatus(List<String> assetIds, Integer status) {
        Criteria criteria = Criteria.where("assetId").in(assetIds).and("status").is(status);
        Query query = new Query(criteria);
        return mongoPrimaryRepositoryDecorator.find(query, AssetDefinition.class);
    }

    public Long countAppUseAssetList(PageReqCondition<UseAssetListReqDto> pageReqCondition) {
        UseAssetListReqDto condition = pageReqCondition.getCondition();
        List<Criteria> criteriaList = buildAppUserAssetListCondition(condition);


        Query query = new Query(new Criteria().andOperator(criteriaList));
        return mongoPrimaryRepositoryDecorator.count(query,AssetDefinition.class);
    }

    public List<AssetDefinition> selectAppUseAssetList(PageReqCondition<UseAssetListReqDto> pageReqCondition) {
        UseAssetListReqDto condition = pageReqCondition.getCondition();
        List<Criteria> criteriaList = buildAppUserAssetListCondition(condition);
        Query query = new Query(new Criteria().andOperator(criteriaList));
        query.with(Sort.by(Sort.Direction.DESC,"_id"));
        query.limit(pageReqCondition.getPageSize());
        query.skip((pageReqCondition.getPageNum() - 1) * pageReqCondition.getPageSize());

        return mongoPrimaryRepositoryDecorator.find(query,AssetDefinition.class);
    }

    private static List<Criteria> buildAppUserAssetListCondition(UseAssetListReqDto condition) {
        List<Criteria> criteriaList = new ArrayList<>();
        Criteria criteria1 = new Criteria();
        criteria1.and("status").is(AssetDefinition.AVAILABLE_STATUS);
        Criteria criteria2 = new Criteria();
        criteria2.orOperator(Criteria.where("canReference").is(true),Criteria.where("canCopy").is(true));
        criteriaList.add(criteria1);
        criteriaList.add(criteria2);

        if (!condition.getType().equals(UseAssetListReqDto.ALL_TYPE)){
            Criteria criteria = new Criteria();
            criteria.and("type").is(condition.getType());
            criteriaList.add(criteria);
        }

        TenantInfo curTokenTenant = CurThreadInfoUtils.getCurTokenTenant();
        if (condition.getRange().equals(UseAssetListReqDto.CUR_TENANT_RANGE)){
            Criteria criteria = new Criteria();
            criteriaList.add(criteria);
            criteria.orOperator(Criteria.where("adpTenantId").is(curTokenTenant.getTenantId()).and("openRange").is(AssetDefinition.ALL_RANGE),
                    Criteria.where("openRange").is(AssetDefinition.TENANT_RANGE).and("permission.tenantInfos.tenantId").in(curTokenTenant.getTenantId()).and("adpTenantId").is(curTokenTenant.getTenantId()),
                    Criteria.where("openRange").is(AssetDefinition.CUR_TENANT_RANGE).and("adpTenantId").is(curTokenTenant.getTenantId()));
        }else{
            Criteria criteria = new Criteria();
            criteriaList.add(criteria);
            criteria.orOperator(Criteria.where("adpTenantId").ne(curTokenTenant.getTenantId()).and("openRange").is(AssetDefinition.ALL_RANGE),
                    Criteria.where("openRange").is(AssetDefinition.TENANT_RANGE).and("permission.tenantInfos.tenantId").in(curTokenTenant.getTenantId()).and("adpTenantId").ne(curTokenTenant.getTenantId()));
        }

        if (Boolean.TRUE.equals(condition.getCanCopy())){
            Criteria criteria = new Criteria();
            criteriaList.add(criteria);
            criteria.and("canCopy").is(true);
        }
        if (Boolean.TRUE.equals(condition.getCanReference())){
            Criteria criteria = new Criteria();
            criteriaList.add(criteria);
            criteria.and("canReference").is(true);
        }

        if (StringUtils.isNotEmpty(condition.getSearchContent())){
            Pattern pattern = Pattern.compile(".*" + Pattern.quote(condition.getSearchContent()) + ".*", Pattern.CASE_INSENSITIVE);

            Criteria criteria = new Criteria();
            criteriaList.add(criteria);
            criteria.orOperator(Criteria.where("assetId").regex(pattern),
                    Criteria.where("assetName").regex(pattern),
                    Criteria.where("assetDesc").regex(pattern),
                    Criteria.where("publisherName").regex(pattern));
        }
        return criteriaList;
    }

    public AssetDefinition selectByCodeAndType(String code, String type) {
        Criteria criteria = Criteria.where("code").is(code).and("type").is(type);
        return mongoPrimaryRepositoryDecorator.findOne(new Query(criteria),AssetDefinition.class);
    }

    public List<AssetDefinition> selectByCodesAndType(Collection codes, String type) {
        Criteria criteria = Criteria.where("code").in(codes).and("type").is(type);
        return mongoPrimaryRepositoryDecorator.find(new Query(criteria),AssetDefinition.class);
    }

    public List<String> selectAllAssetId() {
        Criteria criteria = new Criteria();
        Query query = new Query(criteria);
        query.fields().include("assetId");
        return mongoPrimaryRepositoryDecorator.find(query,AssetDefinition.class).stream().map(AssetDefinition::getAssetId).collect(Collectors.toList());
    }

    public void delete(AssetDefinition assetDefinition) {
        mongoPrimaryRepositoryDecorator.delete(assetDefinition);
    }

    /**
     * 批量查询已存在的assetId列表（只查询assetId字段，提升性能）
     *
     * @param assetIds 资产ID列表
     * @return 已存在的assetId列表
     */
    public Set<String> selectExistAssetIds(Collection<String> assetIds) {
        if (CollectionUtils.isEmpty(assetIds)) {
            return Collections.emptySet();
        }

        Criteria criteria = Criteria.where("assetId").in(assetIds);
        Query query = new Query(criteria);
        query.fields().include("assetId");

        return mongoPrimaryRepositoryDecorator.find(query, AssetDefinition.class).stream()
                .map(AssetDefinition::getAssetId)
                .collect(Collectors.toSet());
    }

    /**
     * 批量插入AssetDefinition
     *
     * @param definitions AssetDefinition列表
     */
    public void insertAll(List<AssetDefinition> definitions) {
        if (!CollectionUtils.isEmpty(definitions)) {
            mongoPrimaryRepositoryDecorator.insertAll(definitions);
        }
    }

    public void deleteBySourceId(Asset asset) {

    }

    public List<AssetDefinition> selectByApplication(String application) {
        Criteria criteria = Criteria.where("manageContext.application").is(application);
        Query query = new Query(criteria);

        return mongoPrimaryRepositoryDecorator.find(query, AssetDefinition.class);
    }
}
