package com.digiwin.athena.convertor;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.bean.copier.CopyOptions;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.date.LocalDateTimeUtil;
import cn.hutool.core.util.StrUtil;
import com.digiwin.athena.common.dict.Dict;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.mapstruct.Mapper;
import org.mapstruct.Named;
import org.mapstruct.factory.Mappers;
import org.springframework.beans.factory.annotation.Autowired;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;

@Mapper(componentModel = "spring")
@Slf4j
public class BaseConverter {
    public static BaseConverter INSTANCE = Mappers.getMapper(BaseConverter.class);
    protected static ObjectMapper objectMapper;

    public Integer toInt(String intStr) {
        if (StrUtil.isBlank(intStr)) {
            return null;
        }

        return Integer.parseInt(intStr);
    }

    public LocalDateTime date2LocalDateTime(Date date) {
        if (date == null) {
            return null;
        }
        return LocalDateTimeUtil.of(date);
    }

    public Date localDateTime2Date(LocalDateTime localDateTime) {
        if (localDateTime == null) {
            return null;
        }
        return DateUtil.date(localDateTime);
    }

    @Named("shallowMap")
    public <K, V> Map<K, V> shallowMap(Map<K, V> map) {
        return map;
    }

    @Named("shallowColl")
    public <V> Collection<V> directList(Collection<V> obj) {
        return obj;
    }

    @Named("jsonSerialize")
    public String serialize(Object obj) {
        if (obj == null) {
            return null;
        }

        try {
            return objectMapper.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            log.warn("json序列化异常！msg:{}", e.getMessage());
            return null;
        }
    }

    public <T> List<T> deserializeList(String jsonStr, Class<T> clazz) {
        return deserialize(jsonStr, List.class, clazz);
    }

    public <K, T> Map<K, T> deserializeMap(String jsonStr, Class<K> keyClazz, Class<T> clazz) {
        return deserialize(jsonStr, Map.class, keyClazz, clazz);
    }

    public <T> T deserialize(String jsonStr, Class<T> clazz) {
        try {
            return objectMapper.readValue(jsonStr, clazz);
        } catch (JsonProcessingException e) {
            log.warn("json反序列化异常！msg:{}", e.getMessage());
            return null;
        }
    }

    public <T> T deserialize(String json, Class<?> clazz, Class<?>... genericClass) {
        try {
            JavaType javaType = objectMapper.getTypeFactory().constructParametricType(clazz, genericClass);
            return objectMapper.readValue(json, javaType);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("json反序列化异常！", e);
        }
    }

    /**
     * 复制并转换实体类型，支持递归转换
     */
    public <T> List<T> copyToList(Collection<?> collection, Class<T> targetType) {
        return BeanUtil.copyToList(collection, targetType, CopyOptions.create());
    }

    /**
     * 扫描对象中标记了@Dict注解的字段，根据字段值从枚举类中获取多语言文本
     * 并添加到对象的lang字段中
     *
     * @param obj 要处理的对象
     */
    public void translateDictField(Object obj) {
        try {
            Dict.Translator.translate(obj);
        } catch (Exception e) {
            log.error("翻译字典值异常！", e);
        }
    }

    @Autowired
    public void setObjectMapper(ObjectMapper objectMapper) {
        BaseConverter.objectMapper = objectMapper;
    }
}
