package com.digiwin.athena.dao.mongodao.asset;

import com.digiwin.athena.config.BranchConfig;
import com.digiwin.athena.enums.VersionStatusEnum;
import com.digiwin.athena.mongodb.domain.application.Asset;
import com.digiwin.athena.mongodb.domain.application.AssetDefinition;
import com.digiwin.athena.mongodb.repository.MongoPrimaryRepositoryDecorator;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Repository;

import org.springframework.data.mongodb.core.query.Query;

import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;
import java.util.Set;

/**
 * 资产定义交互层
 */
@Slf4j
@Repository
public class AssetMongoDao {

    @Autowired
    private MongoPrimaryRepositoryDecorator mongoPrimaryRepositoryDecorator;

    public void save(Asset asset){
        mongoPrimaryRepositoryDecorator.save(asset);
    }

    public Asset findOne(String assetId){
        Query query = new Query(Criteria.where("assetId").is(assetId));
        return mongoPrimaryRepositoryDecorator.findOne(query, Asset.class);
    }

    public List<Asset> findByApplication(String application){
        Query query = new Query(Criteria.where("application").is(application));
        return mongoPrimaryRepositoryDecorator.find(query, Asset.class);
    }

    public List<Asset> findByAssetIds(List<String> assetIds){
        Query query = new Query(Criteria.where("assetId").in(assetIds));
        return mongoPrimaryRepositoryDecorator.find(query, Asset.class);
    }

    public void delete(String assetId){
        Query query = new Query(Criteria.where("assetId").is(assetId));
        mongoPrimaryRepositoryDecorator.delete(query, Asset.class);
    }

    public void insert(Asset asset) {
        mongoPrimaryRepositoryDecorator.insert(asset);
    }

    public void delete(Asset asset) {
        mongoPrimaryRepositoryDecorator.delete(asset);
    }

    public List<Asset> selectByCodeAndApplication(String code, String application) {
        Criteria criteria = Criteria.where("application").is(application)
                .and("code").is(code);
        return mongoPrimaryRepositoryDecorator.find(new Query(criteria),Asset.class);
    }

    public List<Asset> selectByAssetId(String assetId) {
        Criteria criteria = Criteria.where("assetId").is(assetId);
        return mongoPrimaryRepositoryDecorator.find(new Query(criteria),Asset.class);
    }

    public Asset selectByAssetIdAndSourceBranchAndAdpVersion(String assetId, String sourceBranch, String adpVersion) {
        Criteria criteria = Criteria.where("assetId").is(assetId)
                .and("sourceBranch").is(sourceBranch)
                .and("adpVersion").is(adpVersion);
        return mongoPrimaryRepositoryDecorator.findOne(new Query(criteria),Asset.class);
    }

    /**
     * 将生效版本置为失效
     * @param editBy
     * @param assetId
     */
    public void updateDraftAdpStatusByAssetId(String editBy, String assetId){
        Update update = new Update();
        update.set("adpStatus","draft");
        update.set("editDate",new Date());
        update.set("editBy",editBy);

        Criteria criteria = Criteria.where("assetId").is(assetId).and("adpStatus").is("effect");
        mongoPrimaryRepositoryDecorator.updateMulti(new Query(criteria),update,Asset.class);
    }

    public Asset selectEffectByAssetId(String assetId) {
        Criteria criteria = Criteria.where("assetId").is(assetId).and("adpStatus").is("effect");
        return mongoPrimaryRepositoryDecorator.findOne(new Query(criteria),Asset.class);
    }

    public List<Asset> selectEffectByAssetIds(Collection<String> assetIds) {
        Criteria criteria = Criteria.where("assetId").in(assetIds).and("adpStatus").is("effect");
        return mongoPrimaryRepositoryDecorator.find(new Query(criteria),Asset.class);
    }

    public List<Asset> selectBasicInfoByObjectId(List<String> objIds) {
        Criteria criteria = Criteria.where("_id").in(objIds);
        Query query = new Query(criteria);
        query.fields().exclude("businessData","content");
        return mongoPrimaryRepositoryDecorator.find(query,Asset.class);
    }

    public Asset selectByObjectId(String objId) {
        Criteria criteria = Criteria.where("_id").is(objId);
        Query query = new Query(criteria);
        return mongoPrimaryRepositoryDecorator.findOne(query,Asset.class);
    }

    public List<Asset> selectByCodeAndApplicationAndSourceBranchAndAdpVersion(Asset asset) {
        Criteria criteria = Criteria.where("code").is(asset.getCode())
                .and("application").is(asset.getApplication())
                .and("sourceBranch").is(asset.getSourceBranch());

        if (StringUtils.isNotEmpty(asset.getAdpVersion())){
            criteria.and("adpVersion").is(asset.getAdpVersion());
        }
        return mongoPrimaryRepositoryDecorator.find(new Query(criteria),Asset.class);
    }

    public void deleteByCodeAndApplicationAndSourceBranchAndAdpVersion(Asset asset) {
        Criteria criteria = Criteria.where("code").is(asset.getCode())
                .and("application").is(asset.getApplication())
                .and("sourceBranch").is(asset.getSourceBranch());

        if (StringUtils.isNotEmpty(asset.getAdpVersion())){
            criteria.and("adpVersion").is(asset.getAdpVersion());
        }
        mongoPrimaryRepositoryDecorator.delete(new Query(criteria),Asset.class);
    }

    /**
     * 查询指定分支下的资产最新的版本
     * @param assetId
     * @param branch
     * @param application
     * @return
     */
    public Asset selectLatestVersionByAssetIdAndSourceBranchAndApplication(String assetId, String branch, String application) {
        Criteria criteria = Criteria.where("assetId").is(assetId)
                .and("application").is(application)
                .and("sourceBranch").is(branch);
        Query query = new Query(criteria);
        query.with(Sort.by(Sort.Direction.DESC,"adpVersion"));
        query.limit(1);

        return mongoPrimaryRepositoryDecorator.findOne(query,Asset.class);
    }

    public Asset selectAssetIdByCodeAndApplication(String code, String application) {
        Criteria criteria = Criteria.where("application").is(application)
                .and("code").is(code);
        Query query = new Query(criteria);
        query.fields().include("assetId");
        query.limit(1);
        return mongoPrimaryRepositoryDecorator.findOne(query,Asset.class);
    }

    public List<Asset> selectByApplicationAndAdpVersionAndBranch(String application, String adpVersion, String branch) {
        Criteria criteria = Criteria.where("application").is(application);
        if (StringUtils.isNotEmpty(adpVersion)){
            criteria.and("adpVersion").is(adpVersion);
        }
        if (StringUtils.isNotEmpty(branch)){
            criteria.and("sourceBranch").is(branch);
        }
        return mongoPrimaryRepositoryDecorator.find(new Query(criteria),Asset.class);
    }

    public List<String> selectAllId() {
        Query query = new Query();
        query.fields().include("_id");
        return mongoPrimaryRepositoryDecorator.find(new Query(),Asset.class).stream().map(Asset::getObjectId).collect(Collectors.toList());
    }

    /**
     * 查询指定资产id下，有没有生效的资产版本
     * @param assetIds
     * @return
     */
    public Long selectEffectCount(List<String> assetIds) {
        Criteria criteria = Criteria.where("assetId").in(assetIds).and("adpStatus").is(VersionStatusEnum.RUN_STATE.getCode());
        return mongoPrimaryRepositoryDecorator.count(new Query(criteria),Asset.class);
    }

    public Set<String> selectExistAssetId(List<String> assetIds) {
        Criteria criteria = Criteria.where("assetId").in(assetIds);
        Query query = new Query(criteria);
        query.fields().include("assetId");
        return mongoPrimaryRepositoryDecorator.find(query,Asset.class).stream().map(Asset::getAssetId).collect(Collectors.toSet());
    }

    public Asset selectLastOneNew(String assetId) {
        Criteria criteria = Criteria.where("assetId").is(assetId).and("sourceBranch").is(BranchConfig.DEV_BRANCH_NAME);
        return mongoPrimaryRepositoryDecorator.findOne(new Query(criteria),Asset.class);
    }

    public List<Asset> selectByApplicationAndBranch(String application, String branch) {
        Criteria criteria = Criteria.where("application").is(application).and("sourceBranch").is(branch);
        return mongoPrimaryRepositoryDecorator.find(new Query(criteria),Asset.class);
    }

    public void insertAll(List<Asset> assetList) {
        mongoPrimaryRepositoryDecorator.insertAll(assetList);
    }

    public void deleteByApplicationAndBranch(String application, String branch) {
        Criteria criteria = Criteria.where("application").is(application).and("sourceBranch").is(branch);
        mongoPrimaryRepositoryDecorator.remove(new Query(criteria),Asset.class);
    }

    public void updateDraftByApplicationAndType(String application, AssetDefinition.AssetTypeEnum assetTypeEnum) {
        Criteria criteria = Criteria.where("application").is(application).and("type").is(assetTypeEnum.getCode());
        Update update = new Update();
        update.set("adpStatus","draft");
        mongoPrimaryRepositoryDecorator.updateMulti(new Query(criteria),update,Asset.class);
    }
}
