package com.digiwin.athena.dao.mongodao.asset;

import com.digiwin.athena.dto.PageReqCondition;
import com.digiwin.athena.mongodb.domain.application.AssetDefinition;
import com.digiwin.athena.mongodb.repository.MongoPrimaryRepositoryDecorator;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Repository;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

@Slf4j
@Repository
public class AssetDefinitionMongoDao {
    private MongoPrimaryRepositoryDecorator mongoPrimaryRepositoryDecorator;

    @Autowired
    public void setMongoPrimaryRepositoryDecorator(MongoPrimaryRepositoryDecorator mongoPrimaryRepositoryDecorator) {
        this.mongoPrimaryRepositoryDecorator = mongoPrimaryRepositoryDecorator;
    }

    public AssetDefinition selectByApplicationAndCode(String code, String application) {
        Criteria criteria = Criteria.where("code").is(code).and("application").is(application);
        Query query = new Query(criteria);
        return mongoPrimaryRepositoryDecorator.findOne(query, AssetDefinition.class);
    }

    public void insert(AssetDefinition assetDefinition) {
        mongoPrimaryRepositoryDecorator.insert(assetDefinition);
    }

    public AssetDefinition selectByAssetId(String assetId) {
        Criteria criteria = Criteria.where("assetId").is(assetId);
        Query query = new Query(criteria);
        return mongoPrimaryRepositoryDecorator.findOne(query, AssetDefinition.class);
    }

    public void updateStatus(Integer status, String editBy, String assetId) {

        Update update = new Update();
        update.set("status", status);
        update.set("editDate", new Date());
        update.set("editBy", editBy);

        Criteria criteria = Criteria.where("assetId").is(assetId);
        mongoPrimaryRepositoryDecorator.updateMulti(new Query(criteria), update, AssetDefinition.class);
    }

    public void save(AssetDefinition assetDefinition) {
        mongoPrimaryRepositoryDecorator.save(assetDefinition);
    }

    public List<AssetDefinition> selectByAssetIds(Collection<String> assetIds) {
        Criteria criteria = Criteria.where("assetId").in(assetIds);
        Query query = new Query(criteria);
        return mongoPrimaryRepositoryDecorator.find(query, AssetDefinition.class);
    }

    public List<AssetDefinition> selectPagination(PageReqCondition<String> pageReqCondition,String tenantId,String type) {

        Criteria criteria = buildPageCondition(pageReqCondition,tenantId,type);

        Query query = new Query(criteria);
        query.limit(pageReqCondition.getPageSize());
        query.skip((pageReqCondition.getPageNum() - 1) * pageReqCondition.getPageSize());
        query.with(Sort.by(Sort.Direction.DESC,"_id"));

        return mongoPrimaryRepositoryDecorator.find(query, AssetDefinition.class);
    }

    public Long countPagination(PageReqCondition<String> pageReqCondition,String tenantId,String type){
        Criteria criteria = buildPageCondition(pageReqCondition,tenantId,type);

        Query query = new Query(criteria);

        return mongoPrimaryRepositoryDecorator.count(query, AssetDefinition.class);
    }

    public Long countApplicationPagination(PageReqCondition<String> pageReqCondition,String appCode,String type){
        Criteria criteria = buildAdvancedCondition(pageReqCondition);

        if (criteria == null){
            criteria = Criteria.where("application").is(appCode);
            if (type != null){
                criteria = criteria.and("type").is(type);
            }
        }else{
            List<Criteria> criteriaList = new ArrayList<>();
            criteriaList.add(criteria);
            criteriaList.add(Criteria.where("application").is(appCode));
            if (type!=null){
                criteriaList.add(Criteria.where("type").is(type));
            }

            criteria = new Criteria().andOperator(criteriaList);
        }
        Query query = new Query(criteria);

        return mongoPrimaryRepositoryDecorator.count(query, AssetDefinition.class);
    }

    public List<AssetDefinition> selectApplicationPagination(PageReqCondition<String> pageReqCondition,String appCode,String type) {

        Criteria criteria = buildAdvancedCondition(pageReqCondition);

        if (criteria == null){
            criteria = Criteria.where("application").is(appCode);
            if (type != null){
                criteria = criteria.and("type").is(type);
            }
        }else{
            Criteria criteria1 = Criteria.where("application").is(appCode);
            if (!StringUtils.isEmpty(type)){
                criteria1.and("type").is(type);
            }

            criteria = new Criteria().andOperator(criteria1,criteria);
        }

        Query query = new Query(criteria);
        query.limit(pageReqCondition.getPageSize());
        query.skip((pageReqCondition.getPageNum() - 1) * pageReqCondition.getPageSize());
        query.with(Sort.by(Sort.Direction.DESC,"_id"));

        return mongoPrimaryRepositoryDecorator.find(query, AssetDefinition.class);
    }

    private Criteria buildPageCondition(PageReqCondition<String> pageReqCondition,String tenantId,String type) {
        Criteria criteria = buildAdvancedCondition(pageReqCondition);

        Criteria rangeCriteria = new Criteria();
        Criteria criteria1 = Criteria.where("openRange").is(AssetDefinition.ALL_RANGE);
        Criteria criteria2 = Criteria.where("openRange").is(AssetDefinition.CUR_TENANT_RANGE).and("adpTenantId").is(tenantId);
        Criteria criteria3 = Criteria.where("openRange").is(AssetDefinition.TENANT_RANGE).and("tenantInfos.tenantId").in(tenantId);
        rangeCriteria.orOperator(criteria3,criteria1,criteria2);

        List<Criteria> criteriaList = new ArrayList<>();
        criteriaList.add(rangeCriteria);
        criteriaList.add(Criteria.where("status").is(1));
        if (type!=null){
            criteriaList.add(Criteria.where("type").is(type));
        }

        if (criteria == null){
            criteria = new Criteria();
            criteria.andOperator(criteriaList);
        }else{
            criteriaList.add(criteria);
            criteria = new Criteria().andOperator(criteriaList);
        }

        return criteria;
    }

    private Criteria buildAdvancedCondition(PageReqCondition<String> pageReqCondition) {
        Criteria criteria = null;
        if (pageReqCondition.getAdvancedQueryParams() != null) {
            Criteria adVancedCriteria = pageReqCondition.getAdvancedQueryParams().buildCriteria();
            if (adVancedCriteria != null) {
                criteria = adVancedCriteria;
            }
        }

        if (criteria == null && StringUtils.isNotEmpty(pageReqCondition.getCondition())) {
            Pattern pattern = Pattern.compile(".*" + Pattern.quote(pageReqCondition.getCondition()) + ".*", Pattern.CASE_INSENSITIVE);

            criteria = new Criteria();
            criteria.orOperator(Criteria.where("assetId").regex(pattern),
                    Criteria.where("assetName").regex(pattern),
                    Criteria.where("assetDesc").regex(pattern));
        }
        return criteria;
    }

    public List<String> selectAssetIdAvailableStatusByApplication(String appCode, String type) {

        Criteria criteria = Criteria.where("application").is(appCode).and("status").is(AssetDefinition.AVAILABLE_STATUS).and("type").is(type);
        Query query = new Query(criteria);
        query.fields().include("assetId");
        return mongoPrimaryRepositoryDecorator.find(query, AssetDefinition.class).stream().map(AssetDefinition::getAssetId).collect(Collectors.toList());
    }

    public Long countSameTenantDtdAsset(PageReqCondition<String> pageReqCondition, String tenantId, String type) {
        Criteria criteria = buildAdvancedCondition(pageReqCondition);
        if (criteria == null){
            criteria = Criteria.where("adpTenantId").is(tenantId).and("type").is(type);
        }else{
            criteria = new Criteria().andOperator(Criteria.where("adpTenantId").is(tenantId), Criteria.where("type").is(type), criteria);
        }

        Query query = new Query(criteria);
        query.limit(pageReqCondition.getPageSize());
        query.skip((pageReqCondition.getPageNum() - 1) * pageReqCondition.getPageSize());
        query.with(Sort.by(Sort.Direction.ASC,"_id"));

        return mongoPrimaryRepositoryDecorator.count(query,AssetDefinition.class);
    }

    public List<AssetDefinition> selectSameTenantDtdAsset(PageReqCondition<String> pageReqCondition, String tenantId, String type) {
        Criteria criteria = buildAdvancedCondition(pageReqCondition);
        if (criteria == null){
            criteria = Criteria.where("adpTenantId").is(tenantId).and("type").is(type);
        }else{
            criteria = new Criteria().andOperator(Criteria.where("adpTenantId").is(tenantId), Criteria.where("type").is(type), criteria);
        }

        Query query = new Query(criteria);
        query.limit(pageReqCondition.getPageSize());
        query.skip((pageReqCondition.getPageNum() - 1) * pageReqCondition.getPageSize());
        query.with(Sort.by(Sort.Direction.ASC,"_id"));

        return mongoPrimaryRepositoryDecorator.find(query,AssetDefinition.class);
    }

    public Long countPlatformDtdAsset(PageReqCondition<String> pageReqCondition, String tenantId, String type) {
        Criteria criteria = buildAdvancedCondition(pageReqCondition);
        if (criteria == null){
            criteria = new Criteria().orOperator(Criteria.where("adpTenantId").ne(tenantId)
                    .and("type").is(type).and("openRange").is("all"),
                    Criteria.where("adpTenantId").ne(tenantId)
                    .and("type").is(type).and("openRange").is("tenant").and("tenantInfos.tenantId").in(tenantId));
        }else{
            Criteria criteria2 = new Criteria().orOperator(Criteria.where("adpTenantId").ne(tenantId)
                            .and("type").is(type).and("openRange").is("all"),
                    Criteria.where("adpTenantId").ne(tenantId)
                            .and("type").is(type).and("openRange").is("tenant").and("tenantInfos.tenantId").in(tenantId));

            criteria = new Criteria().andOperator(criteria2, criteria);
        }

        Query query = new Query(criteria);
        query.limit(pageReqCondition.getPageSize());
        query.skip((pageReqCondition.getPageNum() - 1) * pageReqCondition.getPageSize());
        query.with(Sort.by(Sort.Direction.ASC,"_id"));

        return mongoPrimaryRepositoryDecorator.count(query,AssetDefinition.class);
    }

    public List<AssetDefinition> selectPlatformDtdAsset(PageReqCondition<String> pageReqCondition, String tenantId, String type) {
        Criteria criteria = buildAdvancedCondition(pageReqCondition);
        if (criteria == null){
            criteria = new Criteria().orOperator(Criteria.where("adpTenantId").ne(tenantId)
                            .and("type").is(type).and("openRange").is("all"),
                    Criteria.where("adpTenantId").ne(tenantId)
                            .and("type").is(type).and("openRange").is("tenant").and("tenantInfos.tenantId").in(tenantId));
        }else{
            Criteria criteria2 = new Criteria().orOperator(Criteria.where("adpTenantId").ne(tenantId)
                            .and("type").is(type).and("openRange").is("all"),
                    Criteria.where("adpTenantId").ne(tenantId)
                            .and("type").is(type).and("openRange").is("tenant").and("tenantInfos.tenantId").in(tenantId));

            criteria = new Criteria().andOperator(criteria2, criteria);
        }

        Query query = new Query(criteria);
        query.limit(pageReqCondition.getPageSize());
        query.skip((pageReqCondition.getPageNum() - 1) * pageReqCondition.getPageSize());
        query.with(Sort.by(Sort.Direction.ASC,"_id"));

        return mongoPrimaryRepositoryDecorator.find(query,AssetDefinition.class);
    }

    public List<AssetDefinition> selectBasicByApplicationAndCodes(String application, List<String> codes) {

        Criteria criteria = Criteria.where("code").in(codes).and("application").is(application);
        Query query = new Query(criteria);
        query.fields().include("assetId","code","type","assetName","application");
        return mongoPrimaryRepositoryDecorator.find(query, AssetDefinition.class);
    }

    public List<AssetDefinition> selectByAssetIdsAndStatus(List<String> assetIds, Integer status) {
        Criteria criteria = Criteria.where("assetId").in(assetIds).and("status").is(status);
        Query query = new Query(criteria);
        return mongoPrimaryRepositoryDecorator.find(query, AssetDefinition.class);
    }
}
