package com.digiwin.athena.dao.mongodao.asset;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.dto.PageReqCondition;
import com.digiwin.athena.dto.asset.AssetRelationInfoDto;
import com.digiwin.athena.mongodb.domain.application.AssetDefinition;
import com.digiwin.athena.mongodb.domain.application.AssetRelation;
import com.digiwin.athena.mongodb.repository.MongoPrimaryRepositoryDecorator;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.aggregation.Aggregation;
import org.springframework.data.mongodb.core.aggregation.AggregationResults;
import org.springframework.data.mongodb.core.aggregation.MatchOperation;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Repository;
import org.springframework.util.CollectionUtils;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * 资产关系交互层
 */
@Slf4j
@Repository
public class AssetRelationMongoDao {

    @Autowired
    private MongoPrimaryRepositoryDecorator mongoPrimaryRepositoryDecorator;

    public void save(AssetRelation assetRelation) {
        mongoPrimaryRepositoryDecorator.save(assetRelation);
    }

    public List<AssetRelation> find(String assetId, String application){
        Query query = new Query(Criteria.where("assetId").is(assetId).and("application").is(application));
        return mongoPrimaryRepositoryDecorator.find(query, AssetRelation.class);
    }

    public List<AssetRelation> findByApplication(String application){
        Query query = new Query(Criteria.where("application").is(application).and("bizType").is("DTD"));
        return mongoPrimaryRepositoryDecorator.find(query, AssetRelation.class);
    }

    public List<AssetRelation> findByRefApplication(String application){
        Query query = new Query(Criteria.where("application").is(application).and("type").is("ref")
                .and("bizType").is("DTD"));
        return mongoPrimaryRepositoryDecorator.find(query, AssetRelation.class);
    }

    @Deprecated
    public void delete(String assetId, String application, String type){
        Query query = new Query(Criteria.where("assetId").is(assetId).and("application").is(application).and("type").is(type));
        mongoPrimaryRepositoryDecorator.delete(query, AssetRelation.class);
    }

    public AssetRelation selectByAssetIdAndAdpVersionAndUserBranchAndApplication(AssetRelationInfoDto assetRelationInfoDto) {
        Criteria criteria = Criteria.where("assetId").is(assetRelationInfoDto.getAssetId())
                .and("application").is(assetRelationInfoDto.getApplication())
                .and("userBranch").is(assetRelationInfoDto.getUserBranch())
                .and("adpVersion").is(assetRelationInfoDto.getAdpVersion());

        return mongoPrimaryRepositoryDecorator.findOne(new Query(criteria),AssetRelation.class);
    }

    public void insert(AssetRelation assetRelation) {
        mongoPrimaryRepositoryDecorator.insert(assetRelation);
    }



    public void delete(AssetRelationInfoDto assetRelationInfoDto) {
        Criteria criteria = Criteria.where("application").is(assetRelationInfoDto.getApplication())
                .and("userBranch").is(assetRelationInfoDto.getUserBranch())
                .and("adpVersion").is(assetRelationInfoDto.getAdpVersion());

        mongoPrimaryRepositoryDecorator.delete(new Query(criteria),AssetRelation.class);
    }

    public List<AssetRelation> select(AssetRelationInfoDto assetRelationInfoDto, AssetDefinition.AssetTypeEnum assetTypeEnum) {
        Criteria criteria = Criteria.where("application").is(assetRelationInfoDto.getApplication())
                .and("bizType").is(assetTypeEnum.getCode()).and("type").is("ref");

        if (StringUtils.isNotEmpty(assetRelationInfoDto.getUserBranch())){
            criteria.and("userBranch").is(assetRelationInfoDto.getUserBranch());
        }
        if (StringUtils.isNotEmpty(assetRelationInfoDto.getAdpVersion())){
            criteria.and("adpVersion").is(assetRelationInfoDto.getAdpVersion());
        }
        if (StringUtils.isNotEmpty(assetRelationInfoDto.getAssetId())){
            criteria.and("assetId").is(assetRelationInfoDto.getAssetId());
        }

        return mongoPrimaryRepositoryDecorator.find(new Query(criteria),AssetRelation.class);
    }

    public Long selectRefCountByAssetIdAndApplication(String assetId, String application) {
        Criteria criteria = Criteria.where("assetId").is(assetId)
                .and("application").is(application);
        return mongoPrimaryRepositoryDecorator.count(new Query(criteria),AssetRelation.class);
    }

    public void deleteByAssetIdsAndVersionAndBranchAndApp(List<String> assetIds, String adpVersion, String branch, String application) {
        Criteria criteria = Criteria.where("application").is(application)
                .and("userBranch").is(branch)
                .and("adpVersion").is(adpVersion)
                .and("assetId").in(assetIds);

        mongoPrimaryRepositoryDecorator.delete(new Query(criteria),AssetRelation.class);
    }

    public Collection<String> selectRefAppAppByAssetId(String assetId) {
        Criteria criteria = Criteria.where("assetId").is(assetId).and("type").is("ref");
        Query query = new Query(criteria);
        query.fields().include("application");

        return mongoPrimaryRepositoryDecorator.find(query,AssetRelation.class)
                .stream().map(AssetRelation::getApplication).collect(Collectors.toSet());
    }

    public List<AssetRelation> selectByAssetId(String assetId) {
        Criteria criteria = Criteria.where("assetId").is(assetId);
        return mongoPrimaryRepositoryDecorator.find(new Query(criteria),AssetRelation.class);
    }

    public List<String> selectAssetIdByAppCode(String appCode, String type) {
        Criteria criteria = Criteria.where("application").is(appCode).and("bizType").is(type);
        return mongoPrimaryRepositoryDecorator.find(new Query(criteria),AssetRelation.class).stream().map(AssetRelation::getAssetId).collect(Collectors.toList());
    }

    public Long selectCountPagination(PageReqCondition<String> pageReqCondition, String appCode, String type) {
        Criteria criteria = buildAdvancedCondition(pageReqCondition);
        if (criteria == null){
            criteria = Criteria.where("application").is(appCode).and("type").is("ref");
            if (!StringUtils.isEmpty(type)){
                criteria=criteria.and("bizType").is(type);
            }
        }else{
            Criteria criteria1 = Criteria.where("application").is(appCode).and("type").is("ref");
            if (!StringUtils.isEmpty(type)){
                criteria1=criteria1.and("bizType").is(type);
            }
            criteria = new Criteria().andOperator(criteria1,criteria);
        }
//        Query query = new Query(criteria);

        MatchOperation match = Aggregation.match(criteria);
        Aggregation aggregation = Aggregation.newAggregation(match,
                Aggregation.group("assetId"),
                Aggregation.count().as("total"));
        AggregationResults<JSONObject> res = mongoPrimaryRepositoryDecorator.aggregate(aggregation, AssetRelation.class, JSONObject.class);
        if (CollectionUtils.isEmpty(res.getMappedResults())){
            return 0L;
        }
        return res.getMappedResults().get(0).getInteger("total").longValue();
    }

    public List<AssetRelation> selectPagination(PageReqCondition<String> pageReqCondition, String appCode, String type) {
        Criteria criteria = buildAdvancedCondition(pageReqCondition);
        if (criteria == null){
            criteria = Criteria.where("application").is(appCode).and("type").is("ref");
            if (!StringUtils.isEmpty(type)){
                criteria=criteria.and("bizType").is(type);
            }
        }else{
            Criteria criteria1 = Criteria.where("application").is(appCode).and("type").is("ref");
            if (!StringUtils.isEmpty(type)){
                criteria1=criteria1.and("bizType").is(type);
            }
            criteria = new Criteria().andOperator(criteria1,criteria);
        }
        Query query = new Query(criteria);
        query.limit(pageReqCondition.getPageSize());
        query.skip((pageReqCondition.getPageNum() - 1) * pageReqCondition.getPageSize());
        query.with(Sort.by(Sort.Direction.DESC,"_id"));

        MatchOperation match = Aggregation.match(criteria);
        Aggregation aggregation = Aggregation.newAggregation(match,
                Aggregation.group("assetId").first(Aggregation.ROOT).as("doc"),
                Aggregation.sort(Sort.by(Sort.Direction.DESC, "_id")),
                Aggregation.skip((pageReqCondition.getPageNum() - 1) * pageReqCondition.getPageSize()),
                Aggregation.limit(pageReqCondition.getPageSize()));
        AggregationResults<JSONObject> res = mongoPrimaryRepositoryDecorator.aggregate(aggregation,"assetRelation", JSONObject.class);
        List<AssetRelation> result = res.getMappedResults().stream().map(j -> {
            Map doc = (Map) j.get("doc");
            doc.put("objectId", doc.get("_id").toString());
            return JSONObject.parseObject(JSONObject.toJSONString(doc), AssetRelation.class);
        }).collect(Collectors.toList());
        return result;
    }

    private static Criteria buildAdvancedCondition(PageReqCondition<String> pageReqCondition) {
        Criteria criteria = null;
        if (pageReqCondition.getAdvancedQueryParams() != null) {
            Criteria adVancedCriteria = pageReqCondition.getAdvancedQueryParams().buildCriteria();
            if (adVancedCriteria != null) {
                criteria = adVancedCriteria;
            }
        }

        if (criteria == null && StringUtils.isNotEmpty(pageReqCondition.getCondition())) {
            Pattern pattern = Pattern.compile(".*" + Pattern.quote(pageReqCondition.getCondition()) + ".*", Pattern.CASE_INSENSITIVE);

            criteria = new Criteria();
            criteria.orOperator(Criteria.where("assetId").regex(pattern));
        }
        return criteria;
    }

    public AssetRelation selectByObjectId(String objectId) {
        Criteria criteria = Criteria.where("_id").is(objectId);
        return mongoPrimaryRepositoryDecorator.findOne(new Query(criteria),AssetRelation.class);
    }

    public void deleteByObjectId(String objectId) {
        Criteria criteria = Criteria.where("_id").is(objectId);
        mongoPrimaryRepositoryDecorator.delete(new Query(criteria),AssetRelation.class);
    }

    public List<AssetRelation> selectByApplicationAndBranch(String application, String userBranch) {
        Criteria criteria = Criteria.where("application").is(application)
                .and("userBranch").is(userBranch);

        return mongoPrimaryRepositoryDecorator.find(new Query(criteria),AssetRelation.class);
    }

    public void insertAll(List<AssetRelation> relations) {
        mongoPrimaryRepositoryDecorator.insertAll(relations);
    }

    public void deleteByApplicationAndBranch(String application, String userBranch) {
        Criteria criteria = Criteria.where("application").is(application)
                .and("userBranch").is(userBranch);

        mongoPrimaryRepositoryDecorator.delete(new Query(criteria),AssetRelation.class);
    }

    public AssetRelation findOneAssetRelation(String application) {
        Query query = new Query(Criteria.where("application").is(application).and("bizType").is("DTD"));
        return mongoPrimaryRepositoryDecorator.findOne(query, AssetRelation.class);

    }

    public List<AssetRelation> selectByApplicationAndBranchAndVersion(String application, String adpVersion, String branch) {
        Criteria criteria = Criteria.where("application").is(application)
                .and("userBranch").is(branch)
                .and("adpVersion").is(adpVersion)
                .and("type").is("ref");

        return mongoPrimaryRepositoryDecorator.find(new Query(criteria),AssetRelation.class);
    }

    public List<AssetRelation> selectByAssetIdAndType(String assetId, String type) {
        Criteria criteria = Criteria.where("assetId").is(assetId).and("type").is(type);
        return mongoPrimaryRepositoryDecorator.find(new Query(criteria),AssetRelation.class);
    }
}
