package com.digiwin.athena.utils;

import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.graalvm.polyglot.Context;
import org.graalvm.polyglot.Value;

import java.util.List;
import java.util.Map;

@Slf4j
public class JSUtil {

    public static <T> T evalResponse(String jsStr, String param, Class<T> returnType) {
        try(Context context = Context.newBuilder("js")
                .build()) {

            Value bindings = context.getBindings("js");
            bindings.putMember("responseJSON", param);
            bindings.putMember("log",log);

            String js =
                    "(function() {" +
                            "const response = JSON.parse(responseJSON);" +
                            jsStr +
                            "})();";

            Value result = context.eval("js", js);
            return convert(result, returnType);
        }
    }

    public static <T> T evalRequest(String jsStr, String param, Class<T> returnType) {
        try(Context context = Context.newBuilder("js")
                .build()) {

            Value bindings = context.getBindings("js");
            bindings.putMember("requestJSON", param);
            bindings.putMember("log",log);
            String js =
                    "(function() {" +
                            "const request = JSON.parse(requestJSON);" +
                            jsStr +
                            "})();";

            Value result = context.eval("js", js);
            return convert(result, returnType);
        }
    }

    private static <T> T convert(Value result, Class<T> returnType) {
        ObjectMapper mapper = new ObjectMapper();

        // 1. null 处理
        if (result.isNull()) {
            return null;
        }
        // 2. String
        if (result.isString()) {
            String str = result.asString();

            // 如果目标类型是 String
            if (returnType == String.class) {
                return returnType.cast(str);
            }

            // 如果返回的是 JSON 字符串，转换为目标类
            try {
                return mapper.readValue(str, returnType);
            } catch (Exception ignore) {
                // 如果不是 JSON 就会失败，那就报错
            }

            throw new IllegalArgumentException(
                    "JS 返回字符串，但无法转换成目标类型：" + returnType.getName());
        }

        // 3. Number
        if (result.fitsInInt() && returnType == Integer.class) {
            return returnType.cast(result.asInt());
        }
        if (result.fitsInLong() && returnType == Long.class) {
            return returnType.cast(result.asLong());
        }
        if (result.fitsInDouble() && (returnType == Double.class || returnType == double.class)) {
            return returnType.cast(result.asDouble());
        }

        // 4. Boolean
        if (result.isBoolean() && (returnType == Boolean.class || returnType == boolean.class)) {
            return returnType.cast(result.asBoolean());
        }

        // 5. JS Array → List
        if (result.hasArrayElements() && returnType == List.class) {
            return result.as(returnType);
        }

        // 6. JS Object → Map → DTO
        if (result.hasMembers()) {
            Map<String, Object> map = result.as(Map.class);
            return mapper.convertValue(map, returnType);
        }

        // 7. 其他类型（比如直接 as）
        return result.as(returnType);
    }

}
