package com.digiwin.athena.dto;

import io.swagger.v3.oas.annotations.media.Schema;
import org.springframework.data.mongodb.core.query.Criteria;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

public class QueryGroup {
    @Schema(description = "查询条件组之间的关系|AND,OR")
    private String relation;

    private List<QueryCondition> conditions = new ArrayList<>();

    public Criteria buildCriteria() {
        Criteria criteria = new Criteria();

        List<Criteria> andCriList = new ArrayList<>();
        List<Criteria> orCriList = new ArrayList<>();
        for (QueryCondition condition : this.conditions) {
            Criteria sub = condition.buildCriteria();
            if ("AND".equals(condition.getRelation())){
                andCriList.add(sub);
            }else{
                orCriList.add(sub);
            }
        }

        if (!andCriList.isEmpty()){
            criteria.andOperator(andCriList);
        }

        if (!orCriList.isEmpty()){
            List<Criteria> or = new ArrayList<>();
            or.addAll(orCriList);
            or.add(criteria);
            criteria = new Criteria().orOperator(or);
        }

        return criteria;
    }

    public enum QueryOperator {
        Contains("contains"), // 包含
        NotContains("not_contains"), // 不包含
        ArrayContains("array_contains"), // 包含
        ArrayNotContains("array_not_contains"), // 包含
        Equals("equals"), // 等于
        NotEquals("not_equals"), // 不等于
        GreaterThan("greater_than"), // 大于
        LessThan("less_than"), // 小于
        GreaterThanEquals("greater_than_equals"), // 大于等于
        LessThanEquals("less_than_equals"); // 小于等于

        private String operator;

        QueryOperator(String operator) {
            this.operator = operator;
        }

        public String getOperator() {
            return operator;
        }
    }

    public static class QueryCondition {
        @Schema(description = "查询字段")
        private String field;
        @Schema(description = "字段条件操作")
        private String operator;
        private Object value;
        @Schema(description = "查询条件之间的关系|AND,OR")

        private String relation;

        public String getField() {
            return field;
        }

        public void setField(String field) {
            this.field = field;
        }

        public String getOperator() {
            return operator;
        }

        public void setOperator(String operator) {
            this.operator = operator;
        }

        public Object getValue() {
            return value;
        }

        public void setValue(Object value) {
            this.value = value;
        }

        public String getRelation() {
            return relation;
        }

        public void setRelation(String relation) {
            this.relation = relation;
        }

        public Criteria buildCriteria() {
            Criteria where = Criteria.where(this.field);
            if (this.operator.equals(QueryOperator.Equals.getOperator())){
                where.is(this.value);
            }else if (this.operator.equals(QueryOperator.Contains.getOperator())){
                Pattern pattern = Pattern.compile(".*" + Pattern.quote(this.getValue().toString()) + ".*", Pattern.CASE_INSENSITIVE);

                where.regex(pattern);
            }else if (this.operator.equals(QueryOperator.NotContains.getOperator())){
                Pattern pattern = Pattern.compile(".*" + Pattern.quote(this.getValue().toString()) + ".*", Pattern.CASE_INSENSITIVE);
                where.not().regex(pattern);
            }else if (this.operator.equals(QueryOperator.ArrayContains.getOperator())){
                where.is(this.value);
            }else if (this.operator.equals(QueryOperator.ArrayNotContains.getOperator())){
                where.ne(this.value);
            }else if (this.operator.equals(QueryOperator.NotEquals.getOperator())){
                where.ne(this.value);
            }else if (this.operator.equals(QueryOperator.GreaterThan.getOperator())){
                where.gt(this.value);
            }else if (this.operator.equals(QueryOperator.LessThan.getOperator())){
                where.lt(this.value);
            }else if (this.operator.equals(QueryOperator.GreaterThanEquals.getOperator())){
                where.gte(this.value);
            }else if (this.operator.equals(QueryOperator.LessThanEquals.getOperator())){
                where.lte(this.value);
            }
            return where;
        }
    }

    public String getRelation() {
        return relation;
    }

    public void setRelation(String relation) {
        this.relation = relation;
    }

    public List<QueryCondition> getConditions() {
        return conditions;
    }

    public void setConditions(List<QueryCondition> conditions) {
        this.conditions = conditions;
    }
}
