package com.digiwin.athena.utils;

import cn.hutool.core.bean.BeanUtil;
import com.alibaba.fastjson.JSONObject;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.graalvm.polyglot.Context;
import org.graalvm.polyglot.HostAccess;
import org.graalvm.polyglot.Value;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

@Slf4j
public class JSUtil {

    public static <T> T eval(String jsStr, Object param, Class<T> returnType) {
        try (Context context = Context.newBuilder("js")
                .build()) {

            Value bindings = context.getBindings("js");

            Map<String, Object> map = BeanUtil.beanToMap(param);
            map.forEach(bindings::putMember);


            Value result = context.eval("js", jsStr);
            //noinspection unchecked
            return (T) convert(result, returnType);
        }
    }

    public static <T> T evalResponse(String jsStr, String param, Class<T> returnType) {
        try(Context context = Context.newBuilder("js")
                .build()) {

            Value bindings = context.getBindings("js");
            bindings.putMember("responseJSON", param);
            bindings.putMember("log",log);

            String js =
                    "(function() {" +
                            "const response = JSON.parse(responseJSON);" +
                            jsStr +
                            "})();";

            Value result = context.eval("js", js);
            //noinspection unchecked
            return (T) convert(result, returnType);
        }catch (Exception e){
            throw new RuntimeException("js执行异常",e);
        }
    }

    public static void evalRequest(String jsStr, JSONObject requestContext) {
        try (Context context = Context.newBuilder("js")
                .allowHostAccess(HostAccess.ALL)
                .build()) {

            Value bindings = context.getBindings("js");
            bindings.putMember("requestContext", requestContext);
            bindings.putMember("log", log);
            String js =
                    "(function() {" +
                            jsStr +
                            "})();";

            Value result = context.eval("js", js);
        } catch (Exception e) {
            throw new RuntimeException("js执行异常", e);
        }
    }

    public static void main(String[] args) {
        JSONObject param = new JSONObject();

        try(Context context = Context.newBuilder("js").allowHostAccess(HostAccess.ALL)
                .build()) {

            Value bindings = context.getBindings("js");
            bindings.putMember("param", param);
            bindings.putMember("log",log);
            String js =
                    "(function() {" +
                            "param.a = 123" +
                            "})();";

            Value result = context.eval("js", js);
            System.out.println(param);
        }catch (Exception e){
            throw new RuntimeException("js执行异常",e);
        }
    }

    public static <T> Object evalDependencyRule(String jsStr, String param, Class<T> returnType) {
        try(Context context = Context.newBuilder("js")
                .allowHostAccess(HostAccess.ALL)
                .allowHostClassLookup(s -> true)
                .build()) {

            Value bindings = context.getBindings("js");
            bindings.putMember("assetJSON", param);
            bindings.putMember("log",log);
            String js =
                    "(function() {" +
                            "const asset = JSON.parse(assetJSON);" +
                            jsStr +
                            "})();";

            Value result = context.eval("js", js);
            return convert(result, returnType);
        }catch (Exception e){
            throw new RuntimeException("js执行异常",e);
        }
    }

    private static <T> Object convert(Value result, Class<T> returnType) {
        ObjectMapper mapper = new ObjectMapper();

        // 1. null 处理
        if (result.isNull()) {
            return null;
        }
        // 2. String
        if (result.isString()) {
            String str = result.asString();

            // 如果目标类型是 String
            if (returnType == String.class) {
                return returnType.cast(str);
            }

            // 如果返回的是 JSON 字符串，转换为目标类
            try {
                return mapper.readValue(str, returnType);
            } catch (Exception exception) {
                log.error("JS 返回字符串，但无法转换成目标类型",exception);
                // 如果不是 JSON 就会失败，那就报错
                throw new RuntimeException("JS 返回字符串，但无法转换成目标类型：" + returnType.getName());
            }

        }

        // 3. Number
        if (result.fitsInInt() && returnType == Integer.class) {
            return returnType.cast(result.asInt());
        }
        if (result.fitsInLong() && returnType == Long.class) {
            return returnType.cast(result.asLong());
        }
        if (result.fitsInDouble() && (returnType == Double.class || returnType == double.class)) {
            return returnType.cast(result.asDouble());
        }

        // 4. Boolean
        if (result.isBoolean() && (returnType == Boolean.class || returnType == boolean.class)) {
            return returnType.cast(result.asBoolean());
        }

        // 5. JS Array → List
        if (result.hasArrayElements()) {
            Value iterator = result.getIterator();
            List<T> res = new ArrayList<>();
            while(iterator.hasIteratorNextElement()){
                Value iteratorNextElement = iterator.getIteratorNextElement();
                res.add((T)convert(iteratorNextElement, returnType));
            }
            return res;
        }

        // 6. JS Object → Map → DTO
        if (result.hasMembers()) {
            Map<String, Object> map = result.as(Map.class);
            return mapper.convertValue(map, returnType);
        }

        // 7. 其他类型（比如直接 as）
        return result.as(returnType);
    }

}
