package com.digiwin.athena.rabbitMQ;

import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.amqp.core.AcknowledgeMode;
import org.springframework.amqp.core.Binding;
import org.springframework.amqp.core.BindingBuilder;
import org.springframework.amqp.core.DirectExchange;
import org.springframework.amqp.core.Queue;
import org.springframework.amqp.rabbit.config.SimpleRabbitListenerContainerFactory;
import org.springframework.amqp.rabbit.connection.CachingConnectionFactory;
import org.springframework.amqp.rabbit.connection.ConnectionFactory;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.amqp.RabbitProperties;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import java.util.List;


@Slf4j
@Configuration
@ConfigurationProperties(prefix = "rabbitmq.mdc")
@ConditionalOnProperty(name = "rabbitmq.mdc.open", havingValue = "true")
@Data
public class RabbitMQConfig {

    public static final String EXCHANGE_NAME = "athena_designer_exchange";
    public static final String ASSET_DATA_ROUTE_KEY = "assetData.key";

    private String address;
    private String assetDataSyncQueueName;

    @Bean
    public DirectExchange exchange() {
        return new DirectExchange(EXCHANGE_NAME);
    }

    @Bean
    @ConditionalOnProperty(prefix = "rabbitmq.mdc", name = "assetDataSyncQueueName")
    public Queue assetDataSyncQueue() {
        return new Queue(assetDataSyncQueueName, true);
    }
    @Bean
    @ConditionalOnBean(name = "assetDataSyncQueue")
    public Binding assetDataSyncQueueBinding(@Qualifier("assetDataSyncQueue") Queue queue, DirectExchange exchange) {
        return BindingBuilder.bind(queue).to(exchange).with(ASSET_DATA_ROUTE_KEY);
    }

    @Bean
    public ConnectionFactory connectionFactory(){
        CachingConnectionFactory connectionFactory = new CachingConnectionFactory();

        RabbitProperties properties = new RabbitProperties();
        properties.setAddresses(List.of(address));

        connectionFactory.setHost(properties.determineHost());
        connectionFactory.setPort(properties.determinePort());
        connectionFactory.setUsername(properties.determineUsername());
        connectionFactory.setPassword(properties.determinePassword());
        connectionFactory.setVirtualHost(properties.determineVirtualHost());
        return connectionFactory;
    }

    @Bean
    public RabbitTemplate rabbitTemplate(@Qualifier("connectionFactory") ConnectionFactory connectionFactory){
        RabbitTemplate rabbitTemplate = new RabbitTemplate(connectionFactory);
        return rabbitTemplate;
    }

    @Bean(name = "mdcMessageListenerContainer")
    public SimpleRabbitListenerContainerFactory mdcMessageListenerContainer(
            @Qualifier("connectionFactory") ConnectionFactory connectionFactory) {
        SimpleRabbitListenerContainerFactory factory = new SimpleRabbitListenerContainerFactory();
        factory.setConnectionFactory(connectionFactory);
        factory.setAcknowledgeMode(AcknowledgeMode.MANUAL);


        return factory;
    }

    @Bean("manualAckContainerFactory")
    public SimpleRabbitListenerContainerFactory manualAckContainerFactory(ConnectionFactory connectionFactory) {

        SimpleRabbitListenerContainerFactory factory = new SimpleRabbitListenerContainerFactory();
        factory.setConnectionFactory(connectionFactory);
        factory.setAcknowledgeMode(AcknowledgeMode.MANUAL);
        factory.setMissingQueuesFatal(false);
        return factory;
    }
}
