package com.digiwin.athena.config.http;

import cn.hutool.http.HttpInterceptor;
import cn.hutool.http.HttpResponse;
import cn.hutool.json.JSONObject;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONException;
import com.digiwin.athena.base.BusinessException;
import com.digiwin.athena.constant.IamConstant;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.Field;

@Slf4j
public class HutoolLogResponseInterceptor implements HttpInterceptor<HttpResponse> {
    @Override
    public void process(HttpResponse httpResponse) {
        // 记录响应状态和信息
        int status = httpResponse.getStatus();
        String responseBody = httpResponse.body();
        log.info(String.format("hutoolLog status:%d,响应info:%s", httpResponse.getStatus(), httpResponse.toString()));

        // 检查响应状态
        if (!httpResponse.isOk() && status != 401) {
            // 提取 httpConnection 字段
            Object httpConnection = getHttpConnection(httpResponse);

            if (httpConnection != null) {

                JSONObject connectionJson = new JSONObject(httpConnection);
                String url = connectionJson.getStr("url");

                if (url != null && url.endsWith(IamConstant.loginUrl)) {
                    // 处理登录相关的异常,为了处理安全渗透测试
                    String message = extractMessageFromBody(responseBody);
                    if (message != null) {
                        log.error("Login URL response error: {}", message);
                        throw new BusinessException(String.format("hutoolLog 响应异常, status: %d, 错误信息: %s", status, message));
                    }
                } else {
                    // 处理其他响应错误
                    log.error("Response body: {}", responseBody);
                    throw new BusinessException(String.format("hutoolLog 响应异常, status: %d, 响应内容: %s", status, responseBody));
                }
            } else {
                log.error("httpConnection 为空，无法处理响应.");
                throw new BusinessException("httpConnection 为空，无法处理响应.");
            }
        }
    }


    private Object getHttpConnection(HttpResponse httpResponse) {
        try {
            Field httpConnectionField = HttpResponse.class.getDeclaredField("httpConnection");
            httpConnectionField.setAccessible(true);
            return httpConnectionField.get(httpResponse);
        } catch (NoSuchFieldException | IllegalAccessException e) {
            log.error("无法访问 httpConnection 字段: {}", e.getMessage());
            return null;
        }
    }


    private String extractMessageFromBody(String body) {
        try {
            com.alibaba.fastjson.JSONObject jsonObject = JSON.parseObject(body);
            return jsonObject.getString("message");
        } catch (JSONException e) {
            log.error("解析响应体时出错: {}", e.getMessage());
            return null;
        }
    }
}
