package com.digiwin.athena.config;

import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.annotation.EnableAsync;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;

import java.util.concurrent.Executor;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

@EnableAsync(proxyTargetClass = true)
@Configuration
public class ThreadPoolConfig {

    static class CustomDefaultThreadFactory implements ThreadFactory {
        private static final AtomicInteger poolNumber = new AtomicInteger(1);
        private final ThreadGroup group;
        private final AtomicInteger threadNumber = new AtomicInteger(1);
        private final String namePrefix;

        CustomDefaultThreadFactory(String threadPrefix) {
            group = Thread.currentThread().getThreadGroup();
            namePrefix = threadPrefix+"-pool-" +
                    poolNumber.getAndIncrement() +
                    "-thread-";
        }

        public Thread newThread(Runnable r) {
            Thread t = new Thread(group, r,
                    namePrefix + threadNumber.getAndIncrement(),
                    0);
            if (t.isDaemon())
                t.setDaemon(false);
            if (t.getPriority() != Thread.NORM_PRIORITY)
                t.setPriority(Thread.NORM_PRIORITY);
            return t;
        }
    }

    @Bean("gptExecutor")
    public Executor taskExecutor() {
        ThreadPoolTaskExecutor executor = new ThreadPoolTaskExecutor();
        //核心线程池大小
        executor.setCorePoolSize(20);
        //最大线程数
        executor.setMaxPoolSize(30);
        //配置队列容量，默认值为Integer.MAX_VALUE
        executor.setQueueCapacity(1000);
        //活跃时间
        executor.setKeepAliveSeconds(60);
        //线程名字前缀
        executor.setThreadNamePrefix("asyncGPTServiceExecutor -");
        // 增加 TaskDecorator 属性的配置
        executor.setTaskDecorator(new CustomTaskDecorator());
        //设置此执行程序应该在关闭时阻止的最大秒数，以便在容器的其余部分继续关闭之前等待剩余的任务完成他们的执行
        executor.setAwaitTerminationSeconds(60);
        //等待所有的任务结束后再关闭线程池
        executor.setWaitForTasksToCompleteOnShutdown(true);
        executor.initialize();
        return executor;
    }

    @Bean("asyncExecutor")
    public Executor getAsyncExecutor() {
        // 定义线程池
        ThreadPoolTaskExecutor taskExecutor = new ThreadPoolTaskExecutor();
        // 设置核心线程数
        taskExecutor.setCorePoolSize(5);
        // 设置最大线程数
        taskExecutor.setMaxPoolSize(10);
        // 设置队列的大小
        taskExecutor.setQueueCapacity(100);
        // 初始化线程池
        taskExecutor.initialize();
        return taskExecutor;

    }

    @Bean("asaModelExecutor")
    public Executor asaModelExecutor() {
        // 定义线程池
        ThreadPoolTaskExecutor executor = new ThreadPoolTaskExecutor();
        // 设置核心线程数
        executor.setCorePoolSize(5);
        // 设置最大线程数
        executor.setMaxPoolSize(100);
        // 设置队列的大小
        executor.setQueueCapacity(100);
        // 初始化线程池
        executor.initialize();

        return executor;
    }

    @Bean
    public ThreadPoolExecutor securityTaskProcessPool(){

        return new ThreadPoolExecutor(5,
                10,
                5000,
                TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<>(1000),
                new CustomDefaultThreadFactory("securityTaskProcess"),
                new ThreadPoolExecutor.DiscardPolicy());
    }

    @Bean
    public ThreadPoolExecutor businessTaskProcessPool(){

        return new ThreadPoolExecutor(5,
                10,
                5000,
                TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<>(10000),
                new CustomDefaultThreadFactory("businessTaskProcess"),
                new ThreadPoolExecutor.DiscardPolicy());
    }

    @Bean
    public ThreadPoolExecutor dataCenterTaskProcessPool(){

        return new ThreadPoolExecutor(5,
                10,
                5000,
                TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<>(10000),
                new CustomDefaultThreadFactory("dataCenterProcess"),
                new ThreadPoolExecutor.DiscardPolicy());
    }

    @Bean
    public ThreadPoolExecutor createNewBranchPool(){

        return new ThreadPoolExecutor(5,
                10,
                5000,
                TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<>(1000),
                new CustomDefaultThreadFactory("createNewBranchPool"),
                new ThreadPoolExecutor.DiscardPolicy());
    }

    @Bean
    public ThreadPoolExecutor agileDataBusinessPool(){

        return new ThreadPoolExecutor(5,
                10,
                5000,
                TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<>(2000),
                new CustomDefaultThreadFactory("agileDataBusinessPool"),
                new ThreadPoolExecutor.DiscardPolicy());
    }

    @Bean
    public ThreadPoolExecutor assetCenterPool(){

        return new ThreadPoolExecutor(5,
                10,
                5000,
                TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<>(1000),
                new CustomDefaultThreadFactory("assetCenterPool"),
                new ThreadPoolExecutor.DiscardPolicy());
    }
}
