package com.digiwin.athena.config.http;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.base.BusinessException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpRequest;
import org.springframework.http.HttpStatus;
import org.springframework.http.HttpStatusCode;
import org.springframework.http.client.ClientHttpRequestExecution;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.http.client.ClientHttpResponse;
import org.springframework.util.StreamUtils;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;

@Slf4j
public class RestTemplateLogInterceptor implements ClientHttpRequestInterceptor {
    @Override
    public ClientHttpResponse intercept(HttpRequest request, byte[] body, ClientHttpRequestExecution execution) throws IOException {
        log.info(String.format("restTemplateLog 请求路径:%s,请求头:%s,请求参数:%s", request.getURI(), JSONObject.toJSONString(request.getHeaders()), new String(body)));

        ClientHttpResponse response = execution.execute(request, body);
        ClientHttpResponse responseWrapper = new BufferingClientHttpResponseWrapper(response);
        String responseBody = StreamUtils.copyToString(responseWrapper.getBody(), StandardCharsets.UTF_8);

        log.info(String.format("restTemplateLog 请求路径:%s , 响应responseBody:%s", request.getURI(), responseBody));

        if (response.getStatusCode() != HttpStatus.OK) {
            log.error(String.format("restTemplateLog uri:%s 请求异常:%s", request.getURI(), response.getStatusCode().value()));
            throw new BusinessException(request.getURI() + " 请求异常:" + responseBody);
        }
        return responseWrapper;
    }

    private static class BufferingClientHttpResponseWrapper implements ClientHttpResponse {
        private final ClientHttpResponse response;
        private byte[] body;

        public BufferingClientHttpResponseWrapper(ClientHttpResponse response) throws IOException {
            this.response = response;
            this.body = StreamUtils.copyToByteArray(response.getBody());
        }
        @Override
        public InputStream getBody() throws IOException {
            return new ByteArrayInputStream(body);
        }

        @Override
        public HttpStatusCode getStatusCode() throws IOException {
            return response.getStatusCode();
        }

        @Override
        public String getStatusText() throws IOException {
            return response.getStatusText();
        }

        @Override
        public void close() {
            response.close();
        }

        @Override
        public org.springframework.http.HttpHeaders getHeaders() {
            return response.getHeaders();
        }
    }
}