package com.digiwin.athena.config.mongodb;

import com.digiwin.athena.config.BranchConfig;
import com.digiwin.athena.config.ExtendSimpleMongoRepository;
import com.digiwin.athena.utils.CurThreadInfoUtils;
import com.mongodb.ConnectionString;
import com.mongodb.MongoClientSettings;
import com.mongodb.client.MongoClient;
import com.mongodb.client.MongoClients;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.data.mongodb.MongoDatabaseFactory;
import org.springframework.data.mongodb.config.MongoConfigurationSupport;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.SimpleMongoClientDatabaseFactory;
import org.springframework.data.mongodb.core.convert.DbRefResolver;
import org.springframework.data.mongodb.core.convert.DefaultDbRefResolver;
import org.springframework.data.mongodb.core.convert.DefaultMongoTypeMapper;
import org.springframework.data.mongodb.core.convert.MappingMongoConverter;
import org.springframework.data.mongodb.core.mapping.MongoMappingContext;
import org.springframework.data.mongodb.repository.config.EnableMongoRepositories;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/***
 * @Author wangxiao
 * @Date 2021/7/1 9:20
 * @Description mongod配置
 **/
@Data
@Configuration
@EnableMongoRepositories(basePackages = MongodbConfig.BASE_PACKAGE, repositoryBaseClass = ExtendSimpleMongoRepository.class)
@ConfigurationProperties(prefix = "spring.data.mongodb")
@Slf4j
public class MongodbConfig extends MongoConfigurationSupport {
    public static final String BASE_PACKAGE = "com.digiwin.athena.mongodb.repository";
    public static final String MONGO_TEMPLATE_REF = "mongoTemplate";

    private String uri;

    private String database;

    private String testDatabase;
    private String masterDatabase;
    private String bakDatabase;
    private String tenantDatabase;
    private String runtimeDatabase;
    private String individualDevDatabase;
    private String individualTestDatabase;
    private String individualMasterDatabase;
    private String individualBakDatabase;

    @Bean
    @Primary
    public MongoClient mongoClient() {
        MongoClientSettings mongoClientSettings = MongoClientSettings.builder()
                .applyConnectionString(new ConnectionString(uri))
                .applyToConnectionPoolSettings(builder -> builder.maxSize(30)
                        .minSize(10)
                        .maxConnectionIdleTime(1, TimeUnit.MINUTES)
                        .maxConnectionLifeTime(1, TimeUnit.MINUTES)
                        .maxWaitTime(2, TimeUnit.MINUTES))
                .build();
        return MongoClients.create(mongoClientSettings);
    }

    @Primary
    @Bean
    public MongoDatabaseFactory mongoDatabaseFactory() {
        log.error("database:"+database);
        log.error("masterDatabase:"+masterDatabase);
        log.error("bakDatabase:"+bakDatabase);
        log.error("tenantDatabase:"+tenantDatabase);
        return new SimpleMongoClientDatabaseFactory(mongoClient(), database);
    }

    @Bean
    public MongoDatabaseFactory testMongoDatabaseFactory() {
        return new SimpleMongoClientDatabaseFactory(mongoClient(), testDatabase);
    }

    @Bean
    public MongoDatabaseFactory masterMongoDatabaseFactory() {
        return new SimpleMongoClientDatabaseFactory(mongoClient(),masterDatabase);
    }

    @Bean
    public MongoDatabaseFactory bakMongoDatabaseFactory() {
        return new SimpleMongoClientDatabaseFactory(mongoClient(), bakDatabase);
    }
    @Bean
    public MongoDatabaseFactory tenantMongoDatabaseFactory() {
        return new SimpleMongoClientDatabaseFactory(mongoClient(), tenantDatabase);
    }

    @Bean
    public MongoDatabaseFactory runtimeMongoDatabaseFactory() {
        return new SimpleMongoClientDatabaseFactory(mongoClient(), runtimeDatabase);
    }

    @Bean
    public MongoDatabaseFactory individualDevMongoDatabaseFactory() {
        return new SimpleMongoClientDatabaseFactory(mongoClient(), individualDevDatabase);
    }

    @Bean
    public MongoDatabaseFactory individualTestMongoDatabaseFactory() {
        return new SimpleMongoClientDatabaseFactory(mongoClient(), individualTestDatabase);
    }

    @Bean
    public MongoDatabaseFactory individualMasterMongoDatabaseFactory() {
        return new SimpleMongoClientDatabaseFactory(mongoClient(), individualMasterDatabase);
    }

    @Bean
    public MongoDatabaseFactory individualBakMongoDatabaseFactory() {
        return new SimpleMongoClientDatabaseFactory(mongoClient(), individualBakDatabase);
    }

    @Primary
    @Bean
    public MongoTemplate mongoTemplate(MongoDatabaseFactory mongoDatabaseFactory,MappingMongoConverter mappingMongoConverter) throws Exception {
        return new MongoTemplate(mongoDatabaseFactory, mappingMongoConverter);
    }

    @Bean
    public MongoTemplate testMongoTemplate(@Qualifier("testMongoDatabaseFactory") MongoDatabaseFactory testMongoDatabaseFactory,
                                           @Qualifier("testMappingMongoConverter") MappingMongoConverter testMappingMongoConverter) throws Exception {
        return new MongoTemplate(testMongoDatabaseFactory, testMappingMongoConverter);
    }

    @Bean
    public MongoTemplate masterMongoTemplate(@Qualifier("masterMongoDatabaseFactory") MongoDatabaseFactory masterMongoDatabaseFactory,
                                             @Qualifier("masterMappingMongoConverter") MappingMongoConverter masterMappingMongoConverter) throws Exception {
        return new MongoTemplate(masterMongoDatabaseFactory, masterMappingMongoConverter);
    }

    @Bean
    public MongoTemplate bakMongoTemplate(@Qualifier("bakMongoDatabaseFactory") MongoDatabaseFactory bakMongoDatabaseFactory,
                                          @Qualifier("bakMappingMongoConverter") MappingMongoConverter bakMappingMongoConverter) throws Exception {
        return new MongoTemplate(bakMongoDatabaseFactory, bakMappingMongoConverter);
    }

    @Bean
    public MongoTemplate tenantMongoTemplate(@Qualifier("tenantMongoDatabaseFactory") MongoDatabaseFactory tenantMongoDatabaseFactory,
                                          @Qualifier("tenantMappingMongoConverter") MappingMongoConverter bakMappingMongoConverter) throws Exception {
        return new MongoTemplate(tenantMongoDatabaseFactory, bakMappingMongoConverter);
    }

    @Bean
    public MongoTemplate runtimeMongoTemplate(@Qualifier("runtimeMongoDatabaseFactory") MongoDatabaseFactory runtimeMongoDatabaseFactory,
                                             @Qualifier("runtimeMappingMongoConverter") MappingMongoConverter runtimeMappingMongoConverter) throws Exception {
        return new MongoTemplate(runtimeMongoDatabaseFactory, runtimeMappingMongoConverter);
    }

    @Bean
    public MongoTemplate individualDevMongoTemplate(@Qualifier("individualDevMongoDatabaseFactory") MongoDatabaseFactory individualDevMongoDatabaseFactory,
                                                    @Qualifier("individualDevMappingMongoConverter") MappingMongoConverter individualDevMappingMongoConverter) throws Exception {
        return new MongoTemplate(individualDevMongoDatabaseFactory, individualDevMappingMongoConverter);
    }
    @Bean
    public MongoTemplate individualTestMongoTemplate(@Qualifier("individualTestMongoDatabaseFactory") MongoDatabaseFactory individualTestMongoDatabaseFactory,
                                                     @Qualifier("individualTestMappingMongoConverter") MappingMongoConverter individualTestMappingMongoConverter) throws Exception {
        return new MongoTemplate(individualTestMongoDatabaseFactory, individualTestMappingMongoConverter);
    }
    @Bean
    public MongoTemplate individualMasterMongoTemplate(@Qualifier("individualMasterMongoDatabaseFactory") MongoDatabaseFactory individualMasterMongoDatabaseFactory,
                                                       @Qualifier("individualMasterMappingMongoConverter") MappingMongoConverter individualMasterMappingMongoConverter) throws Exception {
        return new MongoTemplate(individualMasterMongoDatabaseFactory, individualMasterMappingMongoConverter);
    }
    @Bean
    public MongoTemplate individualBakMongoTemplate(@Qualifier("individualBakMongoDatabaseFactory") MongoDatabaseFactory individualBakMongoDatabaseFactory,
                                                    @Qualifier("individualBakMappingMongoConverter") MappingMongoConverter individualBakMappingMongoConverter) throws Exception {
        return new MongoTemplate(individualBakMongoDatabaseFactory, individualBakMappingMongoConverter);
    }

    @Bean
    public MappingMongoConverter mappingMongoConverter(MongoDatabaseFactory mongoDatabaseFactory, MongoMappingContext context) throws Exception {
        return getMappingMongoConverter(new DefaultDbRefResolver(mongoDatabaseFactory),context);
    }

    @Bean
    public MappingMongoConverter testMappingMongoConverter(@Qualifier("testMongoDatabaseFactory") MongoDatabaseFactory testMongoDatabaseFactory, MongoMappingContext context) throws Exception {
        return getMappingMongoConverter(new DefaultDbRefResolver(testMongoDatabaseFactory),context);
    }
    @Bean
    public MappingMongoConverter masterMappingMongoConverter(@Qualifier("masterMongoDatabaseFactory") MongoDatabaseFactory masterMongoDatabaseFactory, MongoMappingContext context) throws Exception {
        return getMappingMongoConverter(new DefaultDbRefResolver(masterMongoDatabaseFactory),context);
    }
    @Bean
    public MappingMongoConverter bakMappingMongoConverter(@Qualifier("bakMongoDatabaseFactory") MongoDatabaseFactory bakMongoDatabaseFactory, MongoMappingContext context) throws Exception {
        return getMappingMongoConverter(new DefaultDbRefResolver(bakMongoDatabaseFactory),context);
    }

    @Bean
    public MappingMongoConverter tenantMappingMongoConverter(@Qualifier("tenantMongoDatabaseFactory") MongoDatabaseFactory tenantMongoDatabaseFactory, MongoMappingContext context) throws Exception {
        return getMappingMongoConverter(new DefaultDbRefResolver(tenantMongoDatabaseFactory),context);
    }

    @Bean
    public MappingMongoConverter runtimeMappingMongoConverter(@Qualifier("runtimeMongoDatabaseFactory") MongoDatabaseFactory runtimeMongoDatabaseFactory, MongoMappingContext context) throws Exception {
        return getMappingMongoConverter(new DefaultDbRefResolver(runtimeMongoDatabaseFactory),context);
    }




    @Bean
    public MappingMongoConverter individualDevMappingMongoConverter(@Qualifier("individualDevMongoDatabaseFactory") MongoDatabaseFactory individualDevMongoDatabaseFactory, MongoMappingContext context) throws Exception {
        return getMappingMongoConverter(new DefaultDbRefResolver(individualDevMongoDatabaseFactory),context);
    }

    @Bean
    public MappingMongoConverter individualTestMappingMongoConverter(@Qualifier("individualTestMongoDatabaseFactory") MongoDatabaseFactory individualTestMongoDatabaseFactory, MongoMappingContext context) throws Exception {
        return getMappingMongoConverter(new DefaultDbRefResolver(individualTestMongoDatabaseFactory),context);
    }
    @Bean
    public MappingMongoConverter individualMasterMappingMongoConverter(@Qualifier("individualMasterMongoDatabaseFactory") MongoDatabaseFactory individualMasterMongoDatabaseFactory, MongoMappingContext context) throws Exception {
        return getMappingMongoConverter(new DefaultDbRefResolver(individualMasterMongoDatabaseFactory),context);
    }

    @Bean
    public MappingMongoConverter individualBakMappingMongoConverter(@Qualifier("individualBakMongoDatabaseFactory") MongoDatabaseFactory individualBakMongoDatabaseFactory, MongoMappingContext context) throws Exception {
        return getMappingMongoConverter(new DefaultDbRefResolver(individualBakMongoDatabaseFactory),context);
    }

    private MappingMongoConverter getMappingMongoConverter(DefaultDbRefResolver mongoDbFactory, MongoMappingContext context) throws ClassNotFoundException {
        DbRefResolver dbRefResolver = mongoDbFactory;
        MappingMongoConverter converter = new MappingMongoConverter(dbRefResolver, context);
        //不保存 _class 属性到mongo
        converter.setTypeMapper(new DefaultMongoTypeMapper(null));
        converter.setCustomConversions(customConversions());
        return converter;
    }
    @Override
    protected String getDatabaseName() {
        return database;
    }

    public String getTenantDatabase() {
        return tenantDatabase;
    }

    public String getUri() {
        return uri;
    }

    public void setUri(String uri) {
        this.uri = uri;
    }

    public String getDatabase() {
        return database;
    }

    public void setDatabase(String database) {
        this.database = database;
    }

    public String getTestDatabase() {
        return testDatabase;
    }

    public void setTestDatabase(String testDatabase) {
        this.testDatabase = testDatabase;
    }

    public String getMasterDatabase() {
        return masterDatabase;
    }

    public void setMasterDatabase(String masterDatabase) {
        this.masterDatabase = masterDatabase;
    }

    public String getBakDatabase() {
        return bakDatabase;
    }

    public void setBakDatabase(String bakDatabase) {
        this.bakDatabase = bakDatabase;
    }

    public String getIndividualDevDatabase() {
        return individualDevDatabase;
    }

    public void setIndividualDevDatabase(String individualDevDatabase) {
        this.individualDevDatabase = individualDevDatabase;
    }

    public String getIndividualTestDatabase() {
        return individualTestDatabase;
    }

    public void setIndividualTestDatabase(String individualTestDatabase) {
        this.individualTestDatabase = individualTestDatabase;
    }

    public String getIndividualMasterDatabase() {
        return individualMasterDatabase;
    }

    public void setIndividualMasterDatabase(String individualMasterDatabase) {
        this.individualMasterDatabase = individualMasterDatabase;
    }

    public String getIndividualBakDatabase() {
        return individualBakDatabase;
    }

    public void setIndividualBakDatabase(String individualBakDatabase) {
        this.individualBakDatabase = individualBakDatabase;
    }

    public void setTenantDatabase(String tenantDatabase) {
        this.tenantDatabase = tenantDatabase;
    }

    public Map<String,String> getBranchDBNameMap(){
        HashMap<String, String> map = new HashMap<>();
        if (CurThreadInfoUtils.getIndividualCase()!=null && CurThreadInfoUtils.getIndividualCase()) {
            map.put(BranchConfig.DEV_BRANCH_NAME,individualDevDatabase);
            map.put(BranchConfig.TEST_BRANCH_NAME,individualTestDatabase);
            map.put(BranchConfig.MASTER_BRANCH_NAME,individualMasterDatabase);
            map.put(BranchConfig.BAK_BRANCH_NAME,individualBakDatabase);

        }else{

            map.put(BranchConfig.DEV_BRANCH_NAME,database);
            map.put(BranchConfig.TEST_BRANCH_NAME,testDatabase);
            map.put(BranchConfig.MASTER_BRANCH_NAME,masterDatabase);
            map.put(BranchConfig.BAK_BRANCH_NAME,bakDatabase);

        }
        return map;

    }
}
