package com.digiwin.athena.mongodb.repository;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.enums.VersionStatusEnum;
import com.digiwin.athena.utils.CurThreadInfoUtils;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.mongodb.client.result.DeleteResult;
import com.mongodb.client.result.UpdateResult;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.annotation.Id;
import org.springframework.data.mongodb.core.mapping.Document;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.lang.Nullable;
import org.springframework.stereotype.Repository;

import jakarta.annotation.PostConstruct;
import org.springframework.util.CollectionUtils;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.*;


/**
 * 多版本配置
 * 若为多版本，则加上版本信息进行检索
 */
@Slf4j
@Repository("mongoMultiVersionsRepositoryDecorator")
public class MongoMultiVersionsRepositoryDecorator {

    public static final String HISTORY_SUFFIX = "History";

    @Autowired
    private MongoMultiRepositoryDecorator mongoMultiRepositoryDecorator;
    @Autowired
    private MongoSystemRepositoryDecorator mongoSystemRepositoryDecorator;

    @Autowired
    private MongoTenantRepositoryDecorator mongoTenantRepositoryDecorator;

    private final static Map<String, MultiVersionsCollection> VERSIONS_TABLE = new HashMap<>();

    private MongoUserRepositoryDecorator chooseRepository() {
        if (judgeTenantAdp()) {
            return mongoTenantRepositoryDecorator;
        } else {
            return mongoMultiRepositoryDecorator;
        }
    }

    @PostConstruct
    public void init() {
        Criteria criteria = Criteria.where("versionsFlag").is("Y");
        List<JSONObject> sysTableDefinitions = mongoSystemRepositoryDecorator.find(new Query(criteria), JSONObject.class,"sys_tableDefinition");
        for (JSONObject tableDefinition : sysTableDefinitions) {

            MultiVersionsCollection sysMultiVersionsCollection = new MultiVersionsCollection();
            sysMultiVersionsCollection.setTableName(tableDefinition.getString("collection"));
            sysMultiVersionsCollection.setAdpRemark(tableDefinition.getString("adpRemark"));
            sysMultiVersionsCollection.setNeedUpdateAdpRemark("Y".equals(tableDefinition.getString("versionNeedUpdateAdpRemark")));
            sysMultiVersionsCollection.setPrimaryKey(tableDefinition.getString("versionsPrimaryKey"));


            VERSIONS_TABLE.put(tableDefinition.getString("collection"), sysMultiVersionsCollection);
        }
    }

    private String getCollectionName(Class clazz) {
        Document document = (Document) clazz.getAnnotation(Document.class);
        return StringUtils.isEmpty(document.value()) ? document.collection() : document.value();
    }

    private Boolean judgeVersionSync(String collectionName) {
        return !StringUtils.isBlank(CurThreadInfoUtils.getAdpVersion()) && VERSIONS_TABLE.containsKey(collectionName);
    }

    private Boolean judgeTenantAdp() {
        return !StringUtils.isBlank(CurThreadInfoUtils.getPlatformSource()) && "tenant".equals(CurThreadInfoUtils.getPlatformSource());
    }

    private Boolean hasOtherHeaderFlag (){
        return StringUtils.isNotBlank(CurThreadInfoUtils.getTemplateId());
    }

    private Boolean judgeKeyHasValue(Object obj) {
        try {
            Class<?> currentClass = obj.getClass();
            while (currentClass != null) {
                for (Field field : currentClass.getDeclaredFields()) {
                    if (field.isAnnotationPresent(Id.class)) {
                        // 允许访问私有字段
                        field.setAccessible(true);

                        // 获取字段的值
                        Object value = field.get(obj);

                        // 检查值是否为空
                        if (value != null) {
                            return true;
                        }
                    }
                }
                // 继续向上获取父类的字段
                currentClass = currentClass.getSuperclass();
            }
        } catch (Exception e) {
            log.error("判断id字段是否有值，" + e.getMessage(), e);
            throw new RuntimeException("判断id字段是否有值，" + e.getMessage());
        }
        return false;
    }

    Object getHistoryPrimaryKeyValue(Object obj, String primaryKey) {
        try {
            Class<?> currentClass = obj.getClass();
            while (currentClass != null) {
                for (Field field : currentClass.getDeclaredFields()) {

                    if (field.getName().equals(primaryKey)) {
                        field.setAccessible(true);
                        // 获取字段的值
                        Object value = field.get(obj);
                        return value;
                    }
                }
                // 继续向上获取父类的字段
                currentClass = currentClass.getSuperclass();
            }
            return null;
        } catch (Exception e) {
            log.error("获取主键值失败，" + e.getMessage(), e);
            throw new RuntimeException("获取主键值失败，" + e.getMessage());
        }
    }

    /**
     * 设置流程多版本号
     *
     * @param o          o
     * @param adpVersion adpVersion
     */
    private void setAdpVersion(Object o, String adpVersion) {
        try {
            if (StringUtils.isNotBlank(adpVersion)) {
                Class<?> clazz = Class.forName(o.getClass().getName());
                Method setMethod = clazz.getMethod("setAdpVersion", String.class);
                setMethod.invoke(o, adpVersion);
            }
        } catch (Exception e) {
            log.error("setAdpVersion 失败，" + e.getMessage(), e);
        }
    }

    /**
     * 设置租户级流程Id
     *
     * @param o               o
     * @param tenantProcessId tenantProcessId
     */
    private void setTenantProcessId(Object o, String tenantProcessId) {
        try {
            if (StringUtils.isNotBlank(tenantProcessId)) {
                Class<?> clazz = Class.forName(o.getClass().getName());
                Method setMethod = clazz.getMethod("setTenantProcessId", String.class);
                setMethod.invoke(o, tenantProcessId);
            }
        } catch (Exception e) {
            log.error("setTenantProcessId 失败，" + e.getMessage(), e);
        }
    }

    private void setAdpRemark(Object o, Object adpRemark) {
        try {
            Class<?> clazz = Class.forName(o.getClass().getName());
            Method setMethod = clazz.getMethod("setAdpRemark", String.class);
            setMethod.invoke(o, adpRemark);

        } catch (Exception e) {
            log.error("setMethod 失败，" + e.getMessage(), e);
        }
    }

    private Object findAdpRemark(Object primaryKeyValue, Class clazz,String collectionName) throws NoSuchMethodException, InvocationTargetException, IllegalAccessException {
        Criteria criteria = Criteria.where("_id").is(primaryKeyValue).and("adpVersion").is(CurThreadInfoUtils.getAdpVersion());
        Query query = new Query(criteria);
        appendTenantProcessIdQueryCondition(query);
        org.springframework.data.mongodb.core.query.Field fields = query.fields();
        fields.include("adpRemark");
        Object historyObj = chooseRepository().findOne(query, clazz);

        Method getAdpRemark = clazz.getMethod("getAdpRemark");
        if (Objects.isNull(historyObj)){
            MultiVersionsCollection sysMultiVersionsCollection = VERSIONS_TABLE.get(collectionName);
            return sysMultiVersionsCollection.getAdpRemark();
        }
        return getAdpRemark.invoke(historyObj);
    }

    /**
     * adpVersion有值且当前版本为生效版本则插入当前表数据
     * 若存在子表则插入子表数据
     * adpVersion没有值-则插入当前表数据
     */
    public <T> T insert(T t) {
        T res = null;
        if (StringUtils.isNotBlank(CurThreadInfoUtils.getAdpVersion())) {
            String collectionName = getCollectionName(t.getClass());
            setAdpVersion(t, CurThreadInfoUtils.getAdpVersion());
            setTenantProcessId(t, CurThreadInfoUtils.getTenantProcessId());
            if (VersionStatusEnum.RUN_STATE.getCode().equals(CurThreadInfoUtils.getAdpStatus())) {
                res = chooseRepository().insert(t);
            }
            if (VERSIONS_TABLE.containsKey(collectionName)) {
                try {
                    MultiVersionsCollection sysMultiVersionsCollection = VERSIONS_TABLE.get(collectionName);
                    Class<?> clazz = Class.forName(t.getClass().getName() + HISTORY_SUFFIX);
                    ObjectMapper objectMapper = new ObjectMapper();
                    String s = objectMapper.writeValueAsString(t);
                    Object historyObj = objectMapper.readValue(s, clazz);
                    if (sysMultiVersionsCollection.getNeedUpdateAdpRemark()) {
                        setAdpRemark(historyObj, sysMultiVersionsCollection.getAdpRemark());
                    }
                    return (T) chooseRepository().insert(historyObj);
                } catch (Exception e) {
                    log.error("同步表数据异常，" + e.getMessage(), e);
                    throw new RuntimeException("同步表数据异常，" + e.getMessage());
                }
            }
        } else {
            res = chooseRepository().insert(t);
        }
        return res;
    }

    public <T> Collection<? extends T> insertAll(Collection<? extends T> collection) {
        String collectionName = null;
        Collection<? extends T> res = null;
        for (T t : collection) {
            collectionName = getCollectionName(t.getClass());
            setAdpVersion(t, CurThreadInfoUtils.getAdpVersion());
            setTenantProcessId(t, CurThreadInfoUtils.getTenantProcessId());
        }
        if (StringUtils.isNotBlank(CurThreadInfoUtils.getAdpVersion())) {
            if (VersionStatusEnum.RUN_STATE.getCode().equals(CurThreadInfoUtils.getAdpStatus())) {
                res = chooseRepository().insertAll(collection);
            }
            if (VERSIONS_TABLE.containsKey(collectionName)) {
                MultiVersionsCollection sysMultiVersionsCollection = VERSIONS_TABLE.get(collectionName);
                List<Object> historyCollection = new ArrayList<>();
                try {
                    if(CollectionUtils.isEmpty(res)){
                        res = collection;
                    }
                    for (T t : res) {
                        Class<?> clazz = Class.forName(t.getClass().getName() + HISTORY_SUFFIX);
                        ObjectMapper objectMapper = new ObjectMapper();
                        String s = objectMapper.writeValueAsString(t);
                        Object historyObj = objectMapper.readValue(s, clazz);
                        if (sysMultiVersionsCollection.getNeedUpdateAdpRemark()) {
                            setAdpRemark(historyObj, "流程多版本V1");
                        }
                        historyCollection.add(historyObj);
                    }
                    chooseRepository().insertAll(historyCollection);
                } catch (Exception e) {
                    log.error("同步表数据异常，" + e.getMessage(), e);
                    throw new RuntimeException("同步表数据异常，" + e.getMessage());
                }
            }
        } else {
            res = chooseRepository().insertAll(collection);
        }
        return res;
    }

    public <T> T save(T t) {
        try {
            String collectionName = getCollectionName(t.getClass());
            if (judgeVersionSync(collectionName)) {
                //判断主键id是否有值，有值更新，没值新增
                if (judgeKeyHasValue(t)) {
                    setAdpVersion(t, CurThreadInfoUtils.getAdpVersion());
                    setTenantProcessId(t, CurThreadInfoUtils.getTenantProcessId());
                    //更新 如果是启用中的，需要同步更新主表 ，如果不是启用中的，只需要更新history表
                    Class<?> historyClazz = Class.forName(t.getClass().getName() + HISTORY_SUFFIX);
                    MultiVersionsCollection sysMultiVersionsCollection = VERSIONS_TABLE.get(collectionName);
                    String primaryKey = sysMultiVersionsCollection.getPrimaryKey();
                    Object primaryKeyValue = getHistoryPrimaryKeyValue(t, primaryKey);
                    ObjectMapper objectMapper = new ObjectMapper();
                    String s = objectMapper.writeValueAsString(t);
                    Object historyObj = objectMapper.readValue(s, historyClazz);
                    if (sysMultiVersionsCollection.getNeedUpdateAdpRemark()) {
                        Object historyObjRemark = findAdpRemark(primaryKeyValue, historyClazz,collectionName);
                        setAdpRemark(historyObj, historyObjRemark);
                    }
                    //如果是启用中的，需要同步更新主表 ，如果不是启用中的，只需要更新history表
                    if (VersionStatusEnum.RUN_STATE.getCode().equals(CurThreadInfoUtils.getAdpStatus())) {
                        chooseRepository().save(t);
                    }
                    Object res = chooseRepository().save(historyObj);
                    return (T) res;
                } else {
                    return insert(t);
                }
            } else {
                return chooseRepository().save(t);
            }
        } catch (Exception e) {
            log.error("MongoMultiVersionsRepositoryDecorator save 异常" + e.getMessage(), e);
            throw new RuntimeException("MongoMultiVersionsRepositoryDecorator save 异常" + e.getMessage());
        }
    }

    public <T> long remove(Query query, String collectionName, Class<T> t) {
        // 存在adpVersion，则根据adpVersion进行删除
        appendQueryCondition(query);
        try {
            if (VERSIONS_TABLE.containsKey(collectionName)&& !hasOtherHeaderFlag()) {
                Class<?> historyClazz = Class.forName(t.getName() + HISTORY_SUFFIX);
                if (StringUtils.isNotBlank(CurThreadInfoUtils.getAdpVersion())) {
                    chooseRepository().remove(query, collectionName, t);
                    return chooseRepository().remove(query, collectionName + HISTORY_SUFFIX, historyClazz);
                } else {
                    chooseRepository().remove(query, collectionName + HISTORY_SUFFIX, historyClazz);
                    return chooseRepository().remove(query, collectionName, t);
                }
            } else {
                return chooseRepository().remove(query, collectionName, t);
            }
        } catch (ClassNotFoundException e) {
            log.error("MongoMultiVersionsRepositoryDecorator remove 异常" + e.getMessage(), e);
            throw new RuntimeException("MongoMultiVersionsRepositoryDecorator remove 异常" + e.getMessage());
        }
    }

    public <T> long removeMain(Query query, Class<T> t) {
        appendTenantProcessIdQueryCondition(query);
        return chooseRepository().remove(query, t);
    }

    public <T> DeleteResult deleteMain(Query query, Class<T> t) {
        appendTenantProcessIdQueryCondition(query);
        return chooseRepository().delete(query, t);
    }


    public <T> T findMain(Query query, Class<T> t) {
        String collectionName = getCollectionName(t);
        return chooseRepository().findOne(query, collectionName, t);
    }

    public <T> List<T> findMainList(Query query, Class<T> t) {
        return chooseRepository().find(query, t);
    }

    public <T> DeleteResult delete(Query query, Class<T> t) {
        String collectionName = getCollectionName(t);
        appendQueryCondition(query);
        try {
            if (VERSIONS_TABLE.containsKey(collectionName) && !hasOtherHeaderFlag()) {
                Class<?> historyClazz = Class.forName(t.getName() + HISTORY_SUFFIX);
                if (StringUtils.isNotBlank(CurThreadInfoUtils.getAdpVersion())) {
                    chooseRepository().delete(query, t);
                    return chooseRepository().delete(query, historyClazz);
                } else {
                    chooseRepository().delete(query, historyClazz);
                    return chooseRepository().delete(query, t);
                }
            } else {
                return chooseRepository().delete(query, t);
            }
        } catch (ClassNotFoundException e) {
            log.error("MongoMultiVersionsRepositoryDecorator delete 异常" + e.getMessage(), e);
            throw new RuntimeException("MongoMultiVersionsRepositoryDecorator delete 异常" + e.getMessage());
        }
    }

    public <T> DeleteResult delete(T t) {
        Object objectIdValue = findObjectIdValue(t);

        Query query = new Query(Criteria.where("_id").is(objectIdValue));
        return this.delete(query,t.getClass());

    }

    private <T> Object findObjectIdValue(T t){
        Class<?> aClass = t.getClass();
        while (aClass != null && aClass != Object.class) {
            for (Field field : aClass.getDeclaredFields()) {
                if (field.isAnnotationPresent(Id.class)) {
                    field.setAccessible(true); // 允许访问私有字段
                    try {
                        return field.get(t);
                    } catch (IllegalAccessException e) {
                        log.error("findObjectIdValue exception:"+e.getMessage(),e);
                        throw new RuntimeException("findObjectIdValue exception:"+e.getMessage());
                    }
                }
            }
            aClass = aClass.getSuperclass();
        }
        throw new RuntimeException("未查找到对象的@Id字段值");
    }

    public <T> long remove(Query query, Class<T> t) {
        String collectionName = getCollectionName(t);
        return remove(query, collectionName, t);
    }

    public <T> List<T> find(Query query, Class<T> t) {
        String collectionName = getCollectionName(t);
        return find(collectionName, query, t);
    }

    public <T> List<T> find(String collectionName, Query query, Class<T> t) {
        appendTenantProcessIdQueryCondition(query);
        try {
            if (judgeVersionSync(collectionName)) {
                Class<?> historyClazz = Class.forName(t.getName() + HISTORY_SUFFIX);
                appendAdpVersionQueryCondition(query);
                List<?> res = chooseRepository().find(collectionName + HISTORY_SUFFIX, query, historyClazz);
                return JSONObject.parseArray(JSONObject.toJSONString(res), t);
            } else {
                return chooseRepository().find(collectionName, query, t);
            }
        } catch (ClassNotFoundException e) {
            log.error("MongoMultiVersionsRepositoryDecorator find 异常" + e.getMessage(), e);
            throw new RuntimeException("MongoMultiVersionsRepositoryDecorator find 异常" + e.getMessage());
        }
    }

    /**
     * 补充流程版本查询条件
     *
     * @param query query
     */
    private static void appendAdpVersionQueryCondition(Query query) {
        boolean hasAdpVersion = query.getQueryObject().containsKey("adpVersion");
        // 流程版本
        if (!hasAdpVersion && StringUtils.isNotBlank(CurThreadInfoUtils.getAdpVersion())) {
            Criteria adpVersionCriteria = Criteria.where("adpVersion").is(CurThreadInfoUtils.getAdpVersion());
            query.addCriteria(adpVersionCriteria);
        }
    }

    /**
     * 补充租户级流程Id查询条件
     *
     * @param query query
     */
    private static void appendTenantProcessIdQueryCondition(Query query) {
        // 租户级流程Id
        if (StringUtils.isNotBlank(CurThreadInfoUtils.getTenantProcessId())) {
            Criteria tenantProcessIdCriteria = Criteria.where("tenantProcessId").is(CurThreadInfoUtils.getTenantProcessId());
            query.addCriteria(tenantProcessIdCriteria);
        }
    }

    /**
     * 补充查询条件
     *
     * @param query query
     */
    private static void appendQueryCondition(Query query) {
        boolean hasAdpVersion = query.getQueryObject().containsKey("adpVersion");
        // 流程版本
        if (!hasAdpVersion && !StringUtils.isBlank(CurThreadInfoUtils.getAdpVersion())) {
            Criteria adpVersionCriteria = Criteria.where("adpVersion").is(CurThreadInfoUtils.getAdpVersion());
            query.addCriteria(adpVersionCriteria);
        }
        // 租户级流程Id
        if (!StringUtils.isBlank(CurThreadInfoUtils.getTenantProcessId())) {
            Criteria tenantProcessIdCriteria = Criteria.where("tenantProcessId").is(CurThreadInfoUtils.getTenantProcessId());
            query.addCriteria(tenantProcessIdCriteria);
        }
    }

    public <T> T findOne(Query query, Class<T> t) {
        String collectionName = getCollectionName(t);
        return findOne(query, collectionName, t);
    }

    public <T> T findOne(Query query, String collectionName, Class<T> t) {
        appendTenantProcessIdQueryCondition(query);
        try {
            if (judgeVersionSync(collectionName)) {
                Class<?> historyClazz = Class.forName(t.getName() + HISTORY_SUFFIX);
                appendAdpVersionQueryCondition(query);
                Object res = chooseRepository().findOne(query, collectionName + HISTORY_SUFFIX, historyClazz);
                return JSONObject.parseObject(JSONObject.toJSONString(res), t);
            } else {
                return chooseRepository().findOne(query, collectionName, t);
            }
        } catch (ClassNotFoundException e) {
            log.error("MongoMultiVersionsRepositoryDecorator findOne 异常" + e.getMessage(), e);
            throw new RuntimeException("MongoMultiVersionsRepositoryDecorator findOne 异常" + e.getMessage());
        }
    }

    public long count(Query query, @Nullable Class<?> t) {
        String collectionName = getCollectionName(t);
        appendTenantProcessIdQueryCondition(query);
        try {
            if (judgeVersionSync(collectionName)) {
                Class<?> historyClazz = Class.forName(t.getName() + HISTORY_SUFFIX);
                appendAdpVersionQueryCondition(query);
                return chooseRepository().count(query, historyClazz);
            } else {
                return chooseRepository().count(query, t);
            }
        } catch (ClassNotFoundException e) {
            log.error("MongoMultiVersionsRepositoryDecorator count 异常" + e.getMessage(), e);
            throw new RuntimeException("MongoMultiVersionsRepositoryDecorator count 异常" + e.getMessage());
        }
    }

    public <T> UpdateResult updateFirst(Query query, Update update, Class<T> t) {
        String collectionName = getCollectionName(t);
        appendTenantProcessIdQueryCondition(query);
        try {
            if (judgeVersionSync(collectionName)) {
                Class<?> historyClazz = Class.forName(t.getName() + HISTORY_SUFFIX);
                appendAdpVersionQueryCondition(query);
                return chooseRepository().updateFirst(query, update, historyClazz);
            } else {
                return chooseRepository().updateFirst(query, update, t);
            }
        } catch (ClassNotFoundException e) {
            log.error("MongoMultiVersionsRepositoryDecorator count 异常" + e.getMessage(), e);
            throw new RuntimeException("MongoMultiVersionsRepositoryDecorator count 异常" + e.getMessage());
        }
    }

    public <T> long updateMulti(Query query, Update update, Class<T> t) {
        String collectionName = getCollectionName(t);
        appendTenantProcessIdQueryCondition(query);
        try {
            if (judgeVersionSync(collectionName)) {
                Class<?> historyClazz = Class.forName(t.getName() + HISTORY_SUFFIX);
                appendAdpVersionQueryCondition(query);
                return chooseRepository().updateMulti(query, update, historyClazz);
            } else {
                return chooseRepository().updateMulti(query, update, t);
            }
        } catch (ClassNotFoundException e) {
            log.error("MongoMultiVersionsRepositoryDecorator count 异常" + e.getMessage(), e);
            throw new RuntimeException("MongoMultiVersionsRepositoryDecorator count 异常" + e.getMessage());
        }
    }
}
