package com.digiwin.athena.mongodb.repository;

import com.mongodb.client.result.DeleteResult;
import com.mongodb.client.result.UpdateResult;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.mongodb.MongoDatabaseFactory;
import org.springframework.data.mongodb.core.BulkOperations;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.aggregation.Aggregation;
import org.springframework.data.mongodb.core.aggregation.AggregationResults;
import org.springframework.data.mongodb.core.aggregation.TypedAggregation;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.data.support.PageableExecutionUtils;
import org.springframework.lang.Nullable;
import org.springframework.stereotype.Repository;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @ClassName MongoRepository
 * @Description TODO
 * @Author zhuangli
 * @Date 2021/9/26 14:38
 * @Version 1.0
 **/
@Slf4j
@Repository(value = "mongoDynamicRepository")
public class MongoRepository implements ApplicationContextAware {

    @Value("${spring.data.mongodb.uri}")
    private String uri;

    private static final Map<String, MongoTemplate> MONGO_TEMPLATE_CACHE = new HashMap<>(16);

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        Map<String, MongoTemplate> mongoTemplateMap = applicationContext.getBeansOfType(MongoTemplate.class);
        for (MongoTemplate mongoTemplate : mongoTemplateMap.values()) {
            MONGO_TEMPLATE_CACHE.put(mongoTemplate.getDb().getName(), mongoTemplate);
        }
        log.info("mongo映射" + MONGO_TEMPLATE_CACHE);
    }

    /**
     * 插入一条记录
     *
     * @param databaseName 数据库名
     * @param t            实例
     * @param <T>          实例所属的类
     */
    public <T> T insert(String databaseName, T t) {
        
        return MONGO_TEMPLATE_CACHE.get(databaseName).insert(t);
    }

    public <T> UpdateResult upsert(String dbName, Query query, Update update, String collection) {
        return MONGO_TEMPLATE_CACHE.get(dbName).upsert(query, update, collection);
    }

    /**
     * 插入一条记录
     *
     * @param databaseName   数据库名
     * @param collectionName 集合名
     * @param t              实例
     * @param <T>            实例所属的类
     */
    public <T> T insert(String databaseName, String collectionName, T t) {
        
        return MONGO_TEMPLATE_CACHE.get(databaseName).insert(t, collectionName);
    }

    public <T> T save(String databaseName, T t) {
        
        return MONGO_TEMPLATE_CACHE.get(databaseName).save(t);
    }

    public <T> T save(String databaseName, String collectionName, T t) {
        
        return MONGO_TEMPLATE_CACHE.get(databaseName).save(t, collectionName);
    }

    /**
     * 插入多条记录
     *
     * @param databaseName 数据库名
     * @param tClass       实例的class
     * @param tList        实例
     * @param <T>          实例所属的类
     */
    public <T> void insertAll(String databaseName, Class<T> tClass, List<T> tList) {
        
        MONGO_TEMPLATE_CACHE.get(databaseName).insert(tList, tClass);
    }

    /**
     * 插入多条记录
     *
     * @param databaseName   数据库名
     * @param collectionName 集合名
     * @param tList          实例
     * @param <T>            实例所属的类
     */
    public <T> void insertAll(String databaseName, String collectionName, List<T> tList) {
        
        MONGO_TEMPLATE_CACHE.get(databaseName).insert(tList, collectionName);
    }

    public <T> DeleteResult delete(String databaseName, T obj) {
        
        return MONGO_TEMPLATE_CACHE.get(databaseName).remove(obj);
    }

    public <T> DeleteResult delete(String databaseName, T obj, String collectionName) {

        return MONGO_TEMPLATE_CACHE.get(databaseName).remove(obj,collectionName);
    }


    /**
     * 移除一条或多条记录
     *
     * @param databaseName 数据库名
     * @param tClass       实例的class
     * @param query        查询条件
     * @param <T>          实例所属的类
     * @return
     */
    public <T> long remove(String databaseName, Class<T> tClass, Query query) {
        
        return MONGO_TEMPLATE_CACHE.get(databaseName).remove(query, tClass).getDeletedCount();
    }

    public <T> DeleteResult delete(String databaseName, Class<T> tClass, Query query) {
        
        return MONGO_TEMPLATE_CACHE.get(databaseName).remove(query, tClass);
    }

    /**
     * 移除一条或多条记录
     *
     * @param databaseName   数据库名
     * @param collectionName 集合名
     * @param query          查询条件
     * @return 受影响的记录条数
     */
    public long remove(String databaseName, String collectionName, Query query) {
        
        return MONGO_TEMPLATE_CACHE.get(databaseName).remove(query, collectionName).getDeletedCount();
    }

    public <T> UpdateResult updateFirst(String databaseName, Class<T> tClass, Query query, Update update) {
        
        return MONGO_TEMPLATE_CACHE.get(databaseName).updateFirst(query, update, tClass);
    }

    public <T> UpdateResult updateFirst(String databaseName, String collection, Query query, Update update) {
        
        return MONGO_TEMPLATE_CACHE.get(databaseName).updateFirst(query, update, collection);
    }

    /**
     * 更新多条记录
     *
     * @param databaseName 数据库名
     * @param tClass       实例的class
     * @param query        查询条件
     * @param update       更新内容
     * @param <T>          实例所属的类
     * @return 受影响的记录条数
     */
    public <T> long updateMulti(String databaseName, Class<T> tClass, Query query, Update update) {
        
        return MONGO_TEMPLATE_CACHE.get(databaseName).updateMulti(query, update, tClass).getModifiedCount();
    }

    /**
     * 更新多条记录
     *
     * @param databaseName   数据库名
     * @param collectionName 集合名
     * @param query          查询条件
     * @param update         更新内容
     * @return 受影响的记录条数
     */
    public long updateMulti(String databaseName, String collectionName, Query query, Update update) {
        
        return MONGO_TEMPLATE_CACHE.get(databaseName).updateMulti(query, update, collectionName).getModifiedCount();
    }

    /**
     * 查询多条记录
     *
     * @param databaseName 数据库名
     * @param tClass       实例的class
     * @param query        查询条件
     * @param <T>          实例所属的类
     * @return 实例
     */
    public <T> List<T> find(String databaseName, Class<T> tClass, Query query) {
        
        return MONGO_TEMPLATE_CACHE.get(databaseName).find(query, tClass);
    }

    public <T> List<T> findAll(String databaseName, Class<T> tClass) {
        
        return MONGO_TEMPLATE_CACHE.get(databaseName).findAll(tClass);
    }

    public <T> List<T> findAll(String databaseName, String collectionName, Class<T> tClass) {
        
        return MONGO_TEMPLATE_CACHE.get(databaseName).findAll(tClass, collectionName);
    }

    /**
     * 查询多条记录
     *
     * @param databaseName   数据库名
     * @param collectionName 集合名
     * @param query          查询条件
     * @param tClass         实例的class
     * @param <T>            实例所属的类
     * @return 实例
     */
    public <T> List<T> find(String databaseName, String collectionName, Query query, Class<T> tClass) {
        
        return MONGO_TEMPLATE_CACHE.get(databaseName).find(query, tClass, collectionName);
    }

    /**
     * 查询第一条记录
     *
     * @param databaseName 数据库名
     * @param tClass       实例的class
     * @param query        查询条件
     * @param <T>          实例所属的类
     * @return 实例
     */
    public <T> T findOne(String databaseName, Class<T> tClass, Query query) {
        
        return MONGO_TEMPLATE_CACHE.get(databaseName).findOne(query, tClass);
    }

    /**
     * 查询第一条记录
     *
     * @param databaseName   数据库名
     * @param tClass         实例的class
     * @param collectionName 集合名
     * @param query          查询条件
     * @param <T>            实例所属的类
     * @return 实例
     */
    public <T> T findOne(String databaseName, Class<T> tClass, String collectionName, Query query) {
        
        return MONGO_TEMPLATE_CACHE.get(databaseName).findOne(query, tClass, collectionName);
    }

    public long count(String databaseName, Query query, @Nullable Class<?> entityClass) {
        
        return MONGO_TEMPLATE_CACHE.get(databaseName).count(query, entityClass);
    }

    public <T> Page<T> findWithPage(String databaseName, Pageable pageable, Class<T> tClass, Query query) {
        List<T> resultList = find(databaseName, tClass, query.with(pageable));
        return PageableExecutionUtils.getPage(
                resultList,
                pageable,
                () -> count(databaseName, Query.of(query).limit(-1).skip(-1), tClass));

    }

    public Set<String> getDbCollectionNames(String databaseName) {
        
        return MONGO_TEMPLATE_CACHE.get(databaseName).getCollectionNames();
    }

    public <T> T findById(String databaseName, Object id, Class<T> entityClass) {
        return MONGO_TEMPLATE_CACHE.get(databaseName).findById(id, entityClass);
    }

    public <T> Collection<T> insertAll(String databaseName, Collection<? extends T> collection) {

        return MONGO_TEMPLATE_CACHE.get(databaseName).insertAll(collection);
    }

    public <T> T findOne(String databaseName, Query query, Class<T> entityClass, String collectionName) {
        return MONGO_TEMPLATE_CACHE.get(databaseName).findOne(query, entityClass, collectionName);
    }

    public <T> List<T> find(String databaseName, Query query, Class<T> entityClass, String collectionName) {
        return MONGO_TEMPLATE_CACHE.get(databaseName).find(query, entityClass, collectionName);
    }

    public DeleteResult remove(String databaseName, Query query, String collectionName) {
        return MONGO_TEMPLATE_CACHE.get(databaseName).remove(query, collectionName);
    }

    public <T> T save(String databaseName, T objectToSave, String collectionName) {
        return MONGO_TEMPLATE_CACHE.get(databaseName).save(objectToSave, collectionName);
    }

    public long count(String databaseName, Query query, @Nullable Class<?> entityClass, String collectionName) {
        return MONGO_TEMPLATE_CACHE.get(databaseName).count(query, entityClass, collectionName);
    }

    public <T> Collection<T> insert(String databaseName, Collection<? extends T> batchToSave, Class<?> entityClass) {
        return MONGO_TEMPLATE_CACHE.get(databaseName).insert(batchToSave, entityClass);
    }

    public <T> T insert(String databaseName, T objectToSave, String collectionName) {
        return MONGO_TEMPLATE_CACHE.get(databaseName).insert(objectToSave, collectionName);
    }

    public MongoDatabaseFactory getMongoDbFactory(String databaseName) {
        return MONGO_TEMPLATE_CACHE.get(databaseName).getMongoDatabaseFactory();
    }

    public <T> List<T> findAllAndRemove(String databaseName, Query query, Class<T> entityClass, String collectionName) {
        return MONGO_TEMPLATE_CACHE.get(databaseName).findAllAndRemove(query, entityClass, collectionName);
    }

    public boolean exists(String databaseName, Query query, @Nullable Class<?> entityClass, String collectionName) {
        return MONGO_TEMPLATE_CACHE.get(databaseName).exists(query, entityClass, collectionName);
    }

    public <O> AggregationResults<O> aggregate(String databaseName, Aggregation aggregation, Class<?> inputType, Class<O> outputType) {

        return MONGO_TEMPLATE_CACHE.get(databaseName).aggregate(aggregation, inputType, outputType);
    }

    public <O> AggregationResults<O> aggregate(String databaseName, TypedAggregation<?> aggregation, Class<O> outputType) {
        return MONGO_TEMPLATE_CACHE.get(databaseName).aggregate(aggregation, outputType);
    }

    public <O> AggregationResults<O> aggregate(String databaseName, Aggregation aggregation, String collectioName, Class<O> outputType) {
        return MONGO_TEMPLATE_CACHE.get(databaseName).aggregate(aggregation, collectioName, outputType);
    }

    public BulkOperations bulkOps(String databaseName, BulkOperations.BulkMode mode, Class<?> entityType) {
        return MONGO_TEMPLATE_CACHE.get(databaseName).bulkOps(mode,entityType);
    }

    public Set<String> getCollectionNames(String databaseName) {
        return MONGO_TEMPLATE_CACHE.get(databaseName).getCollectionNames();
    }

    public MongoTemplate getTargetMongoTemplate(String databaseName){
        return MONGO_TEMPLATE_CACHE.get(databaseName);
    }
}
