package com.digiwin.athena.mongodb.repository;

import cn.hutool.core.convert.Convert;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.NumberUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.base.BusinessException;
import com.digiwin.athena.config.mongodb.MongodbConfig;
import com.digiwin.athena.dto.PageReqCondition;
import com.digiwin.athena.dto.Pagination;
import com.mongodb.client.result.DeleteResult;
import com.mongodb.client.result.UpdateResult;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.MongoDatabaseFactory;
import org.springframework.data.mongodb.core.BulkOperations;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.aggregation.Aggregation;
import org.springframework.data.mongodb.core.aggregation.AggregationResults;
import org.springframework.data.mongodb.core.aggregation.TypedAggregation;
import org.springframework.data.mongodb.core.mapping.Document;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.lang.Nullable;
import org.springframework.util.CollectionUtils;

import java.security.SecureRandom;
import java.util.*;

/**
 * @ClassName MongoUserRepository
 * @Description
 * @Author zhuangli
 * @Date 2022/6/17 17:47
 * @Version 1.0
 **/
@Slf4j
public abstract class MongoUserRepositoryDecorator {

    @Autowired
    private MongodbConfig mongodbConfig;
    @Autowired
    @Qualifier("mongoDynamicRepository")
    private MongoRepository mongoRepository;

    abstract Map<String, String> getBranchDBNameMap();

    abstract String getBranchName();

    private static final SecureRandom random = new SecureRandom();

    public String getDBName() {
        Map<String, String> branchDBNameMap = getBranchDBNameMap();
        String branchName = getBranchName();
        String dbName = branchDBNameMap.get(branchName);
        Assert.isTrue(StringUtils.isNotBlank(dbName), "The current branch does not have database");

        return dbName;
    }

    protected String getWDBName() {
        return getDBName();

    }



    /**
     * 注意用法：更新已有document的部分字段
     */
    public <T> UpdateResult upsert(Query query, Update update, String collection) {
//        String apk = getCollectionPK(collection);
//        String pkValue = (String) ((org.bson.Document) update.getUpdateObject().get("$set")).get(apk);
        String DBName = getWDBName();
        preOperateProcess(update, false);
        return mongoRepository.upsert(DBName, query, update, collection);
    }

    /**
     * 更新，如果查询不到跳过更新
     * @param query
     * @param update
     * @param tClass
     * @return
     * @param <T>
     */
    public <T> long updateMulti( Query query, Update update, Class<T> tClass) {
        String DBName = getDBName();
        preOperateProcess(update, false);
        return mongoRepository.updateMulti(DBName, tClass, query, update);
    }

    /**
     * 注意用法用法：更新已有document的部分字段
     */
    public <T> UpdateResult upsert(Query query, Update update, String collection, String application) {
        String DBName = getWDBName();
        preOperateProcess(update, false);
        return mongoRepository.upsert(DBName, query, update, collection);
    }

    /**
     * 注意用法：更新已有document的部分字段
     */
    public <T> UpdateResult upsert(Query query, T t, Class<T> clazz) {
        String json = JSON.toJSONString(t);
        JSONObject jsonObject = JSON.parseObject(json);
        Update update = new Update();
        jsonObject.forEach((k, v) -> {
            if ("createTime".equals(k)) {
                if (v instanceof Long) {
                    update.set(k, DateUtil.date(Convert.toLong(v)));
                } else if (v instanceof String && !NumberUtil.isNumber(String.valueOf(v))) {
                    update.set(k, DateUtil.parse(String.valueOf(v), "yyyy-MM-dd HH:mm:ss"));
                } else {
                    update.set(k, DateUtil.date(Convert.toLong(v)));
                }
            } else {
                update.set(k, v);
            }
        });
        update.set("updateTime", new Date());
        String collectionName = getCollectionNameFromDocumentAnnotedClass(clazz);
//        String apk = getCollectionPK(collectionName);
//        Object application = ReflectUtil.invoke(t, ReflectUtil.getMethod(clazz, "get" + StrUtil.upperFirst(apk)));
        String DBName = getWDBName();
        preOperateProcess(update, false);
        return mongoRepository.upsert(DBName, query, update, collectionName);
    }

    public <T> UpdateResult upsertAgileData(Query query, T t, Class<T> clazz) {
        String json = JSON.toJSONString(t);
        JSONObject jsonObject = JSON.parseObject(json);
        Update update = new Update();
        jsonObject.forEach((k, v) -> {
            if ("createTime".equals(k)) {
                if (v instanceof Long) {
                    update.set(k, DateUtil.date(Convert.toLong(v)));
                } else if (v instanceof String && !NumberUtil.isNumber(String.valueOf(v))) {
                    update.set(k, DateUtil.parse(String.valueOf(v), "yyyy-MM-dd HH:mm:ss"));
                } else {
                    update.set(k, DateUtil.date(Convert.toLong(v)));
                }
            } else {
                update.set(k, v);
            }
        });
        update.set("updateTime", new Date());
        String collectionName = getCollectionNameFromDocumentAnnotedClass(clazz);
//        String apk = getCollectionPK(collectionName);
//        Object application = ReflectUtil.invoke(t, ReflectUtil.getMethod(clazz, "get" + StrUtil.upperFirst(apk)));
        String DBName = getWDBName();
        preOperateProcess(update, false);
        return mongoRepository.upsert(DBName, query, update, collectionName);
    }

    public <T> String getCollectionNameFromDocumentAnnotedClass(Class clazz) {
        Document document = (Document) clazz.getAnnotation(Document.class);
        String collectionName = StringUtils.isEmpty(document.value()) ? document.collection() : document.value();
        collectionName = StringUtils.isEmpty(collectionName) ? clazz.getSimpleName().substring(0, 1).toLowerCase() + clazz.getSimpleName().substring(1) : collectionName;
        return collectionName;
    }

    private <T> String getCollectionNameFromDocumentAnnotedObj(T t) {
        Class clazz = t.getClass();
        return getCollectionNameFromDocumentAnnotedClass(clazz);
    }

    private <T> String getApplication(T t, String collectionName) {
//        Assert.notBlank(collectionName, "Collection name cannot be empty");
//        String apk = getCollectionPK(collectionName);
//        String application;
//        try {
//            application = invokeGetApplication(apk, t);
//        } catch (Exception e) {
//            throw new BusinessException("Failed to obtain application");
//        }
//        return application;
        return "";
    }

    /**
     * 注意用法：记录新增
     */
    public <T> T insert(T t) {
//        String collectionName = getCollectionNameFromDocumentAnnotedObj(t);
//        String application = getApplication(t, collectionName);
        String DBName = getWDBName();
        preOperateProcess(t, true);
        return mongoRepository.insert(DBName, t);
    }
    /**
     * 注意用法：记录新增
     */
    public <T> T insertNotEdit(T t) {
//        String collectionName = getCollectionNameFromDocumentAnnotedObj(t);
//        String application = getApplication(t, collectionName);
        String DBName = getWDBName();
//        preOperateProcess(t, true);
        return mongoRepository.insert(DBName, t);
    }
    /**
     * 注意用法：记录新增
     */
    
    public <T> T insertNotEditAgileData(T t) {
        String collectionName = getCollectionNameFromDocumentAnnotedObj(t);
        String application = getApplication(t, collectionName);
        String DBName = getWDBName();
//        preOperateProcess(t, true);
        return mongoRepository.insert(DBName, t);
    }
    /**
     * 注意用法：记录新增
     */
    public <T> T insert(String collectionName, T t) {
//        String application = getApplication(t, collectionName);
        String DBName = getWDBName();
        preOperateProcess(t, true);
        return mongoRepository.insert(DBName, collectionName, t);
    }

    /**
     * 注意用法：对已有document整体更新
     */
    public <T> T save(T t) {
//        String collectionName = getCollectionNameFromDocumentAnnotedObj(t);
//        String application = getApplication(t, collectionName);
        String DBName = getWDBName();
        preOperateProcess(t, false);
        return mongoRepository.save(DBName, t);
    }
    
    public <T> T saveAgileData(T t) {
        String DBName = getWDBName();
        preOperateProcess(t, false);
        return mongoRepository.save(DBName, t);
    }

    public <T> T saveEntity(T t) {
//        String collectionName = getCollectionNameFromDocumentAnnotedObj(t);
//        String application = getApplication(t, collectionName);
        String DBName = getWDBName();
        return mongoRepository.save(DBName, t);
    }

    /**
     * 注意用法：对已有document整体更新
     */
    
    public <T> T saveNotSetEditAgileData(T t) {
        String collectionName = getCollectionNameFromDocumentAnnotedObj(t);
        String application = getApplication(t, collectionName);
        String DBName = getWDBName();
        return mongoRepository.save(DBName, t);
    }

    /**
     * 注意用法：对已有document整体更新
     */
    public <T> T saveNotSetEdit(T t) {
//        String collectionName = getCollectionNameFromDocumentAnnotedObj(t);
//        String application = getApplication(t, collectionName);
        String DBName = getWDBName();
        return mongoRepository.save(DBName, t);
    }
    /**
     * 注意用法：对已有document整体更新
     */
    public <T> T save(String collectionName, T t) {
//        String application = getApplication(t, collectionName);
        String DBName = getWDBName();
        preOperateProcess(t, false);
        return mongoRepository.save(DBName, collectionName, t);
    }

    /**
     * 注意用法：对已有document整体更新
     */
    public <T> T save(String collectionName, T t, String application) {
        String DBName = getWDBName();
        preOperateProcess(t, false);
        return mongoRepository.save(DBName, collectionName, t);
    }

    public <T> void insertAll(Class<T> tClass, List<T> tList) {
/*        String collectionName = getCollectionNameFromDocumentAnnotedClass(tClass);
        String application = getApplication(tList.get(0), collectionName);*/
        String DBName = getWDBName();
        tList.forEach(t -> preOperateProcess(t, true));
        mongoRepository.insertAll(DBName, tClass, tList);
    }

    public <T> void insertAll(String collectionName, List<T> tList) {
//        String application = getApplication(tList.get(0), collectionName);
        String DBName = getWDBName();
        tList.forEach(t -> preOperateProcess(t, true));
        mongoRepository.insertAll(DBName, collectionName, tList);
    }

    public <T> DeleteResult delete(T obj) {
/*        String collectionName = getCollectionNameFromDocumentAnnotedObj(obj);
        String application = getApplication(obj, collectionName);*/
        String DBName = getWDBName();
        return mongoRepository.delete(DBName, obj);
    }

    public <T> DeleteResult delete(T obj,String collectionName) {
        String DBName = getWDBName();
        return mongoRepository.delete(DBName, obj,collectionName);
    }

    public <T> long remove(Query query, String collectionName, Class<T> tClass) {
/*        List<T> tList = find(query, tClass);
        if (CollectionUtils.isEmpty(tList)) {
            return 0L;
        }
        String application = getApplication(tList.get(0), collectionName);*/
        String DBName = getWDBName();
        return mongoRepository.remove(DBName, tClass, query);
    }

    public <T> DeleteResult delete(Query query, Class<T> tClass) {
/*        String collectionName = getCollectionNameFromDocumentAnnotedClass(tClass);
        List<T> tList = find(query, tClass);
        if (CollectionUtils.isEmpty(tList)) {
            return new DeleteResult() {
                @Override
                public boolean wasAcknowledged() {
                    return false;
                }

                @Override
                public long getDeletedCount() {
                    return 0;
                }
            };
        }
        String application = getApplication(tList.get(0), collectionName);*/
        String DBName = getWDBName();
        return mongoRepository.delete(DBName, tClass, query);
    }
    
    public <T> DeleteResult deleteAgileData(Query query, Class<T> tClass) {
        String collectionName = getCollectionNameFromDocumentAnnotedClass(tClass);
        List<T> tList = find(query, tClass);
        if (CollectionUtils.isEmpty(tList)) {
            return new DeleteResult() {
                @Override
                public boolean wasAcknowledged() {
                    return false;
                }

                @Override
                public long getDeletedCount() {
                    return 0;
                }
            };
        }
        String application = getApplication(tList.get(0), collectionName);
        String DBName = getWDBName();
        return mongoRepository.delete(DBName, tClass, query);
    }
    public <T> long remove(Query query, Class<T> tClass) {
//        List<T> tList = find(query, tClass);
//        String collectionName = getCollectionNameFromDocumentAnnotedClass(tClass);
//        if (CollectionUtils.isEmpty(tList)) {
//            return 0L;
//        }
//        String application = getApplication(tList.get(0), collectionName);
        String DBName = getWDBName();
        return mongoRepository.remove(DBName, tClass, query);
    }
    
    public <T> long removeAgileData(Query query, Class<T> tClass) {
        String DBName = getWDBName();
        return mongoRepository.remove(DBName, tClass, query);
    }
    public <T> long remove(Query query, String application, String collectionName) {
        String DBName = getWDBName();
        return mongoRepository.remove(DBName, collectionName, query);
    }

    public <T> UpdateResult updateFirst(Query query, Update update, Class<T> tClass) {
        String collectionName = getCollectionNameFromDocumentAnnotedClass(tClass);
        Assert.notBlank(collectionName, "Collection name cannot be empty");
//        String apk = getCollectionPK(collectionName);
//        String application = (String) query.getQueryObject().get(apk);
//        String DBName = getWDBName("");
//        JSONObject exist = findOne(query, collectionName, JSONObject.class);
        String DBName = getDBName();
        preOperateProcess(update, false);
        return mongoRepository.updateFirst(DBName, tClass, query, update);
    }
    
    public <T> UpdateResult updateAgileFirst(Query query, Update update, Class<T> tClass) {
        String collectionName = getCollectionNameFromDocumentAnnotedClass(tClass);
        Assert.notBlank(collectionName, "Collection name cannot be empty");
//        String apk = getCollectionPK(collectionName);
//        String application = (String) query.getQueryObject().get(apk);
//        String DBName = getWDBName("");
//        JSONObject exist = findOne(query, collectionName, JSONObject.class);
        String DBName = getDBName();
        preOperateProcess(update, false);
        return mongoRepository.updateFirst(DBName, tClass, query, update);
    }

    public <T> UpdateResult updateFirst(Query query, Update update, String collectionName) {
        String DBName = getDBName();
        preOperateProcess(update, false);
        return mongoRepository.updateFirst(DBName, collectionName, query, update);
    }

    public <T> List<T> find(Query query, Class<T> tClass) {
        String DBName = getDBName();
        return mongoRepository.find(DBName, tClass, query);
    }

    // TODO pzz 测试是否支持投影
    public <T, C> List<T> find(Query query, Class<C> collClass, Class<T> dtoClass) {
        String DBName = getDBName();
        String collection = getCollectionNameFromDocumentAnnotedClass(collClass);
        return mongoRepository.find(DBName, collection, query, dtoClass);
    }

    public <T> List<T> find(String collectionName, Query query, Class<T> tClass) {
        String DBName = getDBName();
        return mongoRepository.find(DBName, collectionName, query, tClass);
    }

    public <T> List<T> findAll(Class<T> tClass) {
        String DBName = getDBName();
        return mongoRepository.findAll(DBName, tClass);
    }

    public <T> T findOne(Query query, Class<T> tClass) {
        String DBName = getDBName();
        return mongoRepository.findOne(DBName, tClass, query);
    }

    public <T> T findOne(Query query, String collectionName, Class<T> tClass) {
        String DBName = getDBName();
        return mongoRepository.findOne(DBName, tClass, collectionName, query);
    }

    public long count(Query query, @Nullable Class<?> entityClass) {
        String DBName = getDBName();
        return mongoRepository.count(DBName, query, entityClass);
    }

    public <T> Page<T> findWithPage(Pageable pageable, Query query, Class<T> tClass) {
        String DBName = getDBName();
        return mongoRepository.findWithPage(DBName, pageable, tClass, query);

    }

    public <C> Pagination<C> findWithPage(Query query, Class<C> tClass, PageReqCondition<?> page) {
        // 查询总数
        long count = count(query, tClass);

        // 设置分页和排序
        query.with(page.toPageable());
        if (!query.isSorted()) {
            query.with(Sort.by(Sort.Direction.DESC, "_id"));
        }

        // 查询数据
        List<C> entities = find(query, tClass);
        return Pagination.buildPagination(page, entities, count);
    }

    // 指定数据库名称进行查询
    public <T> List<T> findByBranch(String branchName, Query query, Class<T> tClass) {
        String simpleName = mongodbConfig.getBranchDBNameMap().get(branchName);
        if (StringUtils.isBlank(simpleName)) {
            log.error("branch 不存在:{}", branchName);
            throw new BusinessException("Branch does not exist");
        }
        return mongoRepository.find(simpleName, tClass, query);
    }

    public <T> List<T> findByDBName(String DBName, Query query, Class<T> tClass) {
        return mongoRepository.find(DBName, tClass, query);
    }

    public <T> T saveByDBName(String DBName, T t) {
        preOperateProcess(t, false);
        return mongoRepository.save(DBName, t);
    }

    public <T> long removeByDBName(String DBName, Query query, Class<T> tClass) {
        List<T> tList = findByDBName(DBName, query, tClass);
        if (CollectionUtils.isEmpty(tList)) {
            return 0L;
        }
        return mongoRepository.remove(DBName, tClass, query);
    }

    // 在应用的所有分支里按指定数据库表名删除数据
    public void removeByCollection(List<String> branchNames, String collection, Query query) {
        branchNames.forEach(branchName -> {
//            String branchName = appBranchPO.getName();
            // 查询应用分支对应的数据库
            String simpleName = mongodbConfig.getBranchDBNameMap().get(branchName);
            if (StringUtils.isBlank(simpleName)) {
                log.error("branch 不存在:{}", branchName);
                throw new BusinessException("Branch does not exist");
            }
            // 指定数据库删除数据
            mongoRepository.remove(simpleName, collection, query);
        });
    }

    // 在指定分支名称里按指定数据库表名删除数据
    public void removeByBranch(String branchName, String collection, Query query) {
        // 查询应用分支对应的数据库
        String simpleName = mongodbConfig.getBranchDBNameMap().get(branchName);
        if (StringUtils.isBlank(simpleName)) {
            log.error("branch 不存在:{}", branchName);
            throw new BusinessException("Branch does not exist");
        }
        // 指定数据库删除数据
        mongoRepository.remove(simpleName, collection, query);
    }


    private <T> void preOperateProcess(T obj, boolean isInsert) {
        /*if (Objects.isNull(obj)
                || (!(obj instanceof MongoBaseDto) && !(obj instanceof Update) && !(obj instanceof JSONObject) && !(obj instanceof ComponentStructuredList))) {
            return;
        }
        // 当前时间
        Date currentDate = new Date();
        // 当前用户id
        String userName = "System";
        try {
            if(Objects.nonNull(UserHelper.getCurrentUser())){
                userName = UserHelper.getCurrentUser().getName();
            }
        } catch (Exception e) {
            log.error("preOperateProcess fetch userId e:{}", e);
        }
        if (obj instanceof Update) {
            ((Update) obj).set("editBy", userName);
            ((Update) obj).set("editDate", currentDate);
            if (isInsert) {
                ((Update) obj).set("createBy", userName);
                ((Update) obj).set("createDate", currentDate);
            }
            org.bson.Document updateDocument = ((Update) obj).getUpdateObject();
            if (updateDocument.containsKey("$set")) {
                org.bson.Document document = (org.bson.Document) updateDocument.get("$set");
                if (document.containsKey("createDate")
                        && Objects.nonNull(document.get("createDate")) && document.get("createDate") instanceof Long) {
                    document.put("createDate", new Date(Long.valueOf(document.get("createDate").toString())));
                }
            }
        } else if (obj instanceof JSONObject) {
            ((JSONObject) obj).put("editBy", userName);
            ((JSONObject) obj).put("editDate", currentDate);
            if (isInsert) {
                ((JSONObject) obj).put("createBy", userName);
                ((JSONObject) obj).put("createDate", currentDate);
            }
        } else if (obj instanceof ComponentStructuredList) {
            ((ComponentStructuredList) obj).setEditBy(userName);
            ((ComponentStructuredList) obj).setEditDate(currentDate);
            // 数据新增
            if (isInsert) {
                ((ComponentStructuredList) obj).setCreateBy(userName);
                ((ComponentStructuredList) obj).setCreateDate(currentDate);
            }
        } else {
            ((MongoBaseDto) obj).setEditBy(userName);
            ((MongoBaseDto) obj).setEditDate(currentDate);
            // 数据新增
            if (isInsert) {
                ((MongoBaseDto) obj).setCreateBy(userName);
                ((MongoBaseDto) obj).setCreateDate(currentDate);
            }
        }*/
    }

    private static final String TARGET_PACKAGE = "com.digiwin.athena.mongodb.domain.";

    private String generateRandomClassName(int length) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < length; i++) {
            char randomChar;
            if (i == 0) {
                // 首字母大写
                randomChar = (char) ('A' + random.nextInt(26));
            } else {
                // 其他字母小写
                randomChar = (char) ('a' + random.nextInt(26));
            }
            sb.append(randomChar);
        }
        return sb.toString();
    }

    private <T> String getTargetCollection(T obj) {
        String collectionName = null;
        Class<?> objClass = obj.getClass();
        java.lang.annotation.Annotation[] annotations = objClass.getAnnotations();
        if (annotations.length > 0) {
            for (java.lang.annotation.Annotation annotation : annotations) {
                if (annotation instanceof Document) {
                    Document document = (Document) annotation;
                    collectionName = document.value();
                    break;
                }
            }
        }
        if (StringUtils.isEmpty(collectionName)) {
            String name = objClass.getName();
            if (name.contains(".")) {
                String[] parts = name.split("\\.");
                collectionName = firstLetterToLowerCase(parts[parts.length - 1]);
            } else {
                collectionName = firstLetterToLowerCase(name);
            }
        }
        return collectionName;
    }

    private String firstLetterToLowerCase(String input) {
        if (StringUtils.isEmpty(input)) {
            return input;
        }
        return input.substring(0, 1).toLowerCase() + input.substring(1);
    }

    public <T> T findById(Object id, Class<T> entityClass) {
        String DBName = getDBName();
        return mongoRepository.findById(DBName, id, entityClass);
    }

    public <T> Collection<T> insertAll(Collection<? extends T> collection) {
        String DBName = getDBName();
        return mongoRepository.insertAll(DBName, collection);
    }
    
    public <T> Collection<T> insertAllAgileData(Collection<? extends T> collection) {
        String DBName = getDBName();
        return mongoRepository.insertAll(DBName, collection);
    }
    public <T> T findOne(Query query, Class<T> entityClass, String collectionName) {
        String DBName = getDBName();
        return mongoRepository.findOne(DBName, query, entityClass, collectionName);
    }

    public <T> List<T> find(Query query, Class<T> entityClass, String collectionName) {
        String DBName = getDBName();
        return mongoRepository.find(DBName, query, entityClass, collectionName);
    }

    public DeleteResult remove(Query query, String collectionName) {
        String DBName = getDBName();
        return mongoRepository.remove(DBName, query, collectionName);
    }

    public <T> T save(T objectToSave, String collectionName) {
        String DBName = getDBName();
        return mongoRepository.save(DBName, objectToSave, collectionName);
    }

    public long count(Query query, @Nullable Class<?> entityClass, String collectionName) {
        String DBName = getDBName();
        return mongoRepository.count(DBName, query, entityClass, collectionName);
    }

    public <T> Collection<T> insert(Collection<? extends T> batchToSave, Class<?> entityClass) {
        String DBName = getDBName();
        return mongoRepository.insert(DBName, batchToSave, entityClass);
    }

    public <T> T insert(T objectToSave, String collectionName) {
        String DBName = getDBName();
        return mongoRepository.insert(DBName, objectToSave, collectionName);
    }

    public MongoDatabaseFactory getMongoDbFactory() {
        String DBName = getDBName();
        return mongoRepository.getMongoDbFactory(DBName);
    }

    public <T> List<T> findAllAndRemove(Query query, Class<T> entityClass, String collectionName) {
        String DBName = getDBName();
        return mongoRepository.findAllAndRemove(DBName, query, entityClass, collectionName);
    }

    public boolean exists(Query query, @Nullable Class<?> entityClass, String collectionName) {
        String DBName = getDBName();
        return mongoRepository.exists(DBName, query, entityClass, collectionName);
    }

    public <O> AggregationResults<O> aggregate(Aggregation aggregation, Class<?> inputType, Class<O> outputType) {
        String DBName = getDBName();
        return mongoRepository.aggregate(DBName, aggregation, inputType, outputType);
    }

    public <O> AggregationResults<O> aggregate(TypedAggregation<?> aggregation, Class<O> outputType) {
        String DBName = getDBName();
        return mongoRepository.aggregate(DBName, aggregation, outputType);
    }

    public <O> AggregationResults<O> aggregate(Aggregation aggregation, String collectionName, Class<O> outputType) {
        String DBName = getDBName();
        return mongoRepository.aggregate(DBName, aggregation, collectionName, outputType);
    }

    public BulkOperations bulkOps(BulkOperations.BulkMode mode, Class<?> entityType) {
        String DBName = getDBName();
        return mongoRepository.bulkOps(DBName,mode,entityType);
    }

    public Set<String> getCollectionNames() {
        String dBName = getDBName();
        return mongoRepository.getCollectionNames(dBName);
    }

    public <T> T insertWithOutEdit(T t) {
        String DBName = getWDBName();
        return mongoRepository.insert(DBName, t);
    }

    public MongoTemplate getTargetMongoTemplate(){
        String dBName = getDBName();
        return mongoRepository.getTargetMongoTemplate(dBName);
    }
}
