package com.digiwin.athena.dao;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.digiwin.athena.domain.StandardDictionaryPO;
import com.digiwin.athena.mongodb.repository.MongoSystemRepositoryDecorator;
import jakarta.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Repository;

import java.util.ArrayList;
import java.util.List;

@Slf4j
@Repository
public class StandardDictionaryMongoDao {

    @Resource
    private MongoSystemRepositoryDecorator mongoSystemRepositoryDecorator;

    private final MongoTemplate mongoTemplate;

    @Autowired
    public StandardDictionaryMongoDao(MongoTemplate mongoTemplate) {
        this.mongoTemplate = mongoTemplate;
    }

    public void insert(List<StandardDictionaryPO> standardDictionaryPOList) {
        mongoSystemRepositoryDecorator.insertAll(standardDictionaryPOList);
    }

    public void update(StandardDictionaryPO standardDictionaryPO) {
        mongoSystemRepositoryDecorator.save(standardDictionaryPO);
    }

    public List<StandardDictionaryPO> selectByCode(List<String> codes) {
        Criteria criteria = Criteria.where("code").in(codes);
        Query query = new Query(criteria);
        return mongoSystemRepositoryDecorator.find(query, StandardDictionaryPO.class);
    }

    public List<StandardDictionaryPO> selectAll() {
        return mongoSystemRepositoryDecorator.findAll(StandardDictionaryPO.class);
    }

    public List<String> selectAllTenantId() {
        return mongoSystemRepositoryDecorator.findAll(StandardDictionaryPO.class)
                .stream()
                .map(StandardDictionaryPO::getTenantId)
                .toList();
    }

    public Page<StandardDictionaryPO> queryModelDrivenInnerISV(
            Page<StandardDictionaryPO> page,
            List<String> teamIds,
            String auditStatus,
            String condition) {
        Criteria criteria = buildMatchCriteria(teamIds, auditStatus, condition);
        Query query = new Query(criteria);
        query.with(Sort.by(Sort.Direction.DESC,"editTime"));
        long count = mongoSystemRepositoryDecorator.count(query, StandardDictionaryPO.class);
        page.setTotal(count);

        query.limit(Long.valueOf(page.getSize()).intValue());
        query.skip((page.getCurrent() - 1) * page.getSize());
        List<StandardDictionaryPO> standardDictionaryPOS = mongoSystemRepositoryDecorator.find(query, StandardDictionaryPO.class);
        page.setRecords(standardDictionaryPOS);

        return page;
    }

    private Criteria buildMatchCriteria(
            List<String> teamIds,
            String auditStatus,
            String condition) {
        Criteria criteria = Criteria.where("teamId").in(teamIds);

        criteria = criteria.and(
                String.valueOf(Criteria.where("standard").is("N")
                        .orOperator(
                                Criteria.where("standard").is("Y")
                                        .and("auditStatus").is(auditStatus)
                        ))
        );

        if (condition != null &&!"".equals(condition)) {
            Criteria keyCriteria = Criteria.where("key").regex(".*" + condition + ".*");
            Criteria descCriteria = Criteria.where("description").regex(".*" + condition + ".*");
            Criteria valuesCriteria = Criteria.where("values").regex(".*" + condition + ".*");
            criteria.orOperator(keyCriteria, descCriteria, valuesCriteria);
        }

        return criteria;
    }

    public Page<StandardDictionaryPO> queryModelDrivenOuterISV(
            Page<StandardDictionaryPO> page,
            String tenantId,
            List<String> teamIds,
            String auditStatus,
            String condition) {
        Criteria criteria = buildMatchCriteria(tenantId, teamIds, auditStatus, condition);
        Query query = new Query(criteria);
        query.with(Sort.by(Sort.Direction.DESC,"editTime"));
        long count = mongoSystemRepositoryDecorator.count(query, StandardDictionaryPO.class);
        page.setTotal(count);

        query.limit(Long.valueOf(page.getSize()).intValue());
        query.skip((page.getCurrent() - 1) * page.getSize());
        List<StandardDictionaryPO> standardDictionaryPOS = mongoSystemRepositoryDecorator.find(query, StandardDictionaryPO.class);
        page.setRecords(standardDictionaryPOS);

        return page;
    }

    private Criteria buildMatchCriteria(
            String tenantId,
            List<String> teamIds,
            String auditStatus,
            String condition) {
        Criteria criteria = new Criteria();


        Criteria criteria1 = new Criteria();
        Criteria criteria2 = Criteria.where("standard").is("N").orOperator(Criteria.where("standard").is("Y").and("audit_status").is(auditStatus));
        criteria1.where("tenant_id").is(tenantId).andOperator(criteria2);

        Criteria criteria3 = Criteria.where("standard").is("Y").and("audit_status").is(auditStatus).and("team_id").in(teamIds);

        Criteria criteria4 = new Criteria();
        criteria4.orOperator(criteria1,criteria3);

        if (condition != null &&!"".equals(condition)) {
            Criteria criteria5 = new Criteria();
            Criteria keyCriteria = Criteria.where("key").regex(".*" + condition + ".*");
            Criteria descCriteria = Criteria.where("description").regex(".*" + condition + ".*");
            Criteria valuesCriteria = Criteria.where("values").regex(".*" + condition + ".*");
            criteria5.orOperator(keyCriteria, descCriteria, valuesCriteria);

            criteria.andOperator(criteria4,criteria5);
        }else{
            criteria.andOperator(criteria4);
        }

        return criteria;
    }

    public Page<StandardDictionaryPO> queryInnerISV(
            Page<StandardDictionaryPO> page,
            List<String> teamIds,
            String auditStatus,
            String condition,
            String standard) {
        Criteria criteria = buildMatchCriteria(teamIds, auditStatus, condition, standard);
        Query query = new Query(criteria);
        query.with(Sort.by(Sort.Direction.DESC,"editTime"));
        long count = mongoSystemRepositoryDecorator.count(query, StandardDictionaryPO.class);
        page.setTotal(count);

        query.limit(Long.valueOf(page.getSize()).intValue());
        query.skip((page.getCurrent() - 1) * page.getSize());
        List<StandardDictionaryPO> standardDictionaryPOS = mongoSystemRepositoryDecorator.find(query, StandardDictionaryPO.class);
        page.setRecords(standardDictionaryPOS);
        return page;
    }

    private Criteria buildMatchCriteria(
            List<String> teamIds,
            String auditStatus,
            String condition,
            String standard) {
        Criteria criteria = Criteria.where("teamId").in(teamIds);

        if (auditStatus != null &&!"".equals(auditStatus)) {
            criteria = criteria.and("auditStatus").is(auditStatus);
        }

        if (standard != null &&!"".equals(standard)) {
            criteria = criteria.and("standard").is(standard);
        }

        if (condition != null &&!"".equals(condition)) {
            Criteria keyCriteria = Criteria.where("key").regex(".*" + condition + ".*");
            Criteria descCriteria = Criteria.where("description").regex(".*" + condition + ".*");
            Criteria valuesCriteria = Criteria.where("values").regex(".*" + condition + ".*");
            criteria.orOperator(keyCriteria, descCriteria, valuesCriteria);
        }

        return criteria;
    }

    public Page<StandardDictionaryPO> queryOuterISV(
            Page<StandardDictionaryPO> page,
            String tenantId,
            List<String> teamIds,
            String auditStatus,
            String condition,
            String standard) {
        Criteria criteria = buildMatchCriteria(tenantId, teamIds, auditStatus, condition, standard);
        Query query = new Query(criteria);
        query.with(Sort.by(Sort.Direction.DESC,"editTime"));
        long count = mongoSystemRepositoryDecorator.count(query, StandardDictionaryPO.class);
        page.setTotal(count);

        query.limit(Long.valueOf(page.getSize()).intValue());
        query.skip((page.getCurrent() - 1) * page.getSize());
        List<StandardDictionaryPO> standardDictionaryPOS = mongoSystemRepositoryDecorator.find(query, StandardDictionaryPO.class);
        page.setRecords(standardDictionaryPOS);

        return page;
    }

    private Criteria buildMatchCriteria(
            String tenantId,
            List<String> teamIds,
            String auditStatus,
            String condition,
            String standard) {
        Criteria criteria = new Criteria();

        List<Criteria> orCriteria = new ArrayList<>();
        if (teamIds != null &&!teamIds.isEmpty()) {
            Criteria teamIdCriteria = Criteria.where("teamId").in(teamIds);
            if(StringUtils.isNotEmpty(standard)){
                teamIdCriteria.and("standard").is(standard);
            }

            if(StringUtils.isNotEmpty(auditStatus)){
                teamIdCriteria.and("auditStatus").is(auditStatus);
            }

            orCriteria.add(teamIdCriteria);
//            criteria = criteria.orOperator(teamIdCriteria);
        }

        Criteria tenantIdCriteria = Criteria.where("tenantId").is(tenantId);
        if (StringUtils.isNotEmpty(auditStatus)) {
            tenantIdCriteria = tenantIdCriteria.and("auditStatus").is(auditStatus);
        }
        if (StringUtils.isNotEmpty(standard)) {
            tenantIdCriteria.and("standard").is(standard);
        }
        orCriteria.add(tenantIdCriteria);

        if (condition != null &&!"".equals(condition)) {
            Criteria keyCriteria = Criteria.where("key").regex(".*" + condition + ".*");
            Criteria descCriteria = Criteria.where("description").regex(".*" + condition + ".*");
            Criteria valuesCriteria = Criteria.where("values").regex(".*" + condition + ".*");

            Criteria criteria1 = new Criteria();
            criteria1.orOperator(keyCriteria,descCriteria,valuesCriteria);

            criteria.andOperator(criteria1);
        }
        if(!orCriteria.isEmpty()){
            criteria.orOperator(orCriteria);
        }

        return criteria;
    }

    public List<StandardDictionaryPO> selectByTenantId(String tenantId) {
        Criteria criteria = Criteria.where("tenantId").is(tenantId);
        Query query = new Query(criteria);
        return mongoSystemRepositoryDecorator.find(query, StandardDictionaryPO.class);
    }

    public StandardDictionaryPO selectOneByCondition(Criteria criteria) {
        Query query = new Query(criteria);
        return mongoSystemRepositoryDecorator.findOne(query, StandardDictionaryPO.class);
    }

    public List<StandardDictionaryPO> selectListByCondition(Criteria criteria) {
        Query query = new Query(criteria);
        return mongoSystemRepositoryDecorator.find(query, StandardDictionaryPO.class);
    }

    public IPage<StandardDictionaryPO> selectPageByCondition(IPage<StandardDictionaryPO> page, Criteria criteria) {
        Query query = new Query(criteria);
        return (IPage<StandardDictionaryPO>) mongoSystemRepositoryDecorator.findWithPage(PageRequest.of((int) page.getPages(), (int) page.getSize()), query, StandardDictionaryPO.class);
    }

    public void deleteListByCondition(Criteria criteria) {
        Query query = new Query(criteria);
        mongoSystemRepositoryDecorator.delete(query, StandardDictionaryPO.class);
    }

}
