package com.digiwin.athena.dao;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.digiwin.athena.domain.WordDictionaryManagePO;
import com.digiwin.athena.mongodb.repository.MongoSystemRepositoryDecorator;
import jakarta.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.bson.Document;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.aggregation.Aggregation;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Repository;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Slf4j
@Repository
public class WordDictionaryMongoDao {

    @Resource
    private MongoSystemRepositoryDecorator mongoSystemRepositoryDecorator;

    private final MongoTemplate mongoTemplate;

    @Autowired
    public WordDictionaryMongoDao(MongoTemplate mongoTemplate) {
        this.mongoTemplate = mongoTemplate;
    }

    public void insert(List<WordDictionaryManagePO> standardDictionaryPOList) {
        mongoSystemRepositoryDecorator.insertAll(standardDictionaryPOList);
    }

    public void update(WordDictionaryManagePO standardDictionaryPO) {
        mongoSystemRepositoryDecorator.save(standardDictionaryPO);
    }

    public long count(String id) {
        return mongoSystemRepositoryDecorator.count(new Query(Criteria.where("id").is(id)), WordDictionaryManagePO.class);
    }

    public List<Map<String, Object>> queryByDataName(List<String> dataNames) {
        // 构建聚合管道
        Aggregation aggregation = Aggregation.newAggregation(
                // 1. $match 阶段
                Aggregation.match(Criteria.where("dataName").in(dataNames)),

                // 2. 第一个 $lookup 阶段
                Aggregation.lookup(
                        "tStandardBusinessType",
                        "relateBusinessTypeId",
                        "id",
                        "businessTypeData"
                ),

                // 3. 第一个 $unwind 阶段
                Aggregation.unwind("businessTypeData", true),

                // 4. 第二个 $lookup 阶段
                Aggregation.lookup(
                        "tStandardDictionary",
                        "relateDictionaryId",
                        "id",
                        "dictionaryData"
                ),

                // 5. 第二个 $unwind 阶段
                Aggregation.unwind("dictionaryData", true),

                // 6. $project 阶段（选择并重命名字段）
                Aggregation.project()
                        .and("id").as("id")
                        .and("dataName").as("dataName")
                        .and("descriptionZhTw").as("descriptionZhTw")
                        .and("descriptionZhCn").as("descriptionZhCn")
                        .and("descriptionEnUs").as("descriptionEnUs")
                        .and("typeZhTw").as("typeZhTw")
                        .and("typeZhCn").as("typeZhCn")
                        .and("typeEnUs").as("typeEnUs")
                        .and("unitOrDefaultZhTw").as("unitOrDefaultZhTw")
                        .and("unitOrDefaultZhCn").as("unitOrDefaultZhCn")
                        .and("unitOrDefaultEnUs").as("unitOrDefaultEnUs")
                        .and("tagZhTw").as("tagZhTw")
                        .and("tagZhCn").as("tagZhCn")
                        .and("tagEnUs").as("tagEnUs")
                        .and("remarkZhTw").as("remarkZhTw")
                        .and("remarkZhCn").as("remarkZhCn")
                        .and("remarkEnUs").as("remarkEnUs")
                        .and("approvedStatus").as("approvedStatus")
                        .and("buildAcct").as("buildAcct")
                        .and("buildTime").as("buildTime")
                        .and("approvedAcct").as("approvedAcct")
                        .and("approvedTime").as("approvedTime")
                        .and("lastUpdateTime").as("lastUpdateTime")
                        .and("dirty").as("dirty")
                        .and("relateBusinessTypeId").as("relateBusinessTypeId")
                        .and("relateDictionaryId").as("relateDictionaryId")
                        .and("likenessWord").as("likenessWord")
                        .and("standard").as("standard")
                        .and("manageStatus").as("manageStatus")
                        .and("auditStatus").as("auditStatus")
                        .and("systemId").as("systemId")
                        .and("tenantId").as("tenantId")
                        .and("teamId").as("teamId")
                        .and("editTime").as("editTime")
                        .and("editBy").as("editBy")
                        .and("createTime").as("createTime")
                        .and("createBy").as("createBy") // 包含 t1 的其他字段
                        .and("businessTypeData.code").as("businessCode")
                        .and("businessTypeData.name").as("businessType")
                        .and("businessTypeData.dataType").as("businessDataType")
                        .and("businessTypeData.size").as("businessSize")
                        .and("businessTypeData.fieldPrecision").as("businessPrecision")
                        .and("dictionaryData.dictionaryKey").as("dictionaryKey")
                        .and("dictionaryData.description").as("dictionaryDesc")
                        .and("dictionaryData.dictionaryValues").as("dictionaryValues")
        );

        // 执行聚合查询
        List<Document> documentList = mongoTemplate.aggregate(
                aggregation,
                "tStandardWordDictionary",
                Document.class
        ).getMappedResults();

        List<Map<String, Object>> resultList = new ArrayList<>();
        for (Document document : documentList) {
            Map<String, Object> map = new HashMap<>(document);
            resultList.add(map);
        }
        return resultList;
    }

    public List<Map<String, Object>> selectWordDictionaryPage(Map<String, Object> columnMap) {
        // 构建聚合管道
        Aggregation aggregation = Aggregation.newAggregation(
                // 多表连接部分
                Aggregation.lookup(
                        "tStandardBusinessType",
                        "relateBusinessTypeId",
                        "id",
                        "businessType"
                ),
                Aggregation.lookup(
                        "tStandardDictionary",
                        "relateDictionaryId",
                        "id",
                        "dictionary"
                ),
                // 条件判断部分
                Aggregation.match(buildMatchCriteria(columnMap)),
                // 投影部分，选择需要的字段并重命名
                Aggregation.project()
                        .and("dataName").as("dataName")
                        .and("descriptionZhCn").as("descriptionZhCn"),
                // 其他字段...
                // 排序和分页部分
                Aggregation.sort(Sort.Direction.DESC, "editTime", "id"),
                Aggregation.skip(getSkip(columnMap)),
                Aggregation.limit(getLimit(columnMap))
        );

        // 执行聚合查询
        List<Document> documentList = mongoTemplate.aggregate(
                aggregation,
                "tStandardWordDictionary",
                Document.class
        ).getMappedResults();

        List<Map<String, Object>> resultList = new ArrayList<>();
        for (Document document : documentList) {
            Map<String, Object> map = new HashMap<>(document);
            resultList.add(map);
        }
        return resultList;
    }

    private Criteria buildMatchCriteria(Map<String, Object> columnMap) {
        Criteria criteria = Criteria.where("1").is("1");  // 初始条件，确保条件存在

        if (columnMap.get("sourcePage") != null && "modelDriven".equals(columnMap.get("sourcePage"))) {
            if ((int) columnMap.get("isBM") == 1) {
                criteria = criteria.and("standard").is("Y").and("manageStatus").is("Y");
            } else if ((int) columnMap.get("isBM") == 0) {
                criteria = criteria.and(
                        String.valueOf(Criteria.where("tenantId").is(columnMap.get("tenantId"))
                                .orOperator(Criteria.where("standard").is("Y").and("manageStatus").is("Y")))
                );
            }
        } else {
            if ((int) columnMap.get("hasAuth") == 0 && columnMap.get("standard") != null && !"".equals(columnMap.get("standard"))) {
                criteria = criteria.and(
                        String.valueOf(Criteria.where("tenantId").is(columnMap.get("tenantId"))
                                .orOperator(Criteria.where("standard").is("Y").and("manageStatus").is("Y")))
                ).and("standard").is(columnMap.get("standard"));
            } else if ((int) columnMap.get("hasAuth") == 0 && (columnMap.get("standard") == null || "".equals(columnMap.get("standard")))) {
                criteria = criteria.and(
                        String.valueOf(Criteria.where("tenantId").is(columnMap.get("tenantId"))
                                .orOperator(Criteria.where("standard").is("Y").and("manageStatus").is("Y")))
                );
            } else if ((int) columnMap.get("hasAuth") == 1 && columnMap.get("standard") != null && !"".equals(columnMap.get("standard"))) {
                criteria = criteria.and("standard").is(columnMap.get("standard"));
            }
        }

        if (columnMap.get("keyword") != null && !"".equals(columnMap.get("keyword"))) {
            criteria = criteria.and(
                    String.valueOf(Criteria.where("dataName").regex(".*" + columnMap.get("keyword") + ".*")
                            .orOperator(
                                    Criteria.where("descriptionZhCn").regex(".*" + columnMap.get("keyword") + ".*"),
                                    Criteria.where("descriptionEnUs").regex(".*" + columnMap.get("keyword") + ".*"),
                                    Criteria.where("descriptionZhTw").regex(".*" + columnMap.get("keyword") + ".*"),
                                    Criteria.where("likenessWord").regex(".*" + columnMap.get("keyword") + ".*")
                            ))
            );
        }

        return criteria;
    }

    private long getSkip(Map<String, Object> columnMap) {
        Integer pageNum = (Integer) columnMap.get("pageNum");
        Integer pageSize = (Integer) columnMap.get("pageSize");
        if (pageNum != null && pageSize != null) {
            return (pageNum - 1) * pageSize;
        }
        return 0;
    }

    private long getLimit(Map<String, Object> columnMap) {
        Integer pageSize = (Integer) columnMap.get("pageSize");
        if (pageSize != null) {
            return pageSize;
        }
        return 0;
    }

    public Integer selectCountNum(Map<String, Object> columnMap) {
        Aggregation aggregation = Aggregation.newAggregation(
                Aggregation.match(buildMatchCriteria1(columnMap)),
                Aggregation.count().as("count")
        );

        List<Document> documentList = mongoTemplate.aggregate(
                aggregation,
                "tStandardWordDictionary",
                Document.class
        ).getMappedResults();

        List<Map<String, Object>> resultList = new ArrayList<>();
        for (Document document : documentList) {
            Map<String, Object> map = new HashMap<>(document);
            resultList.add(map);
        }
        if (resultList.isEmpty()) {
            return 0;
        }
        return ((Number) resultList.get(0).get("count")).intValue();
    }

    private Criteria buildMatchCriteria1(Map<String, Object> columnMap) {
        Criteria criteria = Criteria.where("1").is("1");  // 初始条件，确保条件存在

        if (columnMap.get("sourcePage") != null && "modelDriven".equals(columnMap.get("sourcePage"))) {
            if ((int) columnMap.get("hasAuth") == 1) {
                criteria = criteria.and("standard").is("Y").and("manageStatus").is("Y");
            } else if ((int) columnMap.get("hasAuth") == 0) {
                criteria = criteria.and(
                        String.valueOf(Criteria.where("tenantId").is(columnMap.get("tenantId"))
                                .orOperator(Criteria.where("standard").is("Y").and("manageStatus").is("Y")))
                );
            }
        } else {
            if ((int) columnMap.get("hasAuth") == 0 && columnMap.get("standard") != null && !"".equals(columnMap.get("standard"))) {
                criteria = criteria.and(
                        String.valueOf(Criteria.where("tenantId").is(columnMap.get("tenantId"))
                                .orOperator(Criteria.where("standard").is("Y").and("manageStatus").is("Y")))
                ).and("standard").is(columnMap.get("standard"));
            } else if ((int) columnMap.get("hasAuth") == 0 && (columnMap.get("standard") == null || "".equals(columnMap.get("standard")))) {
                criteria = criteria.and(
                        String.valueOf(Criteria.where("tenantId").is(columnMap.get("tenantId"))
                                .orOperator(Criteria.where("standard").is("Y").and("manageStatus").is("Y")))
                );
            } else if ((int) columnMap.get("hasAuth") == 1 && columnMap.get("standard") != null && !"".equals(columnMap.get("standard"))) {
                criteria = criteria.and("standard").is(columnMap.get("standard"));
            }
        }

        if (columnMap.get("keyword") != null && !"".equals(columnMap.get("keyword"))) {
            criteria = criteria.and(
                    String.valueOf(Criteria.where("dataName").regex(".*" + columnMap.get("keyword") + ".*")
                            .orOperator(
                                    Criteria.where("descriptionEnUs").regex(".*" + columnMap.get("keyword") + ".*"),
                                    Criteria.where("descriptionZhCn").regex(".*" + columnMap.get("keyword") + ".*"),
                                    Criteria.where("descriptionZhTw").regex(".*" + columnMap.get("keyword") + ".*"),
                                    Criteria.where("likenessWord").regex(".*" + columnMap.get("keyword") + ".*")
                            ))
            );
        }

        return criteria;
    }

    public List<Map<String, Object>> queryModelDrivenInnerISV(
            Page<WordDictionaryManagePO> page,
            List<String> teamIds,
            String auditStatus,
            String keyWord) {
        Aggregation aggregation = Aggregation.newAggregation(
                // 多表连接
                Aggregation.lookup(
                        "tStandardBusinessType",
                        "relateBusinessTypeId",
                        "id",
                        "businessType"
                ),
                Aggregation.lookup(
                        "tStandardDictionary",
                        "relateDictionaryId",
                        "id",
                        "dictionary"
                ),
                // 构建查询条件
                Aggregation.match(buildMatchCriteria(teamIds, auditStatus, keyWord)),
                // 投影，选择需要的字段并重命名
                Aggregation.project()
                        .and("dataName").as("dataName")
                        .and("descriptionZhTw").as("descriptionZhTw")
                        .and("descriptionZhCn").as("descriptionZhCn")
                        .and("descriptionEnUs").as("descriptionEnUs")
                        .and("relateBusinessTypeId").as("relateBusinessTypeId")
                        .and("relateDictionaryId").as("relateDictionaryId")
                        .and("editBy").as("editBy")
                        .and("editTime").as("editTime")
                        .and("createTime").as("createTime")
                        .and("createBy").as("createBy")
                        .and("tenantId").as("tenantId")
                        .and("manageStatus").as("manageStatus")
                        .and("likenessWord").as("likenessWord")
                        .and("id").as("id")
                        .and("standard").as("standard")
                        .and("teamId").as("teamId")
                        .and("auditStatus").as("auditStatus")
                        .and("businessType.code").as("bizCode")
                        .and("businessType.name").as("bizName")
                        .and("businessType.dataType").as("dataType")
                        .and("businessType.size").as("size")
                        .and("businessType.fieldPrecision").as("fieldPrecision")
                        .and("dictionary.dictionaryKey").as("dictionaryKey")
                        .and("dictionary.dictionaryValues").as("dictionaryValues"),
        // 排序
        Aggregation.sort(Sort.Direction.DESC, "editTime"),
                // 分页
                Aggregation.skip((page.getPages() - 1) * page.getSize()),
                Aggregation.limit(page.getSize())
        );

        List<Document> documentList = mongoTemplate.aggregate(
                aggregation,
                "tStandardWordDictionary",
                Document.class
        ).getMappedResults();

        List<Map<String, Object>> resultList = new ArrayList<>();
        for (Document document : documentList) {
            Map<String, Object> map = new HashMap<>(document);
            resultList.add(map);
        }
        return resultList;
    }

    private Criteria buildMatchCriteria(List<String> teamIds, String auditStatus, String keyWord) {
        Criteria criteria = Criteria.where("teamId").in(teamIds);

        criteria = criteria.and(
                String.valueOf(Criteria.where("standard").is("N")
                        .orOperator(
                                Criteria.where("standard").is("Y")
                                        .and("manageStatus").is("Y")
                                        .and("auditStatus").is(auditStatus)
                        ))
        );

        if (keyWord != null &&!keyWord.isEmpty()) {
            criteria = criteria.and(
                    String.valueOf(Criteria.where("dataName").regex(".*" + keyWord + ".*")
                            .orOperator(
                                    Criteria.where("descriptionEnUs").regex(".*" + keyWord + ".*"),
                                    Criteria.where("descriptionZhCn").regex(".*" + keyWord + ".*"),
                                    Criteria.where("descriptionZhTw").regex(".*" + keyWord + ".*"),
                                    Criteria.where("likenessWord").regex(".*" + keyWord + ".*")
                            ))
            );
        }

        return criteria;
    }

    public List<Map<String, Object>> queryModelDrivenOuterISV(
            Page<WordDictionaryManagePO> page,
            String tenantId,
            List<String> teamIds,
            String auditStatus,
            String keyWord) {
        Aggregation aggregation = Aggregation.newAggregation(
                // 多表连接
                Aggregation.lookup(
                        "tStandardBusinessType",
                        "relateBusinessTypeId",
                        "id",
                        "businessType"
                ),
                Aggregation.lookup(
                        "tStandardDictionary",
                        "relateDictionaryId",
                        "id",
                        "dictionary"
                ),
                // 构建查询条件
                Aggregation.match(buildMatchCriteria(tenantId, teamIds, auditStatus, keyWord)),
                // 投影，选择需要的字段并重命名
                Aggregation.project()
                        .and("dataName").as("dataName")
                        .and("descriptionZhTw").as("descriptionZhTw")
                        .and("descriptionZhCn").as("descriptionZhCn")
                        .and("descriptionEnUs").as("descriptionEnUs")
                        .and("relateBusinessTypeId").as("relateBusinessTypeId")
                        .and("relateDictionaryId").as("relateDictionaryId")
                        .and("editBy").as("editBy")
                        .and("editTime").as("editTime")
                        .and("createTime").as("createTime")
                        .and("createBy").as("createBy")
                        .and("tenantId").as("tenantId")
                        .and("manageStatus").as("manageStatus")
                        .and("likenessWord").as("likenessWord")
                        .and("id").as("id")
                        .and("standard").as("standard")
                        .and("teamId").as("teamId")
                        .and("auditStatus").as("auditStatus")
                        .and("businessType.code").as("bizCode")
                        .and("businessType.name").as("bizName")
                        .and("businessType.dataType").as("dataType")
                        .and("businessType.size").as("size")
                        .and("businessType.fieldPrecision").as("fieldPrecision")
                        .and("dictionary.dictionaryKey").as("dictionaryKey")
                        .and("dictionary.dictionaryValues").as("dictionaryValues"),
        // 排序
        Aggregation.sort(Sort.Direction.DESC, "editTime"),
                // 分页
                Aggregation.skip((page.getPages() - 1) * page.getSize()),
                Aggregation.limit(page.getSize())
        );

        List<Document> documentList = mongoTemplate.aggregate(
                aggregation,
                "tStandardWordDictionary",
                Document.class
        ).getMappedResults();

        List<Map<String, Object>> resultList = new ArrayList<>();
        for (Document document : documentList) {
            Map<String, Object> map = new HashMap<>(document);
            resultList.add(map);
        }
        return resultList;
    }

    private Criteria buildMatchCriteria(
            String tenantId,
            List<String> teamIds,
            String auditStatus,
            String keyWord) {
        Criteria criteria = Criteria.where("tenantId").is(tenantId).and(
                String.valueOf(Criteria.where("standard").is("N")
                        .orOperator(
                                Criteria.where("standard").is("Y")
                                        .and("manageStatus").is("Y")
                                        .and("auditStatus").is(auditStatus)
                        ))
        ).orOperator(
                Criteria.where("teamId").in(teamIds)
                        .and("manageStatus").is("Y")
                        .and("standard").is("Y")
                        .and("auditStatus").is(auditStatus)
        );

        if (keyWord != null &&!keyWord.isEmpty()) {
            criteria = criteria.and(
                    String.valueOf(Criteria.where("dataName").regex(".*" + keyWord + ".*")
                            .orOperator(
                                    Criteria.where("descriptionEnUs").regex(".*" + keyWord + ".*"),
                                    Criteria.where("descriptionZhCn").regex(".*" + keyWord + ".*"),
                                    Criteria.where("descriptionZhTw").regex(".*" + keyWord + ".*"),
                                    Criteria.where("likenessWord").regex(".*" + keyWord + ".*")
                            ))
            );
        }

        return criteria;
    }

    public List<Map<String, Object>> queryInnerISV(
            Page<WordDictionaryManagePO> page,
            List<String> teamIds,
            String auditStatus,
            String keyWord,
            String standard) {
        Aggregation aggregation = Aggregation.newAggregation(
                // 多表连接
                Aggregation.lookup(
                        "tStandardBusinessType",
                        "relateBusinessTypeId",
                        "id",
                        "businessType"
                ),
                Aggregation.lookup(
                        "tStandardDictionary",
                        "relateDictionaryId",
                        "id",
                        "dictionary"
                ),
                // 构建查询条件
                Aggregation.match(buildMatchCriteria(teamIds, auditStatus, keyWord, standard)),
                // 投影，选择需要的字段并重命名
                Aggregation.project()
                        .and("dataName").as("dataName")
                        .and("descriptionZhTw").as("descriptionZhTw")
                        .and("descriptionZhCn").as("descriptionZhCn")
                        .and("descriptionEnUs").as("descriptionEnUs")
                        .and("relateBusinessTypeId").as("relateBusinessTypeId")
                        .and("relateDictionaryId").as("relateDictionaryId")
                        .and("editBy").as("editBy")
                        .and("editTime").as("editTime")
                        .and("createTime").as("createTime")
                        .and("createBy").as("createBy")
                        .and("tenantId").as("tenantId")
                        .and("manageStatus").as("manageStatus")
                        .and("likenessWord").as("likenessWord")
                        .and("id").as("id")
                        .and("standard").as("standard")
                        .and("teamId").as("teamId")
                        .and("auditStatus").as("auditStatus")
                        .and("businessType.code").as("bizCode")
                        .and("businessType.name").as("bizName")
                        .and("businessType.dataType").as("dataType")
                        .and("businessType.size").as("size")
                        .and("businessType.fieldPrecision").as("fieldPrecision")
                        .and("dictionary.dictionaryKey").as("dictionaryKey")
                        .and("dictionary.dictionaryValues").as("dictionaryValues"),
        // 排序
        Aggregation.sort(Sort.Direction.DESC, "editTime"),
                // 分页
                Aggregation.skip((page.getPages() - 1) * page.getSize()),
                Aggregation.limit(page.getSize())
        );

        List<Document> documentList = mongoTemplate.aggregate(
                aggregation,
                "tStandardWordDictionary",
                Document.class
        ).getMappedResults();

        List<Map<String, Object>> resultList = new ArrayList<>();
        for (Document document : documentList) {
            Map<String, Object> map = new HashMap<>(document);
            resultList.add(map);
        }
        return resultList;
    }

    private Criteria buildMatchCriteria(
            List<String> teamIds,
            String auditStatus,
            String keyWord,
            String standard) {
        Criteria criteria = Criteria.where("teamId").in(teamIds);

        if (auditStatus != null &&!"".equals(auditStatus)) {
            criteria = criteria.and("auditStatus").is(auditStatus);
        }

        if (standard != null &&!"".equals(standard)) {
            criteria = criteria.and("standard").is(standard);
        }

        if (keyWord != null &&!"".equals(keyWord)) {
            criteria = criteria.and(
                    String.valueOf(Criteria.where("dataName").regex(".*" + keyWord + ".*")
                            .orOperator(
                                    Criteria.where("descriptionEnUs").regex(".*" + keyWord + ".*"),
                                    Criteria.where("descriptionZhCn").regex(".*" + keyWord + ".*"),
                                    Criteria.where("descriptionZhTw").regex(".*" + keyWord + ".*"),
                                    Criteria.where("likenessWord").regex(".*" + keyWord + ".*")
                            ))
            );
        }

        return criteria;
    }

    public List<Map<String, Object>> queryOuterISV(
            Page<WordDictionaryManagePO> page,
            String tenantId,
            List<String> teamIds,
            String auditStatus,
            String keyWord,
            String standard) {
        Aggregation aggregation = Aggregation.newAggregation(
                // 多表连接
                Aggregation.lookup(
                        "tStandardBusinessType",
                        "relateBusinessTypeId",
                        "id",
                        "businessType"
                ),
                Aggregation.lookup(
                        "tStandardDictionary",
                        "relateDictionaryId",
                        "id",
                        "dictionary"
                ),
                // 构建查询条件
                Aggregation.match(buildMatchCriteria(tenantId, teamIds, auditStatus, keyWord, standard)),
                // 投影，选择需要的字段并重命名
                Aggregation.project()
                        .and("dataName").as("dataName")
                        .and("descriptionZhTw").as("descriptionZhTw")
                        .and("descriptionZhCn").as("descriptionZhCn")
                        .and("descriptionEnUs").as("descriptionEnUs")
                        .and("relateBusinessTypeId").as("relateBusinessTypeId")
                        .and("relateDictionaryId").as("relateDictionaryId")
                        .and("editBy").as("editBy")
                        .and("editTime").as("editTime")
                        .and("createTime").as("createTime")
                        .and("createBy").as("createBy")
                        .and("tenantId").as("tenantId")
                        .and("manageStatus").as("manageStatus")
                        .and("likenessWord").as("likenessWord")
                        .and("id").as("id")
                        .and("standard").as("standard")
                        .and("teamId").as("teamId")
                        .and("auditStatus").as("auditStatus")
                        .and("businessType.code").as("bizCode")
                        .and("businessType.name").as("bizName")
                        .and("businessType.dataType").as("dataType")
                        .and("businessType.size").as("size")
                        .and("businessType.fieldPrecision").as("fieldPrecision")
                        .and("dictionary.dictionaryKey").as("dictionaryKey")
                        .and("dictionary.dictionaryValues").as("dictionaryValues"),
        // 排序
        Aggregation.sort(Sort.Direction.DESC, "editTime"),
                // 分页
                Aggregation.skip((page.getPages() - 1) * page.getSize()),
                Aggregation.limit(page.getSize())
        );

        List<Document> documentList = mongoTemplate.aggregate(
                aggregation,
                "tStandardWordDictionary",
                Document.class
        ).getMappedResults();

        List<Map<String, Object>> resultList = new ArrayList<>();
        for (Document document : documentList) {
            Map<String, Object> map = new HashMap<>(document);
            resultList.add(map);
        }
        return resultList;
    }

    private Criteria buildMatchCriteria(
            String tenantId,
            List<String> teamIds,
            String auditStatus,
            String keyWord,
            String standard) {
        Criteria criteria = Criteria.where("tenantId").is(tenantId);

        if (auditStatus != null &&!"".equals(auditStatus)) {
            criteria = criteria.and("auditStatus").is(auditStatus);
        }

        if (standard != null &&!"".equals(standard)) {
            criteria = criteria.and("standard").is(standard);
        }

        if ("2".equals(auditStatus) || teamIds != null &&!teamIds.isEmpty()) {
            criteria = criteria.and("teamId").in(teamIds).and("manageStatus").is("Y");
        }

        if (auditStatus == null || "".equals(auditStatus)) {
            criteria = criteria.and("auditStatus").is("3");
        } else {
            criteria = criteria.and("auditStatus").is(auditStatus);
        }

        if (standard == null || "".equals(standard) || "Y".equals(standard)) {
            criteria = criteria.and("standard").is("Y");
        } else {
            criteria = criteria.and("standard").is("Q");
        }

        if (keyWord != null &&!"".equals(keyWord)) {
            criteria = criteria.and(
                    String.valueOf(Criteria.where("dataName").regex(".*" + keyWord + ".*")
                            .orOperator(
                                    Criteria.where("descriptionEnUs").regex(".*" + keyWord + ".*"),
                                    Criteria.where("descriptionZhCn").regex(".*" + keyWord + ".*"),
                                    Criteria.where("descriptionZhTw").regex(".*" + keyWord + ".*"),
                                    Criteria.where("likenessWord").regex(".*" + keyWord + ".*")
                            ))
            );
        }

        return criteria;
    }

    public void updateTeamId(String teamId, String id) {
        Criteria criteria = Criteria.where("id").is(id);
        Update update = new Update().set("teamId", teamId);

        mongoTemplate.updateFirst(Query.query(criteria), update, "tStandardWordDictionary");
    }

    public List<WordDictionaryManagePO> selectByCode(List<String> codes) {
        Criteria criteria = Criteria.where("code").in(codes);
        Query query = new Query(criteria);
        return mongoSystemRepositoryDecorator.find(query, WordDictionaryManagePO.class);
    }

    public List<WordDictionaryManagePO> selectAll() {
        return mongoSystemRepositoryDecorator.findAll(WordDictionaryManagePO.class);
    }

    public List<String> selectAllTenantId() {
        return mongoSystemRepositoryDecorator.findAll(WordDictionaryManagePO.class)
                .stream()
                .map(WordDictionaryManagePO::getTenantId)
                .toList();
    }

    public List<WordDictionaryManagePO> selectByTenantId(String tenantId) {
        Criteria criteria = Criteria.where("tenantId").is(tenantId);
        Query query = new Query(criteria);
        return mongoSystemRepositoryDecorator.find(query, WordDictionaryManagePO.class);
    }

    public WordDictionaryManagePO selectOneByCondition(Criteria criteria) {
        Query query = new Query(criteria);
        return mongoSystemRepositoryDecorator.findOne(query, WordDictionaryManagePO.class);
    }

    public List<WordDictionaryManagePO> selectListByCondition(Criteria criteria) {
        Query query = new Query(criteria);
        return mongoSystemRepositoryDecorator.find(query, WordDictionaryManagePO.class);
    }

    public void deleteListByCondition(Criteria criteria) {
        Query query = new Query(criteria);
        mongoSystemRepositoryDecorator.delete(query, WordDictionaryManagePO.class);
    }

}
