package com.digiwin.athena.service.impl;

import cn.hutool.core.io.FileUtil;
import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.ZipUtil;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.digiwin.athena.base.BaseConstant;
import com.digiwin.athena.dao.CompiledDataV2MongoDao;
import com.digiwin.athena.dao.DeployDetailMongoDao;
import com.digiwin.athena.dao.DeployLogMongoDao;
import com.digiwin.athena.domain.CompiledData;
import com.digiwin.athena.domain.DeployDetail;
import com.digiwin.athena.domain.DeployLog;
import com.digiwin.athena.domain.FileStore;
import com.digiwin.athena.dto.AppCompileDataResult;
import com.digiwin.athena.dto.ApplicationData;
import com.digiwin.athena.dto.CompileParam;
import com.digiwin.athena.dto.DeployBasicInfoResDto;
import com.digiwin.athena.dto.DeployDataDto;
import com.digiwin.athena.dto.DeployDetailInfoResDto;
import com.digiwin.athena.dto.DeployLogParamDto;
import com.digiwin.athena.dto.DeployLogResDto;
import com.digiwin.athena.dto.DeployParam;
import com.digiwin.athena.dto.TenantInfo;
import com.digiwin.athena.dto.UserInfoDto;
import com.digiwin.athena.dto.action.MultiLanguageDTO;
import com.digiwin.athena.publish.Publish;
import com.digiwin.athena.publish.dto.PublishParam;
import com.digiwin.athena.service.DeployService;
import com.digiwin.athena.service.FileStoreService;
import com.digiwin.athena.utils.CurThreadInfoUtils;
import com.digiwin.athena.utils.user.UserHelper;
import io.jsonwebtoken.lang.Assert;
import jakarta.annotation.PostConstruct;
import jakarta.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.math.BigDecimal;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.stream.Collectors;

@Slf4j
@Service
public class DeployServiceImpl implements DeployService {

    @Autowired
    private DeployLogMongoDao deployLogMongoDao;
    @Autowired
    private CompiledDataV2MongoDao compiledDataV2MongoDao;

    @Autowired
    @Qualifier("businessTaskProcessPool")
    private ThreadPoolExecutor businessTaskProcessPool;

    @Autowired
    private List<Publish> publishList;

    @Autowired
    private FileStoreService fileStoreService;
    @Resource
    private RedissonClient redissonClient;

    @Autowired
    private DeployDetailMongoDao deployDetailMongoDao;

    public static final Map<String, Publish> publishModule = new HashMap<>();

    private static final String DEPLOY_REDIS_KEY = "athena_designer_deployer_";

    @PostConstruct
    public void init() {

        for (Publish publish : publishList) {
            Class<?> targetClass = AopUtils.getTargetClass(publish);
            Order order = targetClass.getAnnotation(Order.class);

            String orderSerial = this.getOrderSerial(order.value());
            publishModule.put(orderSerial + publish.getClass().getSimpleName(), publish);
        }
    }

    @Override
    public String addDeployPlan(DeployParam deployParam) {
        ApplicationData applicationData = deployParam.getApplicationData();
        CompiledData compiledData = compiledDataV2MongoDao.selectByCode(applicationData.getCompileDataCode());
        Assert.notNull(compiledData, applicationData.getCompileDataCode() + "编译包不存在");

        String deployNo = IdUtil.fastSimpleUUID();

        DeployLog deployLog = new DeployLog();
        deployLog.setApplication(applicationData.getApplication());
        deployLog.setDeployNo(deployNo);
        deployLog.setCompileData(compiledData);
        deployLog.setType(DeployLog.PUBLISH_LOG_TYPE);
        deployLog.setResult(DeployLog.WAITING);
        deployLog.setDeployParam(deployParam);

        UserInfoDto userInfoDto = UserInfoDto.create();
        deployLog.setUser(userInfoDto);

        UserHelper.fillCreateInfo(deployLog);
        UserHelper.fillEditInfo(deployLog);

        DeployDetail deployDetail = new DeployDetail();
        deployDetail.setDeployNo(deployNo);
        deployDetail.setAdpApplication(applicationData.getApplication());
        UserHelper.fillCreateInfo(deployDetail);
        UserHelper.fillEditInfo(deployDetail);

        //判断这个应用有没有正在发布，没有，就把之前的待发布的任务全部取消
        RLock lock = redissonClient.getLock(DEPLOY_REDIS_KEY + applicationData.getApplication());
        boolean isLock = lock.tryLock();
        try {
            if (isLock){
                //将之前的发布任务取消
                List<DeployLog> historyDeployLogList = deployLogMongoDao.selectByAppAndResult(applicationData.getApplication(), Arrays.asList(DeployLog.WAITING,DeployLog.EXECUTING));
                if (!historyDeployLogList.isEmpty()){
                    deployLogMongoDao.updateFailByApplicationAndResult(applicationData.getApplication(), Arrays.asList(DeployLog.WAITING,DeployLog.EXECUTING));
                    List<String> deployNos = historyDeployLogList.stream().map(DeployLog::getDeployNo).collect(Collectors.toList());

                    String langStr = "{\"content\":{\"en_US\":\"Task cancellation\",\"zh_CN\":\"任务取消\",\"zh_TW\":\"任務取消\"}}";
                    addDeployDetail(langStr,"任务取消",DeployDetail.FAIL,deployNos, applicationData.getApplication());
                }

                deployLogMongoDao.insert(deployLog);
                deployDetailMongoDao.insert(deployDetail);
            }else{
                throw new RuntimeException("该应用正在发版");
            }
        } finally {
            lock.unlock();
        }

        String branchKey = CurThreadInfoUtils.getBranchKey();
        String token = CurThreadInfoUtils.getToken();
        TenantInfo curTokenTenant = CurThreadInfoUtils.getCurTokenTenant();
//        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();

        businessTaskProcessPool.execute(() -> {
            try {
                CurThreadInfoUtils.setBranchKey(branchKey);
                CurThreadInfoUtils.setToken(token);
                CurThreadInfoUtils.setCurTokenTenant(curTokenTenant);
//                SecurityContextHolder.getContext().setAuthentication(authentication);

                doDeploy(deployNo, applicationData);
            } finally {
                CurThreadInfoUtils.removeToken();
                CurThreadInfoUtils.removeBranchKey();
                CurThreadInfoUtils.removeCurTokenTenant();
//                SecurityContextHolder.clearContext();
            }
        });
        return deployNo;
    }

    private void doDeploy(String deployNo, ApplicationData applicationData) {
        File zipFile = null;
        File unZipFile = null;
        RLock lock = redissonClient.getLock(DEPLOY_REDIS_KEY + applicationData.getApplication());
        boolean deployLock = lock.tryLock();

        if (deployLock) {
            try {
                CompiledData compiledData = compiledDataV2MongoDao.selectByCode(applicationData.getCompileDataCode());

                String contentLang = "{\"content\":{\"en_US\":\"Enter the release\",\"zh_CN\":\"进入发版\",\"zh_TW\":\"進入發版\"}}";
                addDeployDetail(contentLang,"进入发版",DeployDetail.SUCCESS,Arrays.asList(deployNo), applicationData.getApplication());

                String projectDir = System.getProperty("user.dir");
                FileStore fileStore = fileStoreService.getFile(compiledData.getCompiledDataFileId());
                zipFile = new File(projectDir + File.separator + fileStore.getFileName() + BaseConstant.DOT + fileStore.getFileType());
                unZipFile = new File(projectDir + File.separator + fileStore.getFileName());

                FileUtil.writeBytes(fileStore.getContent(), zipFile);
                unZipFile.mkdir();
                ZipUtil.unzip(zipFile, unZipFile);

                File[] dataFiles = unZipFile.listFiles();

                for (int i = 0; i < dataFiles.length; i++) {
                    File dataFile = dataFiles[i];
                    String originalFilename = dataFile.getName();
                    int index = originalFilename.lastIndexOf(BaseConstant.DOT);
                    String name = originalFilename.substring(0, index);

                    Publish publish = publishModule.get(name);

                    List<String> dataList = FileUtil.readLines(dataFile, Charset.forName("utf-8"));

                    deployLogMongoDao.updateProcess(new BigDecimal((i + 1) / dataFiles.length + 1), deployNo);

                    for (String data : dataList) {
                        List list = publish.jsonStrToObject(data);
                        DeployDataDto deployDataDto = new DeployDataDto();
                        deployDataDto.setAdpApplication(applicationData.getApplication());
                        deployDataDto.setDeployNo(deployNo);

                        deployDataDto.setPublishData(list);
                        publish.publishAll(deployDataDto);
                    }
                }
                String langStr = "{\"content\":{\"en_US\":\"Release completed\",\"zh_CN\":\"发布完成\",\"zh_TW\":\"發布完成\"}}";
                addDeployDetail(langStr,"发布完成",DeployDetail.SUCCESS,Arrays.asList(deployNo), applicationData.getApplication());

                langStr = "{\"content\":{\"en_US\":\"Finish\",\"zh_CN\":\"结束\",\"zh_TW\":\"結束\"}}";
                addDeployDetail(langStr,"结束",DeployDetail.FINISH,Arrays.asList(deployNo), applicationData.getApplication());

                deployLogMongoDao.updateProcessAndResult(new BigDecimal(1),DeployLog.SUCCESS, deployNo);
            } catch (Exception e) {
                log.error(deployNo + "发布异常", e);
                deployLogMongoDao.updateProcessAndResult(new BigDecimal(-1), DeployLog.FAIL, deployNo);
            } finally {
                if (zipFile != null) {
                    FileUtil.del(zipFile);
                }
                if (unZipFile != null) {
                    FileUtil.del(unZipFile);
                }
                lock.unlock();
            }
        }
    }

    private void addDeployDetail(String contentLang,String content,String result,List<String> deployNos, String application) {

        DeployDetail.DetailContent detailContent = new DeployDetail.DetailContent();
        detailContent.setContent(content);
        detailContent.setResult(result);
        detailContent.setTime(new Date());
        Map<String, MultiLanguageDTO> lang = JSONObject.parseObject(contentLang, new TypeReference<Map<String, MultiLanguageDTO>>() {
        });
        detailContent.setLang(lang);
        deployDetailMongoDao.batchUpdatePushContent(deployNos, application,detailContent);
    }

    @Override
    public String compile(CompileParam compileParam) {
        String compiledCode = IdUtil.fastSimpleUUID();
        String projectDir = System.getProperty("user.dir");
        File directory = new File(projectDir + File.separator + compiledCode);
        File zip = null;
        try {
            directory.mkdir();
            writeCompileData(compileParam.getApplication(), directory);

            zip = ZipUtil.zip(directory);
            String originalFilename = zip.getName();
            int index = originalFilename.lastIndexOf(BaseConstant.DOT);
            String name = originalFilename.substring(0, index);
            String type = originalFilename.substring(index + 1);

            String fieldId = fileStoreService.uploadFile(new FileInputStream(zip), name, type);

            CompiledData compiledData = compileParam.createCompiledData();
            compiledData.setCompiledDataFileId(fieldId);
            compiledData.setCode(compiledCode);

            compiledDataV2MongoDao.insert(compiledData);

        } catch (Exception e) {
            log.error("compile 异常" + e.getMessage(), e);
            throw new RuntimeException(e.getMessage());
        } finally {
            FileUtil.del(directory);
            if (zip != null) {
                FileUtil.del(zip);
            }
        }

        return compiledCode;
    }

    @Override
    public void writeCompileData(String application, File directory) {
        for (Publish publish : publishList) {
            Class<?> aClass = AopUtils.getTargetClass(publish);

            if (aClass.getSimpleName().equals("DataStandardPublish") && !"DATASTANDARDS".equals(application)) {
                continue;
            }
            PublishParam publishParam = new PublishParam();
            publishParam.setApplication(application);
            List allPublishData = publish.findAllPublishData(publishParam);
            List compiledPublishData = publish.doCompile(allPublishData);

            if (CollectionUtils.isEmpty(compiledPublishData)){
                continue;
            }

            Order order = aClass.getAnnotation(Order.class);
            File file = new File(directory, getOrderSerial(order.value()) + aClass.getSimpleName() + ".json");
            File parentDir = file.getParentFile();

            if (!parentDir.exists()) {
                boolean created = parentDir.mkdirs();
                if (!created) {
                    throw new RuntimeException("目录创建失败: " + parentDir.getAbsolutePath());
                }
            }

            try (BufferedWriter writer = new BufferedWriter(new FileWriter(file))) {
                writer.write(JSONObject.toJSONString(compiledPublishData));
            } catch (Exception e) {
                log.error("写入文件异常" + e.getMessage(), e);
                throw new RuntimeException(e.getMessage());
            }
        }
    }

    @Override
    public List<AppCompileDataResult> queryCompileData(List<String> applicationList) {
        List<AppCompileDataResult> res = new ArrayList<>();
        for (String app : applicationList) {
            List<CompiledData> compiledData = compiledDataV2MongoDao.selectByApplication(app);

            AppCompileDataResult appCompileDataResult = new AppCompileDataResult();
            appCompileDataResult.setCompileDataList(compiledData);
            appCompileDataResult.setApplication(app);

            res.add(appCompileDataResult);
        }
        return res;
    }

    @Override
    public List<DeployLogResDto> queryDeployLogByEnv(DeployLogParamDto deployLogParamDto) {
        DeployLog lastOne = deployLogMongoDao.selectLastOne(deployLogParamDto.getApplication());
        if (lastOne == null){
            return new ArrayList<>();
        }

        DeployLog successDeployLog = deployLogMongoDao.selectSuccessOneByApplication(deployLogParamDto.getApplication());
        if (DeployLog.EXECUTING.equals(lastOne.getResult()) || DeployLog.WAITING.equals(lastOne.getResult())) {
            if (successDeployLog != null) {
                DeployLogResDto res = DeployLogResDto.create(successDeployLog);
                res.setResult(lastOne.getResult());
                return Arrays.asList(res);
            }
        } else if (DeployLog.SUCCESS.equals(lastOne.getResult())) {
            DeployLogResDto res = DeployLogResDto.create(lastOne);
            return Arrays.asList(res);
        } else if (DeployLog.FAIL.equals(lastOne.getResult())) {
            if (successDeployLog!=null){
                DeployLogResDto res = DeployLogResDto.create(successDeployLog);
                return Arrays.asList(res);
            }
        }
        return new ArrayList<>();
    }

    @Override
    public DeployBasicInfoResDto queryLatestDeployInfo(String application) {
        DeployLog deployLog = deployLogMongoDao.selectLastOne(application);
        if (deployLog == null){
            return null;
        }
        return DeployBasicInfoResDto.create(deployLog);
    }

    @Override
    public DeployDetailInfoResDto queryDeployDetail(String deployNo, String application) {
        DeployLog deployLog = deployLogMongoDao.selectByDeployNoAndApp(deployNo,application);
        Assert.notNull(deployLog,deployNo+"没有对应的发版记录");

        DeployDetailInfoResDto deployDetailInfoResDto = new DeployDetailInfoResDto();
        DeployDetail deployDetail = deployDetailMongoDao.selectByDeployNoAndApp(deployNo,application);

        deployDetailInfoResDto.setProcess(deployLog.getProcess());
        deployDetailInfoResDto.setDetailContents(deployDetail.getContents());

        return deployDetailInfoResDto;
    }

    private String getOrderSerial(Integer order) {
        String orderStr = order.toString();
        if (orderStr.length() < 4) {
            int len = 4 - orderStr.length();
            for (int i = 0; i < len; i++) {
                orderStr = 0 + orderStr;
            }
        }
        return orderStr;
    }

}
