package com.digiwin.athena.publish;

import com.digiwin.athena.dao.DeployDetailMongoDao;
import com.digiwin.athena.domain.DeployDetail;
import com.digiwin.athena.dto.BusinessDto;
import com.digiwin.athena.dto.DeployDataDto;
import com.digiwin.athena.publish.dto.PublishParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Arrays;
import java.util.Date;
import java.util.List;

/**
 * 设计成果发布接口，暂放在PageDsl
 *
 * @author Jiangmx
 */
@Slf4j
public abstract class Publish<T extends BusinessDto> extends PublishExtendOperation<T> {

    @Autowired
    private DeployDetailMongoDao deployDetailMongoDao;

    public void publishOne(PublishParam publishParam) {
        T publishData = findSinglePublishData(publishParam);

        preHandle(publishData);

        doCompile(Arrays.asList(publishData));

        deleteSingleRuntimeData(publishData);
        insertSingleRuntimeData(publishData);

        postHandle(publishData);
    }

    public void publishAll(DeployDataDto<T> deployDataDto) {

        DeployDetail.DetailContent detailContent = null;
        try {
            List<T> allPublishData = deployDataDto.getPublishData();

            startPreHandle(deployDataDto);

            for (T data : allPublishData) {

                Date start = new Date();
                detailContent = getDeployDetailContent(data);
                detailContent.setTime(start);

                preHandle(data);

                updateDeployDetail(deployDataDto,detailContent);

                deleteSingleRuntimeData(data);

                insertSingleRuntimeData(data);

                postHandle(data);

                Date end = new Date();
                detailContent.setTime(end);
                detailContent.setResult(DeployDetail.SUCCESS);
                detailContent.setExecuteTimeMills(end.getTime()-start.getTime());
                updateDeployDetail(deployDataDto,detailContent);

            }

            endPostHandle(deployDataDto);

        } catch (Exception e) {
            log.error("publishAll exception:"+e.getMessage(),e);
            detailContent.setResult(DeployDetail.FAIL)
                    .setErrorStack(getStackTraceAsStrings(e))
                    .setCause(e.toString());
            updateDeployDetail(deployDataDto,detailContent);
            throw new RuntimeException(e);
        }
    }

    protected static String[] getStackTraceAsStrings(Throwable t) {
        StackTraceElement[] elements = t.getStackTrace();
        String[] stackStrings = new String[elements.length];
        for (int i = 0; i < elements.length; i++) {
            // toString() 会返回完整的 类名.方法名(文件名:行号)
            stackStrings[i] = elements[i].toString();
        }
        return stackStrings;
    }

    public DeployDetail.DetailContent getDeployDetailContent(T data){return new DeployDetail.DetailContent();}

    private void updateDeployDetail(DeployDataDto<T> deployDataDto, DeployDetail.DetailContent detailContent) {
        if (StringUtils.isEmpty(deployDataDto.getDeployNo())||StringUtils.isEmpty(deployDataDto.getAdpApplication())){
            return;
        }
        deployDetailMongoDao.updatePushContent(deployDataDto.getDeployNo(),deployDataDto.getAdpApplication(),detailContent);
    }

    /**
     * 删除运行时数据
     * @param
     * @return
     */
    abstract void deleteRuntimeData(DeployDataDto<T> deployDataDto);


    /**
     * 插入单个作业运行时数据
     * @param publishData
     */
    abstract void insertSingleRuntimeData(T publishData);

    /**
     * 删除单个作业运行时数据
     * @param publishData
     */
    abstract void deleteSingleRuntimeData(T publishData);



}