package com.digiwin.athena.http.iam.service.impl;

import cn.hutool.cache.Cache;
import cn.hutool.cache.impl.LRUCache;
import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.stream.CollectorUtil;
import cn.hutool.core.util.StrUtil;
import com.digiwin.athena.base.BusinessException;
import com.digiwin.athena.http.iam.IamProp;
import com.digiwin.athena.http.iam.api.IamTenantApi;
import com.digiwin.athena.http.iam.dto.*;
import com.digiwin.athena.http.iam.service.TenantInfoService;
import com.digiwin.athena.http.iam.service.TokenService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.time.Duration;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 租户信息缓存服务实现
 * 在designer-web模块中实现具体的API调用逻辑
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class IamTenantInfoFacadeService implements TenantInfoService {
    private static Cache<String, TenantInfoDto> tenantInfoCache
            = new LRUCache<>(100, Duration.ofDays(1).toMillis());

    private final IamProp prop;
    private final IamTenantApi iamTenantApi;
    private final TokenService tokenService;

    public static void setTenantInfoCache(Cache<String, TenantInfoDto> tenantInfoCache) {
        IamTenantInfoFacadeService.tenantInfoCache = tenantInfoCache;
    }

    /**
     * 获取租户信息（带缓存）
     *
     * @param tenantId 归属租户ID
     * @return 租户信息
     */
    @Override
    public TenantInfoDto getTenantInfoWithCache(String tenantId) {
        return tenantInfoCache.get(tenantId, false, () -> getTenantInfoDto(tenantId));
    }

    @Override
    public TenantInfoDto getTenantInfoDto(String tenantId) {
        if (StrUtil.isBlank(tenantId)) {
            return null;
        }

        return getTenantInfoDto(Collections.singleton(tenantId))
                .get(tenantId);
    }

    @Override
    public Map<String, TenantInfoDto> getTenantInfoWithCache(Collection<String> tenantIdColl) {
        if (CollUtil.isEmpty(tenantIdColl)) {
            return Collections.emptyMap();
        }

        // TODO pzz performance 改为批量查询，逐个缓存
        return tenantIdColl.stream()
                .map(this::getTenantInfoWithCache)
                .filter(Objects::nonNull)
                .collect(Collectors.toMap(TenantInfoDto::getTenantId, v -> v));
    }

    @Override
    public Map<String, TenantInfoDto> getTenantInfoDto(Collection<String> tenantIdColl) {
        if (CollUtil.isEmpty(tenantIdColl)) {
            return Collections.emptyMap();
        }

        // 缓存中没有，调用API获取
        try {
            IamLoginInfo integrationToken = tokenService.getIntegrationTokenWithCache(CollUtil.getFirst(tenantIdColl));
            if (integrationToken == null) {
                throw new BusinessException("获取集成Token异常！");
            }

            return fetchTenantInfoFromApi(prop.getAppToken(), integrationToken.getToken(), CollUtil.newHashSet(tenantIdColl));
        } catch (Exception e) {
            log.error("获取租户信息失败: tenantIdColl={}, msg={}", tenantIdColl, e.getMessage(), e);
            return Collections.emptyMap();
        }
    }

    /**
     * 从API获取租户信息
     *
     * @param tenantId 归属租户ID
     * @return 租户信息
     */
    protected Map<String, TenantInfoDto> fetchTenantInfoFromApi(String appToken, String token, Set<String> tenantId) {
        if (CollUtil.isEmpty(tenantId)) {
            return Collections.emptyMap();
        }

        ResultDto<List<IamTenantBasicInfo>> resp = iamTenantApi.fetchTenantBasicInfoList(appToken, token,
                IamTenantReqDto.of(tenantId));

        if (!resp.isSuccess() || CollUtil.isEmpty(resp.getData())) {
            log.error("查询iam租户信息失败！resp:{}", resp);
            return Collections.emptyMap();
        }

        return resp.getData().stream()
                .map(TenantInfoDto::from)
                .collect(
                        CollectorUtil.toMap(TenantInfoDto::getTenantId,
                                v -> v,
                                (v1, v2) -> v1)
                );
    }
}
