package com.digiwin.athena.validate;

import com.digiwin.athena.dto.action.MultiLanguageDTO;
import jakarta.validation.Constraint;
import jakarta.validation.ConstraintValidator;
import jakarta.validation.ConstraintValidatorContext;
import jakarta.validation.Payload;

import java.lang.annotation.Documented;
import java.lang.annotation.Repeatable;
import java.lang.annotation.Retention;
import java.lang.annotation.Target;
import java.util.Map;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

import static java.lang.annotation.ElementType.*;
import static java.lang.annotation.RetentionPolicy.RUNTIME;


/**
 * 校验lang多语言文本长度注解
 */
@Documented
@Constraint(validatedBy = {ValidLangLen.LangLenValidator.class, ValidLangLen.MapLangLenValidator.class})
@Target({METHOD, FIELD, PARAMETER})
@Retention(RUNTIME)
@Repeatable(ValidLangLen.List.class)
public @interface ValidLangLen {
    int min() default 0;

    int max() default Integer.MAX_VALUE;

    String message() default "{org.hibernate.validator.constraints.Length.message}";

    Class<?>[] groups() default {};

    Class<? extends Payload>[] payload() default {};

    /**
     * Defines several {@code @NotBlank} constraints on the same element.
     *
     * @see ValidLangLen
     */
    @Target({METHOD, FIELD, PARAMETER})
    @Retention(RUNTIME)
    @Documented
    @interface List {
        ValidLangLen[] value();
    }

    abstract class BaseLangLenValidator<T> implements ConstraintValidator<ValidLangLen, T> {
        protected int min;
        protected int max;

        @Override
        public void initialize(ValidLangLen constraintAnnotation) {
            this.min = constraintAnnotation.min();
            this.max = constraintAnnotation.max();
        }

        protected boolean judgeLen(Consumer<String> notValidCallback, MultiLanguageDTO<?> lang) {
            if (lang == null) {
                return true;
            }

            boolean isValid = true;
            if (lang.getZh_CN() != null && notBetween(String.valueOf(lang.getZh_CN()).length(), min, max)) {
                notValidCallback.accept("zh_CN");
                isValid = false;
            }

            if (lang.getEn_US() != null && notBetween(String.valueOf(lang.getEn_US()).length(), min, max)) {
                notValidCallback.accept("en_US");
                isValid = false;
            }

            if (lang.getZh_TW() != null && notBetween(String.valueOf(lang.getZh_TW()).length(), min, max)) {
                notValidCallback.accept("zh_TW");
                isValid = false;
            }

            return isValid;
        }

        protected boolean notBetween(int value, int min, int max) {
            return value < min || value > max;
        }

        protected boolean _isValid(BiConsumer<ConstraintValidatorContext.ConstraintViolationBuilder, String> notValidCallback,
                                   MultiLanguageDTO<?> lang, ConstraintValidatorContext context) {
            ConstraintValidatorContext.ConstraintViolationBuilder builder =
                    context.buildConstraintViolationWithTemplate(context.getDefaultConstraintMessageTemplate());

            boolean isValid = judgeLen(
                    fieldName -> notValidCallback.accept(builder, fieldName),
                    lang);

            if (!isValid) {
                context.disableDefaultConstraintViolation();
                builder.addConstraintViolation();
            }
            return isValid;
        }
    }

    class LangLenValidator extends BaseLangLenValidator<MultiLanguageDTO<?>> {
        @Override
        public boolean isValid(MultiLanguageDTO lang, ConstraintValidatorContext context) {
            return _isValid(ConstraintValidatorContext.ConstraintViolationBuilder::addPropertyNode,
                    lang, context);
        }
    }

    class MapLangLenValidator extends BaseLangLenValidator<Map<String, MultiLanguageDTO<?>>> {
        private static final String KEY = "name";

        @Override
        public boolean isValid(Map<String, MultiLanguageDTO<?>> map, ConstraintValidatorContext context) {
            if (map == null || map.isEmpty()) {
                return true;
            }

            MultiLanguageDTO<?> lang = map.get(KEY);
            return _isValid((builder, fieldName) ->
                            builder.addPropertyNode(fieldName)
                                    .inContainer(Map.class, 1)
                                    .inIterable().atKey(KEY),
                    lang, context);
        }
    }

}
