package com.digiwin.athena.publish;

import cn.hutool.core.collection.CollUtil;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.digiwin.athena.dao.DeployParamRecordMongoDao;
import com.digiwin.athena.dao.RPageDesignDao;
import com.digiwin.athena.dao.RPageDslDao;
import com.digiwin.athena.domain.DeployParamRecord;
import com.digiwin.athena.domain.PageDesign;
import com.digiwin.athena.domain.PageDesignTime;
import com.digiwin.athena.domain.PageDsl;
import com.digiwin.athena.dto.DeployDataDto;
import com.digiwin.athena.dto.PageDesignBusinessDto;
import com.digiwin.athena.dto.SystemParam;
import com.digiwin.athena.http.distribution.*;
import com.digiwin.athena.utils.user.UserHelper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

@Service("pageDesignExtendWrapper")
@Slf4j
@ConditionalOnProperty(name = "system_env",havingValue = "FenXiao")
public class FxPageDesignExtendWrapper extends PublishExtendOperationWrapper<PageDesignBusinessDto>{

    @Autowired
    private FxApiHelper fxApiHelper;

    @Autowired
    private DeployParamRecordMongoDao deployParamRecordMongoDao;

    @Override
    public List<PageDesignBusinessDto> doCompile(List<PageDesignBusinessDto> data) {

        JSONArray sysEnumList = fxApiHelper.getSysEnum();

        for (Object o : sysEnumList) {
            JSONObject jo = (JSONObject) o;

            JSONObject variable = new JSONObject();
            variable.put("name", jo.getString("id"));
            variable.put("defaultValue", jo.getString("value"));
            variable.put("scope", "system");

            for (PageDesignBusinessDto businessDto : data) {
                businessDto.getPageDslList().forEach(b -> b.getDsl().getVariables().add(variable));
            }
        }
        return data;
    }

    @Override
    public void endPostHandle(DeployDataDto<PageDesignBusinessDto> deployDataDto) {

        List<PageDesignBusinessDto> publishDataList = deployDataDto.getPublishData();
        Set<String> codes = publishDataList.stream().map(p -> p.getPageDesign().getCode()).collect(Collectors.toSet());


        DeployParamRecord deployParamRecord = deployParamRecordMongoDao.selectByApplicationAndType(deployDataDto.getAdpApplication(),"pageDesignDeployData");
        if (deployParamRecord!=null) {
            List<JSONObject> values = JSONObject.parseObject(JSONObject.toJSONString(deployParamRecord.getValue()), new TypeReference<>() {
            });


            List<DeleteFxJobReqDto> deleteFxJobReqDtos = new ArrayList<>();
            Iterator<JSONObject> iterator = values.iterator();
            while (iterator.hasNext()){
                JSONObject next = iterator.next();
                String code = next.getString("code");
                if (!codes.contains(code)){

                    HashMap<String, String> param = new HashMap<>();
                    param.put("aah001",code);
                    FxRegisterJobDetailDto jobDetailInfo = fxApiHelper.getJobDetailInfo(param);

                    if (jobDetailInfo!=null){
                        DeleteFxJobReqDto deleteFxJobReqDto = new DeleteFxJobReqDto();
                        deleteFxJobReqDto.setAah001(code);
                        deleteFxJobReqDto.setAahlock(jobDetailInfo.getAahlock());
                        deleteFxJobReqDtos.add(deleteFxJobReqDto);
                    }

                    iterator.remove();
                }
            }

            if (!deleteFxJobReqDtos.isEmpty()){
                fxApiHelper.deleteJob(deleteFxJobReqDtos);
                deployParamRecord.setValue(values);
                deployParamRecordMongoDao.save(deployParamRecord);
            }

        }
    }

    @Override
    public PageDesignBusinessDto postHandle(PageDesignBusinessDto publishData) {
        PageDesign pageDesign = publishData.getPageDesign();
        DeployParamRecord deployParamRecord = deployParamRecordMongoDao.selectByApplicationAndType(pageDesign.getAdpApplication(), "pageDesignDeployData");

        List<JSONObject> valueList;
        if (deployParamRecord == null) {
            deployParamRecord = new DeployParamRecord();
            deployParamRecord.setAdpApplication(pageDesign.getAdpApplication());
            deployParamRecord.setType("pageDesignDeployData");
            UserHelper.fillCreateInfo(deployParamRecord);
            UserHelper.fillEditInfo(deployParamRecord);

            valueList = new ArrayList<>();
        } else {
            UserHelper.fillEditInfo(deployParamRecord);
            valueList = JSONObject.parseObject(JSONObject.toJSONString(deployParamRecord.getValue()), new TypeReference<>() {
            });
            valueList.removeIf(v -> v.getString("code").equals(pageDesign.getCode()));
        }

        JSONObject newParam = new JSONObject();
        newParam.put("code", pageDesign.getCode());
        newParam.put("name", pageDesign.getName());
        valueList.add(newParam);
        deployParamRecord.setValue(valueList);

        registerFxJob(publishData);

        if (deployParamRecord.getObjectId() == null) {
            deployParamRecordMongoDao.insert(deployParamRecord);
        } else {
            deployParamRecordMongoDao.save(deployParamRecord);
        }

        return publishData;
    }

    @Override
    public PageDesignBusinessDto preHandle(PageDesignBusinessDto publishData) {
        //同步文件
        List<PageDsl> pageDslList = publishData.getPageDslList();
        pageDslList.stream().map(PageDsl::getDesignTime)
                .filter(Objects::nonNull)
                .map(PageDesignTime::getDataImportInfo)
                .filter(Objects::nonNull)
                .filter(d-> !CollectionUtils.isEmpty(d.getComponentList()))
                .map(d->JSONObject.parseObject(JSONObject.toJSONString(d),SyncFileInfoReqDto.class))
                .forEach(fxApiHelper::syncFileInfo);

        return publishData;
    }

    private void registerFxJob(PageDesignBusinessDto businessDto) {
        PageDesign pageDesign = businessDto.getPageDesign();
        //查询这个作业是否已经注册过
        GetFxJobListReqDto getFxJobListReqDto = new GetFxJobListReqDto();
        getFxJobListReqDto.setGlikesel(pageDesign.getCode());

        HashMap<String, String> param = new HashMap<>();
        param.put("aah001",pageDesign.getCode());

        FxRegisterJobDetailDto jobDetailInfo = fxApiHelper.getJobDetailInfo(param);

        //没有注册过就注册下作业
        if (jobDetailInfo == null) {
            RegisterFxJobReqDto fxJobMenuParam = RegisterFxJobReqDto.createFxJobMenuParam(pageDesign.getCode(), pageDesign.getName());
            buildComponentLists(pageDesign, businessDto.getPageDslList(), fxJobMenuParam);

            fxApiHelper.registerJob(fxJobMenuParam);
        }else {
            String newName = pageDesign.getName();
            jobDetailInfo.setAah001_n1_o(jobDetailInfo.getAah001_n1());
            jobDetailInfo.setAah001_n1(newName);
            buildComponentLists(pageDesign, businessDto.getPageDslList(), jobDetailInfo);

            fxApiHelper.updateJobInfo(jobDetailInfo);
        }
    }

    /**
     * 构建组件列表
     */
    private void buildComponentLists(PageDesign pageDesign, List<PageDsl> dslList, Object reqDto) {
        List<PageDesignTime.Authority> list = CollUtil.emptyIfNull(dslList).stream()
                .flatMap(v -> Optional.ofNullable(v)
                        .map(PageDsl::getDesignTime)
                        .map(PageDesignTime::getAuthority)
                        .orElse(Collections.emptyList())
                        .stream()
                ).toList();

        if (list.isEmpty()) {
            return;
        }

        List<JSONObject> aajList = new ArrayList<>(); // 按钮名称多语言
        List<JSONObject> adeList = new ArrayList<>();  // 作业区块/字段组件表

        for (PageDesignTime.Authority authority : list) {
            if (authority.getComponentInfo() == null || authority.getKey() == null) {
                continue;
            }

            String category = authority.getComponentInfo().getCategory();
            if (PageDesignTime.AUTH_COMP_CALS_FUNC.equals(category)) {
                // 功能权限 - 按钮
                aajList.add(RegisterFxJobReqDto.buildAaj(pageDesign.getCode(), authority.getKey(), authority.getNameLangDto()));
            } else if (PageDesignTime.AUTH_COMP_CALS_COMP.equals(category)) {
                // 组件权限 - 字段组件
                adeList.add(RegisterFxJobReqDto.buildAde(authority.getKey(), authority.getNameLangDto()));
            }
        }

        if (reqDto instanceof RegisterFxJobReqDto dto) {
            dto.setAajList(aajList);
            dto.setAdeList(adeList);
        } else if (reqDto instanceof FxRegisterJobDetailDto dto) {
            dto.setAajList(aajList);
            dto.setAdeList(adeList);
        }
    }

}
