package com.digiwin.athena.publish;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.digiwin.athena.dao.CustomControlRelationV2MongoDao;
import com.digiwin.athena.dao.DPageDesignDao;
import com.digiwin.athena.dao.DPageDslDao;
import com.digiwin.athena.dao.RCustomControlRelationMongoDao;
import com.digiwin.athena.dao.RPageDesignDao;
import com.digiwin.athena.dao.RPageDslDao;
import com.digiwin.athena.dao.UserDefinedCenterMongoDao;
import com.digiwin.athena.mongodb.domain.appCustomConfig.CustomControlRelation;
import com.digiwin.athena.domain.DeployDetail;
import com.digiwin.athena.domain.PageDesign;
import com.digiwin.athena.domain.PageDsl;
import com.digiwin.athena.dto.DeployDataDto;
import com.digiwin.athena.dto.PageDesignBusinessDto;
import com.digiwin.athena.dto.action.MultiLanguageDTO;
import com.digiwin.athena.publish.dto.PublishParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author Jiangmx
 * @description 页面设计发布
 * @date 2025/6/23
 */
@Slf4j
@Service
@Order(100)
public class PageDesignPublish extends Publish<PageDesignBusinessDto> {

    @Autowired
    private DPageDesignDao dPageDesignDao;
    @Autowired
    private DPageDslDao dPageDslDao;
    @Autowired
    private RPageDesignDao rPageDesignDao;
    @Autowired
    private RPageDslDao rPageDslDao;
    @Autowired
    private CustomControlRelationV2MongoDao customControlRelationV2MongoDao;
    @Autowired
    private RCustomControlRelationMongoDao rCustomControlRelationMongoDao;
    @Autowired
    @Qualifier("pageDesignExtendWrapper")
    private PublishExtendOperationWrapper<PageDesignBusinessDto> publishExtendOperationWrapper;
    @Autowired
    private UserDefinedCenterMongoDao userDefinedCenterMongoDao;

    @Override
    public List<PageDesignBusinessDto> doCompile(List<PageDesignBusinessDto> data) {
        if (CollectionUtil.isEmpty(data)) {
            return data;
        }
        return publishExtendOperationWrapper.doCompile(data);
    }

    @Override
    public PageDesignBusinessDto preHandle(PageDesignBusinessDto publishData) {
        return publishExtendOperationWrapper.preHandle(publishData);
    }

    @Override
    public PageDesignBusinessDto postHandle(PageDesignBusinessDto publishData) {
        PageDesign pageDesign = publishData.getPageDesign();
        dPageDesignDao.updatePublishedTime(pageDesign.getCode(), pageDesign.getAdpApplication(), new Date());

        return publishExtendOperationWrapper.postHandle(publishData);
    }

    @Override
    void deleteRuntimeData(DeployDataDto<PageDesignBusinessDto> deployDataDto) {
        List<PageDesignBusinessDto> publishDataList = deployDataDto.getPublishData();
        for (PageDesignBusinessDto pageDesignBusinessDto : publishDataList) {
            deleteSingleRuntimeData(pageDesignBusinessDto);
        }

    }

    @Override
    public List<PageDesignBusinessDto> findAllPublishData(PublishParam publishParam) {
        List<PageDesignBusinessDto> res = new ArrayList<>();
        List<PageDesign> pageDesignList = dPageDesignDao.selectByApplication(publishParam.getApplication());
        List<PageDsl> pageDslList = dPageDslDao.selectByApplication(publishParam.getApplication());
        pageDslList.forEach(p -> p.setObjectId(null));
        Map<String, List<PageDsl>> pageDslMap = pageDslList.stream().collect(Collectors.groupingBy(PageDsl::getPageDesignCode));


        for (PageDesign pageDesign : pageDesignList) {
            PageDesignBusinessDto pageDesignBusinessDto = new PageDesignBusinessDto();
            res.add(pageDesignBusinessDto);

            pageDesign.setObjectId(null);
            pageDesignBusinessDto.setPageDesign(pageDesign);
            if (pageDslMap.containsKey(pageDesign.getCode())) {
                pageDesignBusinessDto.setPageDslList(pageDslMap.get(pageDesign.getCode()));
            }

            List<CustomControlRelation> customControlRelations = customControlRelationV2MongoDao.selectByTargetCodeAndApplication(pageDesign.getCode(), publishParam.getApplication());
            customControlRelations.forEach(c->c.setObjectId(null));
            pageDesignBusinessDto.setControlRelations(customControlRelations);
        }


        return res;
    }

    @Override
    void insertSingleRuntimeData(PageDesignBusinessDto publishData) {
        rPageDesignDao.insert(publishData.getPageDesign());
        List<PageDsl> pageDslList = publishData.getPageDslList();
        if (!pageDslList.isEmpty()) {
            rPageDslDao.insertAll(pageDslList);
        }

        List<CustomControlRelation> controlRelations = publishData.getControlRelations();
        if (!controlRelations.isEmpty()){
            rCustomControlRelationMongoDao.insertAll(controlRelations);
        }
    }

    @Override
    void deleteSingleRuntimeData(PageDesignBusinessDto publishData) {
        PageDesign pageDesign = publishData.getPageDesign();
        rPageDesignDao.deleteByCodeAndApplication(pageDesign.getCode(), pageDesign.getAdpApplication());

        rPageDslDao.deleteByPageDesignCodes(Arrays.asList(pageDesign.getCode()), pageDesign.getAdpApplication());
        rCustomControlRelationMongoDao.deleteByTargetCodeAndApp(pageDesign.getCode(), pageDesign.getAdpApplication());
    }

    @Override
    public PageDesignBusinessDto findSinglePublishData(PublishParam publishParam) {
        PageDesign pageDesign = dPageDesignDao.selectByCodeAndApplication(publishParam.getCode(), publishParam.getApplication());
        Assert.notNull(pageDesign, publishParam.getCode() + "没有查询到对应的记录");
        pageDesign.setObjectId(null);

        List<PageDsl> pageDsls = dPageDslDao.selectByPageDesignCode(pageDesign.getCode(), publishParam.getApplication());
        pageDsls.forEach(p -> p.setObjectId(null));
        PageDesignBusinessDto res = new PageDesignBusinessDto();
        res.setPageDesign(pageDesign);
        res.setPageDslList(pageDsls);

        List<CustomControlRelation> customControlRelations = customControlRelationV2MongoDao.selectByTargetCodeAndApplication(pageDesign.getCode(), publishParam.getApplication());
        customControlRelations.forEach(c->c.setObjectId(null));
        res.setControlRelations(customControlRelations);

        return res;
    }

    @Override
    public DeployDetail.DetailContent getDeployDetailContent(PageDesignBusinessDto data) {
        DeployDetail.DetailContent detailContent = new DeployDetail.DetailContent();
        PageDesign pageDesign = data.getPageDesign();
        String content = "作业发布:"+pageDesign.getName();
        String langStr = "{\"content\":{\"en_US\":\"Job Posting:%s\",\"zh_CN\":\"作业发布:%s\",\"zh_TW\":\"作業發布:%s\"}}";

        Map<String, MultiLanguageDTO> languageDTOMap = pageDesign.getLang();
        if (!CollectionUtil.isEmpty(languageDTOMap) && languageDTOMap.containsKey("name")) {
            MultiLanguageDTO multiLanguageDTO = languageDTOMap.get("name");
            if (multiLanguageDTO != null) {
                langStr = String.format(langStr, multiLanguageDTO.getEn_US(), multiLanguageDTO.getZh_CN(), multiLanguageDTO.getZh_TW());
            }
        }
        detailContent.setContent(content);
        detailContent.setResult(DeployDetail.START);
        detailContent.setClassName(this.getClass().getName());
        Map<String, MultiLanguageDTO> lang = JSONObject.parseObject(langStr, new TypeReference<Map<String, MultiLanguageDTO>>() {
        });
        detailContent.setLang(lang);

        return detailContent;
    }

    @Override
    public void endPostHandle(DeployDataDto<PageDesignBusinessDto> deployDataDto) {
        List<PageDesignBusinessDto> publishDataList = deployDataDto.getPublishData();
        Set<String> codes = publishDataList.stream().map(p -> p.getPageDesign().getCode()).collect(Collectors.toSet());

        Set<String> rCodes = rPageDesignDao.selectCodeByAdpApplication(deployDataDto.getAdpApplication());

        rCodes.removeAll(codes);
        rPageDslDao.deleteByPageDesignCodes(rCodes,deployDataDto.getAdpApplication());
        rPageDesignDao.deleteByCodes(rCodes,deployDataDto.getAdpApplication());
        rCustomControlRelationMongoDao.deleteByTargetCodesAndApp(rCodes,deployDataDto.getAdpApplication());
        userDefinedCenterMongoDao.deleteByBizCode(rCodes);

        publishExtendOperationWrapper.endPostHandle(deployDataDto);

    }

    @Override
    public List<PageDesignBusinessDto> jsonStrToObject(String jsonStr) {

        return JSONObject.parseArray(jsonStr,PageDesignBusinessDto.class);
    }
}