package com.digiwin.athena.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.digiwin.athena.constant.CodeRuleTypeConstant;
import com.digiwin.athena.dao.DPageDesignDao;
import com.digiwin.athena.dao.DPageDslDao;
import com.digiwin.athena.domain.Dsl;
import com.digiwin.athena.domain.PageDesign;
import com.digiwin.athena.domain.PageDesignTime;
import com.digiwin.athena.domain.PageDsl;
import com.digiwin.athena.dto.CreatePageDesignDto;
import com.digiwin.athena.dto.PageDslDto;
import com.digiwin.athena.dto.QueryPageDesignResDto;
import com.digiwin.athena.dto.SavePageDesignBasicInfoReqDto;
import com.digiwin.athena.dto.SavePageDesignReqDto;
import com.digiwin.athena.dto.SystemParam;
import com.digiwin.athena.dto.action.MultiLanguageDTO;
import com.digiwin.athena.service.ActionMetaDataService;
import com.digiwin.athena.service.DPageDesignService;
import com.digiwin.athena.service.MultiPlatformDiffInterface;
import com.digiwin.athena.service.appCustomConfig.event.CustomControlEventManager;
import com.digiwin.athena.service.codeStandard.BuildGeneratorCodeUtil;
import com.digiwin.athena.utils.CurThreadInfoUtils;
import com.digiwin.athena.utils.operation.OperationConstants;
import com.digiwin.athena.utils.operation.OperationRecordWrapper;
import com.digiwin.athena.utils.operation.OperationTableNameConstants;
import com.digiwin.athena.utils.operation.aspect.RecordCommitLocal;
import com.digiwin.athena.utils.user.UserHelper;
import com.digiwin.athena.vo.action.ActionInfoResp;
import com.digiwin.athena.vo.action.ActionVersionReq;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

@Service
@Slf4j
public class DPageDesignServiceImpl implements DPageDesignService {

    @Autowired
    private DPageDslDao dPageDslDao;
    @Autowired
    private DPageDesignDao dPageDesignDao;
    @Autowired
    private BuildGeneratorCodeUtil buildGeneratorCodeUtil;
    @Autowired
    private MultiPlatformDiffInterface multiPlatformDiffInterface;

    @Autowired
    private CustomControlEventManager customControlEventManager;

    @Autowired
    private ActionMetaDataService actionMetaDataService;

    @Override
    public String createPageDesign(CreatePageDesignDto createPageDesignDto) {
        String deCode = buildGeneratorCodeUtil.createStandardCode(createPageDesignDto.getApplication(), Boolean.TRUE.equals(createPageDesignDto.getIsMobile()) ? CodeRuleTypeConstant.H5 : CodeRuleTypeConstant.DE);
        String browseDslCode = buildGeneratorCodeUtil.createStandardCode(createPageDesignDto.getApplication(), CodeRuleTypeConstant.DSL);
        String editDslCode = buildGeneratorCodeUtil.createStandardCode(createPageDesignDto.getApplication(), CodeRuleTypeConstant.DSL);

        PageDesign pageDesign = createPageDesignDto.createPageDesign();
        pageDesign.setCode(deCode);

        List<PageDsl> pageDslList = new ArrayList<>();
        if (CollectionUtils.isEmpty(createPageDesignDto.getPageDslList())) {
            pageDesign.setDslCode(browseDslCode);

            PageDsl browsePageDsl = createPageDesignDto.createPageDsl();
            browsePageDsl.setCode(browseDslCode);
            browsePageDsl.setPageDesignCode(deCode);
            browsePageDsl.setType(PageDesign.BASIC_TABLE_TEMPLATE_TYPE.equals(createPageDesignDto.getTemplateType()) ? PageDsl.BROWSE_TYPE : PageDsl.DESIGN_TYPE);
            browsePageDsl.setName(PageDesign.BASIC_TABLE_TEMPLATE_TYPE.equals(createPageDesignDto.getTemplateType())?"浏览界面":"界面设计");
            if (PageDesign.BASIC_TABLE_TEMPLATE_TYPE.equals(createPageDesignDto.getTemplateType())){
                browsePageDsl.setLang(JSONObject.parseObject("{\"name\": {\"zh_CN\": \"浏览界面\",\"zh_TW\": \"瀏覽介面\",\"en_US\": \"Browse Page\"}}",new TypeReference<Map<String, MultiLanguageDTO>>(){}));
            }else{
                browsePageDsl.setLang(JSONObject.parseObject("{\"name\": {\"zh_CN\": \"界面设计\",\"zh_TW\": \"介面設計\",\"en_US\": \"Interface Design\"}}",new TypeReference<Map<String, MultiLanguageDTO>>(){}));
            }
            pageDslList.add(browsePageDsl);
            if (PageDesign.BASIC_TABLE_TEMPLATE_TYPE.equals(createPageDesignDto.getTemplateType())) {
                PageDsl editPageDsl = createPageDesignDto.createPageDsl();
                editPageDsl.setCode(editDslCode);
                editPageDsl.setPageDesignCode(deCode);
                editPageDsl.setType(PageDsl.EDIT_TYPE);
                editPageDsl.setName("编辑界面");
                editPageDsl.setLang(JSONObject.parseObject("{ \"name\": {  \"zh_CN\": \"编辑界面\",  \"zh_TW\": \"编辑界面\",  \"en_US\": \"Edit Page\" }}",new TypeReference<Map<String, MultiLanguageDTO>>(){}));
                dPageDslDao.insert(editPageDsl);
                pageDslList.add(editPageDsl);
            }

            dPageDslDao.insert(browsePageDsl);
        } else {
            List<PageDslDto> pageDslDtoList = createPageDesignDto.getPageDslList();

            pageDslList = pageDslDtoList.stream().map(pageDslDto -> {
                PageDsl pageDsl = new PageDsl();
                pageDsl.setAdpApplication(createPageDesignDto.getApplication());
                pageDsl.setAdpTenantId(CurThreadInfoUtils.getCurTokenTenant().getTenantId());
                pageDsl.setVersion(createPageDesignDto.getVersion());
                pageDsl.setPageDesignCode(pageDesign.getCode());
                pageDsl.setCode(pageDslDto.getCode());
                pageDsl.setType(pageDslDto.getType());
                pageDsl.setDsl(pageDslDto.getDsl());
                pageDsl.setDesignTime(pageDslDto.getDesignTime());
                pageDsl.setName(pageDslDto.getName());
                pageDsl.setLang(pageDslDto.getLang());
                UserHelper.fillCreateInfo(pageDsl);
                UserHelper.fillEditInfo(pageDsl);

                dPageDslDao.insert(pageDsl);
                return pageDsl;
            }).collect(Collectors.toList());
        }
        dPageDesignDao.insert(pageDesign);


        JSONObject pageDesignJo = JSON.parseObject(JSON.toJSONString(pageDesign));
        pageDesignJo.put("pageDslList",pageDslList);

        RecordCommitLocal.add(new OperationRecordWrapper(pageDesign.getAdpApplication(), OperationTableNameConstants.LCDP_PAGE_DESIGN, pageDesignJo, OperationConstants.CREATE));

        return deCode;
    }

    @Override
    public QueryPageDesignResDto queryPageDesign(String code,String application) {
        List<SystemParam> sysEnumList = multiPlatformDiffInterface.getSysEnumV2();
        List<JSONObject> variables = new ArrayList<>();

        for (SystemParam systemParam : sysEnumList) {
            JSONObject variable = new JSONObject();
            variable.put("name", systemParam.getKey());
            variable.put("defaultValue", systemParam.getValue());
            variable.put("scope", "system");
            variable.put("description",systemParam.getDesc());
            variables.add(variable);
        }


        PageDesign pageDesign = dPageDesignDao.selectByCodeAndApplication(code,application);
        Assert.notNull(pageDesign,code+"不存在对应的记录");
        List<PageDsl> pageDslList = dPageDslDao.selectByPageDesignCode(code,application);
        for (PageDsl pageDsl : pageDslList) {
            if (pageDsl.getDsl()!=null){
                if (CollectionUtils.isEmpty(pageDsl.getDsl().getVariables())){
                    pageDsl.getDsl().setVariables(variables);
                }else{
                    Iterator<JSONObject> iterator = pageDsl.getDsl().getVariables().iterator();
                    //将系统变量过滤掉，添加最新的系统变量
                    while (iterator.hasNext()){
                        JSONObject next = iterator.next();
                        if ("system".equals(next.getString("scope"))){
                            iterator.remove();
                        }
                    }
                    pageDsl.getDsl().getVariables().addAll(variables);
                }
            }
        }
        QueryPageDesignResDto res = QueryPageDesignResDto.create(pageDesign);
        res.setPageDslList(pageDslList);
        res.setPublishedTime(pageDesign.getPublishedTime());
        return res;
    }

    @Override
    public void savePageDesign(List<SavePageDesignReqDto> savePageDesignReqDtoList) {
        Assert.isTrue(!CollectionUtils.isEmpty(savePageDesignReqDtoList), "dsl数据不能为空");
        SavePageDesignReqDto pd = savePageDesignReqDtoList.get(0);
        PageDesign pageDesign = dPageDesignDao.selectByCodeAndApplication(pd.getPageDesignCode(), pd.getAdpApplication());
        Assert.notNull(pageDesign,pd.getPageDesignCode()+"不存在");
        JSONObject pageDesignJo = JSON.parseObject(JSON.toJSONString(pageDesign));
        List<PageDsl> pageDslList = new ArrayList<>();
        pageDesignJo.put("pageDslList",pageDslList);

        for (SavePageDesignReqDto savePageDesignReqDto : savePageDesignReqDtoList) {
            PageDesignTime designTime = savePageDesignReqDto.getDesignTime();
            PageDsl pageDsl = dPageDslDao.selectByCodeAndApplication(savePageDesignReqDto.getCode(), savePageDesignReqDto.getPageDesignCode(), savePageDesignReqDto.getAdpApplication());
            pageDsl.setDsl(savePageDesignReqDto.getDsl());
            pageDsl.setDesignTime(designTime);
            pageDslList.add(pageDsl);

            UserHelper.fillEditInfo(pageDsl);
            dPageDslDao.save(pageDsl);
        }

        customControlEventManager.notify(CustomControlEventManager.EvenType.NEW_PAGE_DESIGN,pageDesignJo);

        RecordCommitLocal.add(new OperationRecordWrapper(pageDesign.getAdpApplication(), OperationTableNameConstants.LCDP_PAGE_DESIGN, pageDesignJo, OperationConstants.UPDATE));
    }

    @Override
    public void savePageDesignBasicInfo(SavePageDesignBasicInfoReqDto savePageDesignBasicInfoReqDto) {
        PageDesign pageDesign = dPageDesignDao.selectByCodeAndApplication(savePageDesignBasicInfoReqDto.getCode(), savePageDesignBasicInfoReqDto.getApplication());
        Assert.notNull(pageDesign,savePageDesignBasicInfoReqDto.getCode()+"没有查询到对应的记录");

        pageDesign.setName(savePageDesignBasicInfoReqDto.getName());
        pageDesign.setLang(savePageDesignBasicInfoReqDto.getLang());
        UserHelper.fillEditInfo(pageDesign);

        dPageDesignDao.save(pageDesign);
        RecordCommitLocal.add(new OperationRecordWrapper(pageDesign.getAdpApplication(), OperationTableNameConstants.LCDP_PAGE_DESIGN, JSON.parseObject(JSON.toJSONString(pageDesign)), OperationConstants.UPDATE));
    }

    @Override
    public void deletePageDesign(String code,String application) {
        PageDesign pageDesign = dPageDesignDao.selectByCodeAndApplication(code, application);
        Assert.notNull(pageDesign,code+"不存在对应的记录");

        dPageDslDao.deleteByPageDesignCodeAndApplication(code,application);
        dPageDesignDao.deleteByCodeAndApplication(code,application);

        RecordCommitLocal.add(new OperationRecordWrapper(pageDesign.getAdpApplication(), OperationTableNameConstants.LCDP_PAGE_DESIGN, JSON.parseObject(JSON.toJSONString(pageDesign)), OperationConstants.DELETE));
    }

    @Override
    public List<PageDesign> queryPageDesign(String application) {

        return dPageDesignDao.selectByApplication(application);
    }

    @Override
    public List<PageDsl> queryPageDslBasic(String application) {
        return dPageDslDao.selectBasicByApplication(application);
    }

    @Override
    public List<PageDsl> getPagesBasicInfo(String code, String application) {
        List<PageDsl> res = dPageDslDao.selectBasicByPageDesignCodeAndApplication(code,application);
        return res;
    }

    @Override
    public QueryPageDesignResDto queryBasicPageDesign(String code, String application) {
        PageDesign pageDesign = dPageDesignDao.selectByCodeAndApplication(code,application);
        Assert.notNull(pageDesign,code+"不存在对应的记录");
        QueryPageDesignResDto res = QueryPageDesignResDto.create(pageDesign);
        return res;
    }

    @Override
    public List<QueryPageDesignResDto> getAppPageDesigns(String application) {
        List<PageDesign> pageDesignList = dPageDesignDao.selectByApplication(application);
        List<PageDsl> pageDsls = dPageDslDao.selectBasicByApplication(application);
        Map<String, List<PageDsl>> pageDslMap = pageDsls.stream().collect(Collectors.groupingBy(PageDsl::getPageDesignCode));
        List<QueryPageDesignResDto> res = new ArrayList<>();

        for (PageDesign pageDesign : pageDesignList) {
            QueryPageDesignResDto queryPageDesignResDto = QueryPageDesignResDto.create(pageDesign);
            queryPageDesignResDto.setPageDslList(pageDslMap.get(pageDesign.getCode()));
            res.add(queryPageDesignResDto);
        }
        return res;
    }

    @Override
    public String copyPageDesign(String code, String application) {
        PageDesign pageDesign = dPageDesignDao.selectByCodeAndApplication(code, application);
        Assert.notNull(pageDesign,code+"没有对应的记录");
        List<PageDsl> pageDsls = dPageDslDao.selectByPageDesignCode(code, application);

        String deCode = buildGeneratorCodeUtil.createStandardCode(application, Boolean.TRUE.equals(pageDesign.getIsMobile()) ? CodeRuleTypeConstant.H5 : CodeRuleTypeConstant.DE);
        pageDesign.setCode(deCode);
        pageDesign.setPublishedTime(null);
        pageDesign.setObjectId(null);
        pageDesign.setName(pageDesign.getName()+"_copy");
        Map<String, MultiLanguageDTO> lang = pageDesign.getLang();
        if (!CollectionUtils.isEmpty(lang)&&lang.containsKey("name")){
            MultiLanguageDTO multiLanguageDTO = lang.get("name");
            multiLanguageDTO.setEn_US(multiLanguageDTO.getEn_US()+"_copy");
            multiLanguageDTO.setZh_TW(multiLanguageDTO.getZh_TW()+"_copy");
            multiLanguageDTO.setZh_CN(multiLanguageDTO.getZh_CN()+"_copy");
        }

        String oldDslCode = pageDesign.getDslCode();
        pageDesign.setDslCode(null);
        UserHelper.fillCreateInfo(pageDesign);
        UserHelper.fillEditInfo(pageDesign);

        for (PageDsl pageDsl : pageDsls) {
            String dslCode = buildGeneratorCodeUtil.createStandardCode(application, CodeRuleTypeConstant.DSL);
            if (pageDsl.getCode().equals(oldDslCode)){
                pageDesign.setDslCode(dslCode);
            }
            pageDsl.setObjectId(null);
            pageDsl.setCode(dslCode);
            pageDsl.setPageDesignCode(deCode);
            UserHelper.fillCreateInfo(pageDsl);
            UserHelper.fillEditInfo(pageDsl);

            dPageDslDao.save(pageDsl);
        }
        dPageDesignDao.save(pageDesign);

        JSONObject pageDesignJo = JSON.parseObject(JSON.toJSONString(pageDesign));
        pageDesignJo.put("pageDslList",pageDsls);

        RecordCommitLocal.add(new OperationRecordWrapper(pageDesign.getAdpApplication(), OperationTableNameConstants.LCDP_PAGE_DESIGN, pageDesignJo, OperationConstants.CREATE));
        return deCode;
    }
    @Override
    public PageDsl createPage(PageDslDto pageDslDto) {
        PageDesign pageDesign = dPageDesignDao.selectByCodeAndApplication(pageDslDto.getPageDesignCode(), pageDslDto.getApplication());
        Assert.notNull(pageDesign,pageDslDto.getPageDesignCode()+"不能存在");

        String dslCode = buildGeneratorCodeUtil.createStandardCode(pageDslDto.getApplication(), CodeRuleTypeConstant.DSL);
        PageDsl pageDsl = pageDslDto.createPageDsl();
        pageDsl.setCode(dslCode);

        dPageDslDao.insert(pageDsl);
        JSONObject pageDesignJo = JSON.parseObject(JSON.toJSONString(pageDesign));
        pageDesignJo.put("pageDslList", Arrays.asList(pageDsl));
        RecordCommitLocal.add(new OperationRecordWrapper(pageDesign.getAdpApplication(), OperationTableNameConstants.LCDP_PAGE_DESIGN, pageDesignJo, OperationConstants.CREATE));
        return pageDsl;
    }

    @Override
    public void deletePageDsl(String code, String pageDesignCode, String application) {
        PageDsl pageDsl = dPageDslDao.selectByCodeAndApplication(code, pageDesignCode, application);
        Assert.notNull(pageDsl,"不存在此页面");

        dPageDslDao.remove(pageDsl);
        PageDesign pageDesign = dPageDesignDao.selectByCodeAndApplication(pageDesignCode, application);

        JSONObject pageDesignJo = JSON.parseObject(JSON.toJSONString(pageDesign));
        pageDesignJo.put("pageDslList", Arrays.asList(pageDsl));

        RecordCommitLocal.add(new OperationRecordWrapper(pageDesign.getAdpApplication(), OperationTableNameConstants.LCDP_PAGE_DESIGN, pageDesignJo, OperationConstants.DELETE));
    }

    @Override
    public void updatePageDslBasic(PageDslDto pageDslDto) {
        PageDsl pageDsl = dPageDslDao.selectByCodeAndApplication(pageDslDto.getCode(), pageDslDto.getPageDesignCode(), pageDslDto.getApplication());
        Assert.notNull(pageDsl,"不存在此页面");

        pageDsl.setName(pageDslDto.getName());
        pageDsl.setLang(pageDslDto.getLang());

        UserHelper.fillEditInfo(pageDsl);
        dPageDslDao.save(pageDsl);

        PageDesign pageDesign = dPageDesignDao.selectByCodeAndApplication(pageDslDto.getPageDesignCode(), pageDslDto.getApplication());

        JSONObject pageDesignJo = JSON.parseObject(JSON.toJSONString(pageDesign));
        pageDesignJo.put("pageDslList", Arrays.asList(pageDsl));
        RecordCommitLocal.add(new OperationRecordWrapper(pageDesign.getAdpApplication(), OperationTableNameConstants.LCDP_PAGE_DESIGN, pageDesignJo, OperationConstants.UPDATE));
    }

    public void updateActionVersion() {
        List<PageDsl> pageDsls =  dPageDslDao.findAll();
        Set<String> actionIdSet = new HashSet<>();
        pageDsls.forEach(pageDsl -> {
            Dsl dsl = pageDsl.getDsl();
            List<JSONObject> dataConnectors = dsl.getDataConnectors();
            if (!ObjectUtils.isEmpty(dataConnectors)) {
                dataConnectors.forEach(dataConnector -> {
                    JSONObject extras = dataConnector.getJSONObject("extras");
                    if (!ObjectUtils.isEmpty(extras)) {
                        String serviceName = extras.getString("serviceName");
                        actionIdSet.add(serviceName);
                    }
                });
            }
        });
        List<String> actionIdList = actionIdSet.stream().collect(Collectors.toList());
        Map<String, ActionInfoResp> actionVersionMap = actionMetaDataService.getActionInfoByServiceName(new ActionVersionReq().setServiceNames(actionIdList));
        pageDsls.forEach(pageDsl -> {
            Dsl dsl = pageDsl.getDsl();
            List<JSONObject> dataConnectors = dsl.getDataConnectors();
            if (!ObjectUtils.isEmpty(dataConnectors)) {
                dataConnectors.forEach(dataConnector -> {
                    JSONObject extras = dataConnector.getJSONObject("extras");
                    if (!ObjectUtils.isEmpty(extras)) {
                        String serviceName = extras.getString("serviceName");
                        ActionInfoResp versionResp = actionVersionMap.get(serviceName);
                        if (!ObjectUtils.isEmpty(versionResp)) {
                            String version = versionResp.getServiceVersion();
                            extras.put("serviceVersion", version);
                            extras.put("source", "system");
                            dataConnector.put("extras", extras);
                        }
                    }
                });
            }
            dPageDslDao.save(pageDsl);
        });
    }
}
