package com.digiwin.athena.validate;

import cn.hutool.core.annotation.AnnotationUtil;
import cn.hutool.core.util.ReflectUtil;
import cn.hutool.core.util.StrUtil;
import io.swagger.v3.oas.annotations.media.Schema;
import jakarta.validation.ConstraintViolation;
import lombok.extern.slf4j.Slf4j;
import org.hibernate.validator.internal.engine.path.NodeImpl;
import org.hibernate.validator.internal.engine.path.PathImpl;

import java.lang.reflect.Field;
import java.util.LinkedList;
import java.util.Map;

@Slf4j
public class ValidationUtil {

    /**
     * 从 ConstraintViolation 转换字段路径。
     * 它尝试从其 Schema 注释中检索字段的名称或描述。
     *
     * @param violation ConstraintViolation 对象。
     * @return 转换后的字段名称，或转换失败或未找到 Schema 注释时的原始字段名称。
     */
    public static String transformFieldPath(ConstraintViolation<?> violation) {
        if (!(violation.getPropertyPath() instanceof PathImpl)) {
            log.warn("参数校验不通过信息回显：参数校验实现不为Hibernate Validator!type:{}", violation.getClass());
            return violation.getPropertyPath().toString();
        }

        //noinspection PatternVariableCanBeUsed
        PathImpl path = (PathImpl) violation.getPropertyPath();
        NodeImpl leafNode = path.getLeafNode();

        LinkedList<String> fieldNameList = new LinkedList<>();
        do {
            String fieldName = leafNode.getName();
            String fieldDesc = getFieldDesc(violation, leafNode, fieldName);

            fieldNameList.addFirst(fieldDesc);
            leafNode = leafNode.getParent();
        } while (leafNode != null && leafNode.getParent().getValue() != null);

        return StrUtil.join("/", fieldNameList);
    }

    private static String getFieldDesc(ConstraintViolation<?> violation, NodeImpl leafNode, String fieldName) {
        try {
            Field field = getField(leafNode, fieldName);
            if (field == null) {
                log.warn("参数校验不通过信息回显：字段不存在! propertyPath:{},fieldName:{},parent:{}",
                        violation.getPropertyPath(), fieldName, leafNode.getParent().getValue());
                return fieldName;
            }

            Schema schema = AnnotationUtil.getAnnotation(field, Schema.class);
            if (schema == null) {
                return fieldName;
            }

            fieldName = StrUtil.blankToDefault(schema.name(), schema.description());
        } catch (Exception ex) {
            log.error("参数校验不通过信息回显：字段名称转换异常！", ex);
        }

        return fieldName;
    }


    private static Field getField(NodeImpl leafNode, String fieldName) {
        Object parentVal = leafNode.getParent().getValue();
        if (parentVal == null) {
            return null;
        }

        if (leafNode.getKey() != null && parentVal instanceof Map) {
            parentVal = ((Map<?, ?>) parentVal).get(leafNode.getKey());
        }

        return ReflectUtil.getField(parentVal.getClass(), fieldName);
    }
}
