package com.digiwin.athena.controller;

import com.digiwin.athena.auth.annotation.FuncAuth;
import com.digiwin.athena.base.ResultBean;
import com.digiwin.athena.domain.PageDsl;
import com.digiwin.athena.dto.*;
import com.digiwin.athena.http.distribution.FxApiHelper;
import com.digiwin.athena.http.distribution.FxRegisterJobInfoResDto;
import com.digiwin.athena.http.distribution.GetFxJobListReqDto;
import com.digiwin.athena.publish.Publish;
import com.digiwin.athena.publish.dto.PublishParam;
import com.digiwin.athena.service.DPageDesignService;
import io.jsonwebtoken.lang.Assert;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.NotEmpty;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.Date;
import java.util.List;

@RestController
@RequestMapping("/pageDesign/d")
@Tag(name = "设计时作业",description = "6e6954a1-8237-4526-9114-9c1238b0822b")
@Validated
public class DPageDesignController {

    @Autowired
    private DPageDesignService dPageDesignService;
    @Autowired
    @Qualifier("pageDesignPublish")
    private Publish publish;
    @Autowired
    private FxApiHelper fxApiHelper;

    @PostMapping("/createPageDesign")
    @Operation(summary = "创建作业")
    public ResultDto<String>  createPageDesign(@RequestBody @Validated CreatePageDesignDto createPageDesignDto){
        String code = dPageDesignService.createPageDesign(createPageDesignDto);
        return ResultDto.createSuccess(code);
    }

    @PostMapping("/createPageDsl")
    public ResultDto<PageDsl> createPageDsl(@RequestBody PageDslDto pageDslDto){
        PageDsl res = dPageDesignService.createPage(pageDslDto);
        return ResultDto.createSuccess(res);
    }

    @GetMapping("/deletePageDsl")
    public ResultDto<Boolean> deletePageDsl(String code,String pageDesignCode,String application){
        dPageDesignService.deletePageDsl(code,pageDesignCode,application);
        return ResultDto.createSuccess(true);
    }

    @PostMapping("/updatePageDslBasic")
    public ResultDto<Boolean> updatePageDslBasic(@RequestBody PageDslDto pageDslDto){
        dPageDesignService.updatePageDslBasic(pageDslDto);
        return ResultDto.createSuccess(true);
    }


    @GetMapping("/queryBasicPageDesign")
    @Operation(summary = "查询作业基础信息")
    public ResultDto<QueryPageDesignResDto> queryBasicPageDesign(@NotEmpty(message = "code不能为空") String code,
                                                            @NotEmpty(message = "application不能为空") String application){
        QueryPageDesignResDto res = dPageDesignService.queryBasicPageDesign(code,application);
        return ResultDto.createSuccess(res);
    }


    @GetMapping("/queryPageDesign")
    @Operation(summary = "查询作业")
    public ResultDto<QueryPageDesignResDto> queryPageDesign(@NotEmpty(message = "code不能为空") String code,
                                                            @NotEmpty(message = "application不能为空") String application){
        QueryPageDesignResDto res = dPageDesignService.queryPageDesign(code,application);
        return ResultDto.createSuccess(res);
    }

    @PostMapping("/savePageDesign")
    @Operation(summary = "保存界面设计")
    public ResultDto<Boolean> savePageDesign(@RequestBody @Valid List<SavePageDesignReqDto> savePageDesignReqDtoList){
        Assert.isTrue(!CollectionUtils.isEmpty(savePageDesignReqDtoList),"保存界面数据不能为空");
        dPageDesignService.savePageDesign(savePageDesignReqDtoList);
        return ResultDto.createSuccess(true);
    }

    @PostMapping("/savePageDesignBasicInfo")
    @Operation(summary = "修改作业基础信息")
    public ResultDto<Boolean> savePageDesignBasicInfo(@RequestBody @Validated SavePageDesignBasicInfoReqDto savePageDesignBasicInfoReqDto) {
        dPageDesignService.savePageDesignBasicInfo(savePageDesignBasicInfoReqDto);
        return ResultDto.createSuccess(true);
    }

    @GetMapping("/deletePageDesign")
    @Operation(summary = "删除作业")
    public ResultDto<Boolean> deletePageDesign(@NotEmpty(message = "code不能为空") String code,
                                               @NotEmpty(message = "application不能为空") String application){
        dPageDesignService.deletePageDesign(code,application);
        return ResultDto.createSuccess(true);
    }

    @PostMapping("/singlePublish")
    @Operation(summary = "发布作业")
    @FuncAuth(resourceType = "application", resourceId = "Var:PublishParam#application",actions = (":*"))
    public ResultBean<Date> singlePublish(@RequestBody @Validated PublishParam publishParam) {
        publish.publishOne(publishParam);
        QueryPageDesignResDto queryPageDesignResDto = dPageDesignService.queryBasicPageDesign(publishParam.getCode(), publishParam.getApplication());
        return ResultBean.success(queryPageDesignResDto.getPublishedTime());
    }

    @GetMapping("/getPagesBasicInfo")
    @Operation(summary = "查询指定作业下页面信息")
    public ResultDto<List<PageDsl>> getPagesBasicInfo(@Schema(description = "作业code") @NotEmpty(message = "作业code不能为空") String code,
                                             @Schema(description = "应用code") @NotEmpty(message = "应用code不能为空") String application){
        List<PageDsl> res = dPageDesignService.getPagesBasicInfo(code,application);
        return ResultDto.createSuccess(res);
    }

    @GetMapping("/getAppPageDesigns")
    @Operation(summary = "查询应用下所有作业")
    public ResultDto<List<QueryPageDesignResDto>> getAppPageDesigns(@Schema(description = "应用code") @NotEmpty(message = "应用code不能为空") String application){
        List<QueryPageDesignResDto> res = dPageDesignService.getAppPageDesigns(application);
        return ResultDto.createSuccess(res);
    }

    @GetMapping("/copyPageDesign")
    @FuncAuth(resourceType = "application", resourceId = "1",actions = (":*"))
    @Operation(summary = "复制作业")
    public ResultDto<String> copyPageDesign(@NotEmpty(message = "作业code不能为空") String code,
                                            @NotEmpty(message = "应用code不能为空") String application){
        String newCode = dPageDesignService.copyPageDesign(code,application);
        return ResultDto.createSuccess(newCode);
    }

    @GetMapping("/test1")
    public  List<FxRegisterJobInfoResDto> test1(){
//        RegisterFxJobReqDto fxJobMenuParam = RegisterFxJobReqDto.createFxJobMenuParam("DE_c9113682100013f7", "rfvfrq");
//        fxApiHelper.registerJob(fxJobMenuParam);
//        DeleteFxJobReqDto deleteFxJobReqDto = new DeleteFxJobReqDto();
//        deleteFxJobReqDto.setAah001("DE_c9113682100013f6");
        GetFxJobListReqDto getFxJobListReqDto = new GetFxJobListReqDto();
        getFxJobListReqDto.setGlikesel("DE_c9113682100013f7");
        return fxApiHelper.getJobList(getFxJobListReqDto);
    }
}
