package com.digiwin.athena.service.codeStandard;

import com.alibaba.fastjson.JSON;
import com.digiwin.athena.base.BusinessException;
import com.digiwin.athena.constant.CodeRuleTypeConstant;
import com.digiwin.athena.mongodb.domain.other.PresetData;
import com.digiwin.athena.mongodb.repository.MongoSystemRepositoryDecorator;
import com.digiwin.athena.service.codeStandard.dto.GenerateCodeDto;
import com.digiwin.athena.service.codeStandard.dto.RuleConfig;
import com.digiwin.athena.utils.CurThreadInfoUtils;
import com.digiwin.athena.utils.SnowflakeToHexStrUtil;
import com.digiwin.athena.utils.pageDesignModel.BusinessResourceTreeUtil;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Service;

import java.util.Collections;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

@Service
@Slf4j
public class CodeGeneratorService {

    // 存储所有的 handler
    @Autowired
    private Map<String, CodeGenerateHandler> handlers;

    @Autowired
    private ObjectMapper objectMapper;


    @Autowired
    private MongoSystemRepositoryDecorator mongoSystemRepositoryDecorator;

    public String generateCode(GenerateCodeDto generateCodeDto)  {

        log.info("generateCode start,generateCodeDto param is:{}", JSON.toJSONString(generateCodeDto));

        RuleConfig ruleConfig = getRulesFromDB(generateCodeDto.getTypeCode());

        assert ruleConfig != null;
        String code = createCodeByRule(ruleConfig, generateCodeDto.getParamMap());
        if (CurThreadInfoUtils.getIndividualCase()){
            code = CodeRuleTypeConstant.UC + "_" + code;
        }
        return code;

    }



    /**
     * 根据规则生成code
     *
     * @param ruleConfig 规则
     * @param paramMap paramMap
     */
    private String createCodeByRule(RuleConfig ruleConfig, Map<String, String> paramMap) {

        StringBuilder finalCode = new StringBuilder();
        for (RuleConfig.CodeRule rule : ruleConfig.getRules()) {
            CodeGenerateHandler handler = handlers.get(rule.getGenerateHandler());
            if (handler == null) {
                throw new BusinessException("Handler not found for: " + rule.getGenerateHandler());
            }

            Map<String, String> params = Optional.ofNullable(rule.getParams())
                    .orElseGet(Collections::emptyList)
                    .stream()
                    .filter(Objects::nonNull)
                    .collect(Collectors.toMap(
                            RuleConfig.CodeRuleParam::getKey,
                            RuleConfig.CodeRuleParam::getValue
                    ));

            finalCode.append(handler.generateCode(params, paramMap));
        }
        String generatedCode = finalCode.toString();
        log.info("Generated standard code is: {}", generatedCode);
        return generatedCode;

    }


    /**
     * 查询code生成规则
     *
     * @param typeCode 类型code
     */
    private RuleConfig getRulesFromDB(String typeCode) {
        Criteria criteria = Criteria.where("key").is(typeCode);
        PresetData presetData = mongoSystemRepositoryDecorator.findOne(Query.query(criteria), PresetData.class);

        if (presetData == null) {
            log.warn("No configuration found for typeCode: {}", typeCode);
            throw new BusinessException("No configuration found for typeCode");
        }

        Object obj = presetData.getObj();
        log.info("Fetching configuration for typeCode: {}, returned data: {}", typeCode, JSON.toJSONString(obj));

        return objectMapper.convertValue(obj, RuleConfig.class);
    }


    /**
     * 新增规则入库
     *
     * @param generateCodeDto generateCodeDto
     */
    public void insertData(GenerateCodeDto generateCodeDto) {
        RuleConfig ruleConfig = generateCodeDto.getRuleConfig();

        if (ruleConfig == null || ruleConfig.getCode() == null) {
            throw new BusinessException("RuleConfig or its code cannot be null.");
        }

        PresetData presetData = new PresetData();
        presetData.setKey(ruleConfig.getCode());
        presetData.setObj(ruleConfig);

        try {
            mongoSystemRepositoryDecorator.save(presetData);
        } catch (Exception e) {
            log.error("Failed to save PresetData: ", e);
            throw new BusinessException("Failed to save PresetData", e);
        }
    }

    /**
     * 删除规则数据
     *
     * @param key key
     */
    public void deleteData(String key) {

        try {
            PresetData presetData =
                    mongoSystemRepositoryDecorator.findOne(BusinessResourceTreeUtil.buildQuery("key", key), PresetData.class);

            if (presetData == null) {
                log.warn("No configuration found for key: {}", key);
            } else {
                log.info("Deleting PresetData: {}", JSON.toJSONString(presetData));
                mongoSystemRepositoryDecorator.delete(presetData);
            }
        } catch (Exception e) {
            log.error("Failed to delete PresetData: ", e);
            throw new BusinessException("Failed to delete PresetData", e);
        }
    }


    public String generateCodeOfFront(GenerateCodeDto generateCodeDto) {
        Map<String, String> paramMap = generateCodeDto.getParamMap();
        String str = SnowflakeToHexStrUtil.snowflakeToHex();
        // 雪花算法生成的id转16进制
        paramMap.put(CodeRuleTypeConstant.INPUT_CODE, str);
        return generateCode(generateCodeDto);

    }



}
