package com.digiwin.athena.service.impl;

import cn.hutool.core.util.IdUtil;
import com.digiwin.athena.dao.FileStoreMongoDao;
import com.digiwin.athena.domain.FileStore;
import com.digiwin.athena.dto.FxUploadFileAResDto;
import com.digiwin.athena.http.distribution.FxApiHelper;
import com.digiwin.athena.http.distribution.UploadFileResDto;
import com.digiwin.athena.http.distribution.UploadParamDto;
import com.digiwin.athena.service.FileStoreService;
import com.digiwin.athena.utils.GzipUtil;
import com.digiwin.athena.utils.MD5Util;
import com.digiwin.athena.utils.user.UserHelper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.web.multipart.MultipartFile;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

@Service
@Slf4j
public class FileStoreServiceImpl implements FileStoreService {

    private static final Integer CHUNK_SIZE = 15 * 1024 * 1024;

    @Autowired
    private FileStoreMongoDao fileStoreMongoDao;
    @Autowired
    private FxApiHelper fxApiHelper;


    @Override
    public String fxUploadFile(MultipartFile file,String tableNo,String dslCode) throws IOException {
        UploadParamDto uploadParamDto = new UploadParamDto();
        ByteArrayResource byteArrayResource = new ByteArrayResource(file.getBytes()){
            @Override
            public String getFilename() {
                return file.getOriginalFilename();
            }
        };
        uploadParamDto.setAttach(byteArrayResource);
        uploadParamDto.setTableNo(tableNo);
        uploadParamDto.setDocumentNo(dslCode);
        UploadFileResDto res = fxApiHelper.upload(uploadParamDto);
        return res.getUrl();

    }


    @Override
    public FxUploadFileAResDto fxUploadFileA(MultipartFile file, String programCode) throws IOException {
        UploadParamDto uploadParamDto = new UploadParamDto();
        ByteArrayResource byteArrayResource = new ByteArrayResource(file.getBytes()){
            @Override
            public String getFilename() {
                return file.getOriginalFilename();
            }
        };
        uploadParamDto.setAttach(byteArrayResource);
        uploadParamDto.setProgramCode(programCode);
        UploadFileResDto uploadFileResDto = fxApiHelper.fxUploadFile(uploadParamDto);

        FxUploadFileAResDto res = new FxUploadFileAResDto();
        res.setSize(uploadFileResDto.getFileSize());
        res.setUrl(uploadFileResDto.getUrl());
        res.setName(uploadFileResDto.getFileName());
        res.setFileId(uploadFileResDto.getId().toString());
        res.setFileType(uploadFileResDto.getFileType());

        return res;
    }

    @Override
    public String uploadFile(InputStream inputStream, String fileName, String fileType){
        try {
            byte[] buffer = new byte[8192];
            int bytesRead;
            try (ByteArrayOutputStream outputStream = new ByteArrayOutputStream()) {
                while ((bytesRead = inputStream.read(buffer)) != -1) {
                    outputStream.write(buffer, 0, bytesRead);
                }
                byte[] byteArray = outputStream.toByteArray();

                InputStream stream1 = new ByteArrayInputStream(byteArray);
                String fileHash = MD5Util.getFileHash(stream1);
                //判断是否已经存在相同的文件了
                FileStore existFileStore = fileStoreMongoDao.selectFileBasicInfoByMD5(fileHash);
                if (existFileStore != null) {
                    Long count = fileStoreMongoDao.selectExistByMD5(fileHash);
                    if (existFileStore.getDataSheet().equals(count.intValue())){
                        return existFileStore.getFileId();
                    }
                }

                byte[] compress = GzipUtil.compress(byteArray);
                List<byte[]> contentArr = splitByteArray(compress);

                String fileId = IdUtil.fastSimpleUUID();
                for (int i = 0; i < contentArr.size(); i++) {
                    byte[] c = contentArr.get(i);

                    FileStore fileStore = new FileStore();
                    fileStore.setFileId(fileId);
                    fileStore.setFileName(fileName);
                    fileStore.setDataSheet(contentArr.size());
                    fileStore.setFileType(fileType);
                    fileStore.setContent(c);
                    fileStore.setOrder(i);
                    fileStore.setMd5(fileHash);
                    UserHelper.fillEditInfo(fileStore);
                    UserHelper.fillCreateInfo(fileStore);
                    fileStoreMongoDao.insert(fileStore);
                }
                return fileId;
            }


        } catch (IOException e) {
            log.error("文件读取失败:"+e.getMessage(),e);
            throw new RuntimeException("文件读取失败");
        }finally {
            try {
                inputStream.close();
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

    @Override
    public FileStore getFile(String fileId) throws IOException {
        List<FileStore> fileStores = fileStoreMongoDao.selectByFileId(fileId);
        Assert.isTrue(!fileStores.isEmpty(),"不存在对应的文件");

        int totalLength = 0;
        for (FileStore fileStore : fileStores) {
            totalLength+=fileStore.getContent().length;
        }

        byte[] result = new byte[totalLength];
        int currentPos = 0;
        for (FileStore fs : fileStores) {
            byte[] content = fs.getContent();
            if (content != null) {
                System.arraycopy(content, 0, result, currentPos, content.length);
                currentPos += content.length;
            }
        }

        byte[] decompress = GzipUtil.decompress(result);

        FileStore old = fileStores.get(0);
        FileStore fileStore = new FileStore();
        fileStore.setFileName(old.getFileName());
        fileStore.setFileType(old.getFileType());
        fileStore.setFileId(old.getFileId());
        fileStore.setContent(decompress);
        return fileStore;
    }


    public static List<byte[]> splitByteArray(byte[] data) {
        List<byte[]> chunks = new ArrayList<>();
        int offset = 0;
        while (offset < data.length) {
            int end = Math.min(offset + CHUNK_SIZE, data.length);
            byte[] chunk = Arrays.copyOfRange(data, offset, end);
            chunks.add(chunk);
            offset = end;
        }
        return chunks;
    }
}
