package com.digiwin.athena.service.codeStandard;

import com.digiwin.athena.base.BusinessException;
import com.digiwin.athena.constant.CodeRuleTypeConstant;
import com.digiwin.athena.service.codeStandard.dto.GenerateCodeDto;
import com.digiwin.athena.utils.MD5;
import com.digiwin.athena.utils.SnowflakeToHexStrUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

@Slf4j
@Service
public class BuildGeneratorCodeUtil {

    @Autowired
    private CodeGeneratorService codeGeneratorService;

    /**
     * 事件生成标准化code
     *
     * @param application  应用code
     * @param endCode      编码结尾code，分为自动生成和手动输入
     * @param autoOrCustom 动生成和手动输入
     */
    public String createStandardCodeOfEvent(String application, String endCode, String autoOrCustom) {
        try {
            GenerateCodeDto dto = new GenerateCodeDto();
            dto.setTypeCode(CodeRuleTypeConstant.EVENT_TYPE);
            Map<String, String> map = new HashMap<>();
            map.put(CodeRuleTypeConstant.APPLICATION, application);
            map.put(autoOrCustom, "");
            map.put(CodeRuleTypeConstant.INPUT_CODE, endCode);
            dto.setParamMap(map);
            return codeGeneratorService.generateCode(dto);
        } catch (Exception e) {
            log.error("事件生成标准化code异常,异常信息为:{}", e.getMessage(), e);
            throw new BusinessException("事件生成标准化code异常:" + e.getMessage());
        }
    }


    /**
     * 生成标准化code
     *
     * @param application 应用code
     * @param type        类型
     */
    public String createStandardCode(String application, String type) {

        try {
            GenerateCodeDto dto = new GenerateCodeDto();
            dto.setTypeCode(CodeRuleTypeConstant.DEFAULT_TYPE);
            Map<String, String> map = new HashMap<>();
            map.put(CodeRuleTypeConstant.APPLICATION, application);
            map.put(CodeRuleTypeConstant.ABBREVIATION, type);
            map.put(CodeRuleTypeConstant.INPUT_CODE, SnowflakeToHexStrUtil.snowflakeToHex());
            dto.setParamMap(map);
            return codeGeneratorService.generateCode(dto);
        } catch (Exception e) {
            log.error("生成标准化code异常,异常信息为:{},对应类型为:{}", e.getMessage(), type, e);
            throw new BusinessException("生成标准化code异常:" + e.getMessage());
        }
    }

    /**
     * 查询方案生成标准化code
     *
     * @param application 应用code
     * @param inputCode   页面输入code
     */
    public String createStandardCodeDataView(String application, String type, String inputCode) {

        try {
            GenerateCodeDto dto = new GenerateCodeDto();
            dto.setTypeCode(CodeRuleTypeConstant.DEFAULT_TYPE);
            Map<String, String> map = new HashMap<>();
            map.put(CodeRuleTypeConstant.APPLICATION, application);
            map.put(CodeRuleTypeConstant.ABBREVIATION, type);
            map.put(CodeRuleTypeConstant.INPUT_CODE, inputCode);
            dto.setParamMap(map);
            return codeGeneratorService.generateCode(dto);
        } catch (Exception e) {
            log.error("查询方案生成标准化code异常,异常信息为:{}", e.getMessage(), e);
            throw new BusinessException("查询方案生成标准化code异常:" + e);
        }
    }


    /**
     * 个案生成标准化code
     *
     * @param application 应用code
     * @param type        类型
     */
    public String individualCase(String application, String type) {

        return CodeRuleTypeConstant.UC + "_" + createStandardCode(application, type);
    }


    /**
     * 公有的定制配置生成标准化code
     *
     * @param tenantId 租户id
     * @param type     类型
     */
    public String createStandardCodeConfig(String tenantId, String type) {

        try {
            GenerateCodeDto dto = new GenerateCodeDto();
            dto.setTypeCode(CodeRuleTypeConstant.CUSTOM_CONFIG_TYPE);
            Map<String, String> map = new HashMap<>();
            map.put(CodeRuleTypeConstant.TENANT_ID, tenantId);
            map.put(CodeRuleTypeConstant.ABBREVIATION, type);
            map.put(CodeRuleTypeConstant.INPUT_CODE, SnowflakeToHexStrUtil.snowflakeToHex());
            dto.setParamMap(map);
            return codeGeneratorService.generateCode(dto);
        } catch (Exception e) {
            log.error("公有的定制配置生成标准化code异常,异常信息为:{},对应类型为:{}", e.getMessage(), type, e);
            throw new BusinessException("公有的定制配置生成标准化code异常:" + e.getMessage());
        }
    }

    public String createAssetId(String code,String type) {
        String md5 = MD5.getMD5(code + "_" + type);
        return CodeRuleTypeConstant.AT + "_" + MD5.getMD5(md5).substring(0, 16);
    }

    public String generateActivityCode(String type) {
        // 生成UUID
        String uuid = UUID.randomUUID().toString();

        switch (type) {
            case "manual":
            case "invokeSubProject":
                return "StartSubProjectNode_" + uuid;
            case "sendMessage":
                return "SendMessage_" + uuid;
            case "getAllUsers":
                return "GetAllUsers_" + uuid;
            case "esp":
                return "InvokeEsp_" + uuid;
            case "http":
                return "HttpNode_" + uuid;
            case "composite":
                return "GroupByData_" + uuid;
            case "notice":
                return "Notice_" + uuid;
            case "script":
                return "ScriptNode_" + uuid;
            case "approve":
                return "ApproveNode_" + uuid;
            case "createData":
                return "DataGroup_" + uuid;
            case "wait":
                return "WaitingNode_" + uuid;
            case "groovy":
                return "GroovyNode_" + uuid;
        }
        log.info("当前应用没有匹配到类型:{}",type);
        return type + "_" + uuid;
    }


}
