package com.digiwin.athena.dao;


import com.digiwin.athena.mongodb.domain.appCustomConfig.ApplicationCustomConfig;
import com.digiwin.athena.mongodb.domain.appCustomConfig.CustomControlRes;
import com.digiwin.athena.mongodb.domain.appCustomConfig.SysApplicationCustomConfig;
import com.digiwin.athena.mongodb.domain.appCustomConfig.SysApplicationCustomConfigShare;
import com.digiwin.athena.mongodb.repository.MongoMultiRepositoryDecorator;
import com.digiwin.athena.mongodb.repository.MongoSystemRepositoryDecorator;
import com.digiwin.athena.utils.SwitchDatabaseUtils;
import com.digiwin.athena.utils.UpdateUtils;
import jakarta.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Field;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Repository;

import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

@Slf4j
@Repository
public class ApplicationCustomConfigV2MongoDao {

    @Resource
    private MongoMultiRepositoryDecorator mongoMultiRepositoryDecorator;

    @Resource
    private MongoSystemRepositoryDecorator mongoSystemRepositoryDecorator;

    public void insert(ApplicationCustomConfig applicationCustomConfig) {
        mongoMultiRepositoryDecorator.insert(applicationCustomConfig);
    }

    public void updateById(ApplicationCustomConfig applicationCustomConfig) {
        Criteria criteria = Criteria.where("id").is(applicationCustomConfig.getId()).and("application").is(applicationCustomConfig.getApplication());
        Query query = new Query(criteria);
        mongoMultiRepositoryDecorator.updateFirst(query, UpdateUtils.getUpdate(applicationCustomConfig), ApplicationCustomConfig.class);
    }

    public List<ApplicationCustomConfig> find(Query query) {
        return mongoMultiRepositoryDecorator.find(query, ApplicationCustomConfig.class);
    }

    public ApplicationCustomConfig findOne(String adpCode) {
        List<ApplicationCustomConfig> applicationCustomConfigList = find(new Query(Criteria.where("adpCode").is(adpCode)));
        return applicationCustomConfigList.isEmpty() ? null : applicationCustomConfigList.get(0);
    }

    public void deleteByAdpCode(String adpCode) {

        mongoMultiRepositoryDecorator.remove(new Query(Criteria.where("adpCode").is(adpCode)), ApplicationCustomConfig.class);
    }

    /**
     * 检查该是否已经存在该插件名称
     *
     * @return
     */
    public boolean checkIsExistPluginName(String name) {
        Criteria criteria = Criteria.where("name").is(name);
        return judgeSelectResult(criteria);
    }

    /**
     * 检查该是否已经存在该插件名称，排除当前id
     *
     * @return
     */
    public boolean checkIsExistPluginNameExcludeCurId(String name,String adpCode) {
        Criteria criteria = Criteria.where("name").is(name)
                .and("adpCode").ne(adpCode);
        return judgeSelectResult(criteria);
    }

    public ApplicationCustomConfig findByAdpCode(String adpCode) {
        Criteria criteria = Criteria.where("adpCode").is(adpCode);

        return mongoMultiRepositoryDecorator.findOne(new Query(criteria),ApplicationCustomConfig.class);
    }

    public List<SysApplicationCustomConfig> findSysByAdpTenantId(String tenantId) {
        Criteria criteria = Criteria.where("adpTenantId").is(tenantId);
        Query query = simpleInfoListQueryCondition(criteria);
        return mongoSystemRepositoryDecorator.find(query,SysApplicationCustomConfig.class);
    }

    public List<SysApplicationCustomConfigShare> findSysShareByShareTenantId(String tenantId) {
        Criteria criteria = Criteria.where("shareTenantId").is(tenantId);
        Query query = new Query(criteria);
        return mongoSystemRepositoryDecorator.find(query,SysApplicationCustomConfigShare.class);
    }

    public List<SysApplicationCustomConfigShare> findSysShareByAdpCode(String adpCode) {
        Criteria criteria = Criteria.where("adpCode").is(adpCode);
        Query query = new Query(criteria);
        return mongoSystemRepositoryDecorator.find(query,SysApplicationCustomConfigShare.class);
    }

    public SysApplicationCustomConfig findSysByAdpCode(String adpCode) {
        Criteria criteria = Criteria.where("adpCode").is(adpCode);

        return mongoSystemRepositoryDecorator.findOne(new Query(criteria),SysApplicationCustomConfig.class);
    }

    public void insertSys(SysApplicationCustomConfig sysApplicationCustomConfig) {
        mongoSystemRepositoryDecorator.insert(sysApplicationCustomConfig);
    }

    public void insertSysShare(SysApplicationCustomConfigShare sysApplicationCustomConfigShare) {
        mongoSystemRepositoryDecorator.insert(sysApplicationCustomConfigShare);
    }

    public List<ApplicationCustomConfig> selectListByApplication(String application) {
        Criteria criteria = Criteria.where("application").is(application);
        Query query = simpleInfoListQueryCondition(criteria);

        return mongoMultiRepositoryDecorator.find(query,ApplicationCustomConfig.class);
    }

    private Query simpleInfoListQueryCondition(Criteria criteria) {
        Query query = new Query(criteria).with(Sort.by(Sort.Direction.DESC, "createDate"));
        Field fields = query.fields();
        fields.include("adpCode");
        fields.include("name");
        fields.include("description");
        fields.include("lang");
        fields.include("adpTenantId");
        fields.include("path");
        fields.include("chunkName");
        fields.include("isMF");
        fields.include("customControl.type");
        fields.include("customControl.title");
        fields.include("customControl.lang");
        return query;
    }

    public void save(ApplicationCustomConfig applicationCustomConfig) {
        mongoMultiRepositoryDecorator.save(applicationCustomConfig);
    }

    public boolean checkIsExistControlType(List<String> typeList) {
        Criteria criteria = Criteria.where("customControl.type").in(typeList);
        return judgeSelectResult(criteria);
    }

    public List<String> selectExistControlType(List<String> typeList) {
        Criteria criteria = Criteria.where("customControl.type").in(typeList);
        return getRepeatControlType(criteria,typeList);
    }

    public boolean checkIsExistControlTypeExcludeCur(List<String> typeList, String adpCode) {
        Criteria criteria = Criteria.where("customControl.type").in(typeList).and("adpCode").ne(adpCode);
        return judgeSelectResult(criteria);
    }

    private boolean judgeSelectResult(Criteria criteria) {
        Query query = new Query(criteria).limit(1);
        long count2 = mongoSystemRepositoryDecorator.count(query, SysApplicationCustomConfig.class);
        if (count2 > 0) {
            return true;
        }
        Long count1 = SwitchDatabaseUtils.switchStandardDatabase(()->mongoMultiRepositoryDecorator.count(query, ApplicationCustomConfig.class));
        if (count1>0){
            return true;
        }

        count1 = SwitchDatabaseUtils.switchIndividualDatabase(()->mongoMultiRepositoryDecorator.count(query, ApplicationCustomConfig.class));
        return count1 > 0;
    }

    public List<String> selectExistControlTypeExcludeCur(List<String> typeList, String adpCode) {
        Criteria criteria = Criteria.where("customControl.type").in(typeList).and("adpCode").ne(adpCode);
        return getRepeatControlType(criteria,typeList);
    }

    public List<SysApplicationCustomConfigShare> selectExistShareTenantId(String adpCode) {
        Criteria criteria = Criteria.where("adpCode").is(adpCode).and("status").is("Y");
        Query query = new Query(criteria);
        List<SysApplicationCustomConfigShare> sysApplicationCustomConfigs = mongoSystemRepositoryDecorator.find(query, SysApplicationCustomConfigShare.class);
        return sysApplicationCustomConfigs;
    }

    private List<String> getRepeatControlType(Criteria criteria,List<String> typeList) {
        Query query = new Query(criteria);
        Field fields = query.fields();
        fields.include("customControl.type");
        Set<String> typeSet = new HashSet<>(typeList);

        List<SysApplicationCustomConfig> sysApplicationCustomConfigs = mongoSystemRepositoryDecorator.find(query, SysApplicationCustomConfig.class);
        List<ApplicationCustomConfig> applicationCustomConfigs = SwitchDatabaseUtils.switchStandardDatabase(()->mongoMultiRepositoryDecorator.find(query, ApplicationCustomConfig.class));
        List<ApplicationCustomConfig> individualApplicationCustomConfigs = SwitchDatabaseUtils.switchIndividualDatabase(()->mongoMultiRepositoryDecorator.find(query, ApplicationCustomConfig.class));

        List<List<CustomControlRes>> controlList = applicationCustomConfigs.stream().map(ApplicationCustomConfig::getCustomControl).collect(Collectors.toList());
        controlList.addAll(individualApplicationCustomConfigs.stream().map(ApplicationCustomConfig::getCustomControl).collect(Collectors.toList()));
        controlList.addAll(sysApplicationCustomConfigs.stream().map(SysApplicationCustomConfig::getCustomControl).collect(Collectors.toList()));

        return controlList.stream().flatMap(List::stream).map(CustomControlRes::getType).filter(t -> typeSet.contains(t)).distinct().collect(Collectors.toList());
    }

    public void saveSys(SysApplicationCustomConfig sysApplicationCustomConfig) {
        mongoSystemRepositoryDecorator.save(sysApplicationCustomConfig);
    }

    public void deleteSysByAdpCode(String adpCode) {
        mongoSystemRepositoryDecorator.remove(new Query(Criteria.where("adpCode").is(adpCode)), SysApplicationCustomConfig.class);
    }

    public List<ApplicationCustomConfig> selectCustomControlByApplication(String application) {
        Criteria criteria = Criteria.where("application").is(application);
        Query query = new Query(criteria);
        Field fields = query.fields();
        fields.include("customControl");
        fields.include("name");
        fields.include("lang");

        return mongoMultiRepositoryDecorator.find(query,ApplicationCustomConfig.class);
    }

    public List<SysApplicationCustomConfig> selectSysCustomControlByTenantId(String tenantId) {
        Criteria criteria = Criteria.where("adpTenantId").is(tenantId);
        Query query = new Query(criteria);
        Field fields = query.fields();
        fields.include("customControl");
        fields.include("name");
        fields.include("lang");

        return mongoSystemRepositoryDecorator.find(query,SysApplicationCustomConfig.class);
    }

    public List<ApplicationCustomConfig> selectAppControlType(String application) {
        Criteria criteria1 = Criteria.where("application").is(application);
        Query query1 = new Query(criteria1);
        Field fields1 = query1.fields();
        fields1.include("adpCode");
        fields1.include("customControl.type");

        return mongoMultiRepositoryDecorator.find(query1, ApplicationCustomConfig.class);
    }

    public List<SysApplicationCustomConfig> selectSysControlType(String tenantId) {
        Criteria criteria2 = Criteria.where("adpTenantId").is(tenantId);
        Query query2 = new Query(criteria2);
        Field fields2 = query2.fields();
        fields2.include("adpCode");
        fields2.include("customControl.type");
        return mongoSystemRepositoryDecorator.find(query2,SysApplicationCustomConfig.class);
    }

    public List<SysApplicationCustomConfig> selectSysExcludeCustomControl(List<String> adpCodes) {
        Criteria criteria = Criteria.where("adpCode").in(adpCodes);
        Query query = new Query(criteria);
        Field fields = query.fields();
        fields.exclude("customControl");
        return mongoSystemRepositoryDecorator.find(query, SysApplicationCustomConfig.class);
    }

    public List<SysApplicationCustomConfig> findSysByAdpCodeList(List<String> codeList) {
        Criteria criteria = Criteria.where("adpCode").in(codeList);
        Query query = simpleInfoListQueryCondition(criteria);
        return mongoSystemRepositoryDecorator.find(query,SysApplicationCustomConfig.class);
    }

    public List<SysApplicationCustomConfig> findSysByAdpCodeList4Share(List<String> codeList) {
        Criteria criteria = Criteria.where("adpCode").in(codeList);
        Query query = new Query(criteria).with(Sort.by(Sort.Direction.DESC, "createDate"));
        return mongoSystemRepositoryDecorator.find(query,SysApplicationCustomConfig.class);
    }

    public List<ApplicationCustomConfig> selectExcludeCustomControl(String application) {
        Criteria criteria = Criteria.where("application").is(application);
        Query query = new Query(criteria);
        Field fields = query.fields();
        fields.exclude("customControl");
        return mongoMultiRepositoryDecorator.find(query, ApplicationCustomConfig.class);
    }
}
