package com.digiwin.athena.dtdapp.service;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.digiwin.athena.dtdapp.dao.zentao.ZtUserDAO;
import com.digiwin.athena.dtdapp.mapperzentao.ZtDJHolidayDateMapper;
import com.digiwin.athena.dtdapp.mapperzentao.ZtPlusProjectMapper;
import com.digiwin.athena.dtdapp.pojo.constants.Constant;
import com.digiwin.athena.dtdapp.pojo.dto.zentaoplus.*;
import com.digiwin.athena.dtdapp.pojo.dto.zt.ZtProjectDTO;
import com.digiwin.athena.dtdapp.pojo.dto.ztkanban.queryTaskReq;
import com.digiwin.athena.dtdapp.pojo.entity.zentao.*;
import com.digiwin.athena.dtdapp.pojo.enums.KanBanStatus;
import com.digiwin.athena.dtdapp.pojo.enums.Status;
import com.digiwin.athena.dtdapp.pojo.vo.zentaoplus.*;
import com.digiwin.athena.dtdapp.pojo.vo.ztkanban.*;
import com.digiwin.athena.dtdapp.util.DateUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.math.MathContext;
import java.math.RoundingMode;
import java.text.*;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Service
@Slf4j
public class DtdappProjectService {

    private final static String INVALID_TIME = "0000-00-00";

    // TODO add more weekends and handle work day in sunday and saturday
    private final static HashSet<String> WEEKENDS = new HashSet<>(Arrays.asList("2024-05-01", "2024-05-02", "2024-05-03", "2024-06-10", "2024-09-16", "2024-09-17", "2024-10-01", "2024-10-02", "2024-10-03", "2024-10-04", "2024-10-07", "2024-10-08"));

    @Autowired
    private ZtPlusProjectMapper projectMapper;

    @Autowired
    private ZtDJHolidayDateMapper holidayDateMapper;

    @Autowired
    private ZtUserDAO ztUserDAO;

    public Collection<ProjectWithTeamInfoVo> getProjectWithTeamInfoList(String parentProjectName, String projectName, String status) {
        List<ProjectWithTeamInfoDto> infoDto = projectMapper.getProjectWithTeamInfoList(parentProjectName, projectName, status);
        Map<Integer, ProjectWithTeamInfoVo> projectIdToTeamInfoVoMap = new HashMap<>();
        HashSet<String> userNames = new HashSet<>();
        for (ProjectWithTeamInfoDto dto : infoDto) {
            userNames.add(dto.getUserName());
            projectIdToTeamInfoVoMap.computeIfAbsent(dto.getId(),
                            ignore -> new ProjectWithTeamInfoVo(dto.getId(), dto.getName(), dto.getStatus(), dto.getEstimate(), dto.getConsumed(), dto.getBegin(), dto.getEnd()))
                    .getConsumedInfoVos().add(new ProjectTeamConsumedInfoVo(dto.getUserName(), dto.getRole(), dto.getUserEstimate(), dto.getUserConsumed()));
        }
        for (ProjectWithTeamInfoVo vo : projectIdToTeamInfoVoMap.values()) {
            Set<String> users = vo.getConsumedInfoVos().stream().map(ProjectTeamConsumedInfoVo::getUserName).collect(Collectors.toSet());
            for (String each : userNames) {
                if (!users.contains(each)) {
                    vo.getConsumedInfoVos().add(new ProjectTeamConsumedInfoVo(each, "", 0.0, 0.0));
                }
            }
            vo.getConsumedInfoVos().sort(Comparator.comparing(ProjectTeamConsumedInfoVo::getUserName));
        }
        return projectIdToTeamInfoVoMap.values();
    }

    public Collection<ProjectTeamDetailVo> getProjectTeamDetailList(int projectId, String userName, String beginDate, String endDate) {
        List<ProjectTeamDetailDto> detailDto = projectMapper.getProjectTeamDetailList(projectId, userName, beginDate, endDate);
        Map<Integer, ProjectTeamDetailVo> userIdToTeamDetailVoMap = new HashMap<>();
        for (ProjectTeamDetailDto dto : detailDto) {
            ProjectTeamDetailVo projectTeamDetailVo = userIdToTeamDetailVoMap.computeIfAbsent(dto.getUserId(),
                    ignore -> {
                        ProjectTeamDetailVo detailVo = new ProjectTeamDetailVo(dto.getUserName(), dto.getRole());
                        initDateEstimateAndConsumed(detailVo, beginDate, endDate);
                        return detailVo;
                    });
            appendEstimate(projectTeamDetailVo, dto);
        }
        List<UserEffortDto> userEffortsDto = projectMapper.getProjectUserEffort(projectId, userName, beginDate, endDate);
        for (UserEffortDto dto : userEffortsDto) {
            ProjectTeamDetailVo projectTeamDetailVo = userIdToTeamDetailVoMap.computeIfAbsent(dto.getUserId(),
                    ignore -> {
                        ProjectTeamDetailVo detailVo = new ProjectTeamDetailVo(dto.getUserName(), dto.getRole());
                        initDateEstimateAndConsumed(detailVo, beginDate, endDate);
                        return detailVo;
                    });
            projectTeamDetailVo.getDateConsumed().computeIfPresent(dto.getDate(), (ignored, v) -> BigDecimal.valueOf(v + dto.getConsumed()).setScale(2, RoundingMode.CEILING).doubleValue());
        }
        handleEstimateMoreThanTwelveForProject(userIdToTeamDetailVoMap);
        userIdToTeamDetailVoMap.values().forEach(v -> {
            v.setEstimate(BigDecimal.valueOf(v.getDateEstimate().values().stream().mapToDouble(Double::doubleValue).sum()).setScale(2, RoundingMode.CEILING).doubleValue());
            v.setConsumed(BigDecimal.valueOf(v.getDateConsumed().values().stream().mapToDouble(Double::doubleValue).sum()).setScale(2, RoundingMode.CEILING).doubleValue());
        });
        return userIdToTeamDetailVoMap.values();
    }


    private void initDateEstimateAndConsumed(ProjectTeamDetailVo detailVo, String beginDate, String endDate) {
        List<String> workingDays = getWorkingDays(beginDate, endDate);
        for (String each : workingDays) {
            detailVo.getDateEstimate().put(each, 0.0);
            detailVo.getDateConsumed().put(each, 0.0);
        }
    }

    private void initDateEstimateAndConsumed(UserEstimateAndConsumedVo estimateAndConsumedVo, String beginDate, String endDate) {
        List<String> workingDays = getWorkingDays(beginDate, endDate);
        for (String each : workingDays) {
            estimateAndConsumedVo.getDateEstimate().put(each, 0.0);
            estimateAndConsumedVo.getDateConsumed().put(each, 0.0);
        }
    }

    private void initDateEstimateAndConsumed(UserProjectConsumedVo estimateAndConsumedVo, String beginDate, String endDate) {
        List<String> workingDays = getWorkingDays(beginDate, endDate);
        for (String each : workingDays) {
            estimateAndConsumedVo.getDateEstimate().put(each, 0.0);
            estimateAndConsumedVo.getDateConsumed().put(each, 0.0);
        }
    }

    private void appendEstimate(ProjectTeamDetailVo target, ProjectTeamDetailDto source) {
        if (source.getEstimate() != 0.0 && null != source.getStarted() && !INVALID_TIME.equals(source.getStarted())) {
            List<String> workingDays = getWorkingDays(source.getStarted(), source.getEnded());
            double avgConsumed = BigDecimal.valueOf(source.getEstimate()).divide(new BigDecimal(workingDays.size()), RoundingMode.CEILING).setScale(1, RoundingMode.CEILING).doubleValue();
            for (String each : workingDays) {
                target.getDateEstimate().computeIfPresent(each, (ignored, v) -> BigDecimal.valueOf(v + avgConsumed).setScale(2, RoundingMode.CEILING).doubleValue());
            }
        }
    }

    private void appendEstimate(UserEstimateAndConsumedVo target, UserTaskEstimatedAndConsumedDto source) {
        if (source.getEstimate() != 0.0 && null != source.getStarted() && !INVALID_TIME.equals(source.getStarted())) {
            List<String> workingDays = getWorkingDays(source.getStarted(), source.getEnded());
            double avgConsumed = BigDecimal.valueOf(source.getEstimate()).divide(new BigDecimal(workingDays.size()), RoundingMode.CEILING).setScale(2, RoundingMode.CEILING).doubleValue();
            for (String each : workingDays) {
                target.getDateEstimate().computeIfPresent(each, (ignored, v) -> BigDecimal.valueOf(v + avgConsumed).setScale(2, RoundingMode.CEILING).doubleValue());
            }
        }
    }

    private void appendEstimate(UserProjectConsumedVo target, UserProjectConsumedDetailDto source) {
        if (source.getEstimate() != 0.0 && null != source.getStarted() && !INVALID_TIME.equals(source.getStarted())) {
            List<String> workingDays = getWorkingDays(source.getStarted(), source.getEnded());
            double avgConsumed = BigDecimal.valueOf(source.getEstimate()).divide(new BigDecimal(workingDays.size()), RoundingMode.CEILING).setScale(1, RoundingMode.CEILING).doubleValue();
            for (String each : workingDays) {
                target.getDateEstimate().computeIfPresent(each, (ignored, v) -> v + avgConsumed);
            }
        }
    }

    @SneakyThrows
    private List<String> getWorkingDays(String startDate, String endDate) {
        if (null == endDate || INVALID_TIME.equals(endDate)) {
            return Collections.singletonList(startDate);
        }
        DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
        Date begin = dateFormat.parse(startDate);
        Date end = dateFormat.parse(endDate);
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(begin);
        List<String> result = new LinkedList<>();
        while (calendar.getTime().before(end) || calendar.getTime().equals(end)) {
            if (Calendar.SUNDAY == calendar.get(Calendar.DAY_OF_WEEK) || Calendar.SATURDAY == calendar.get(Calendar.DAY_OF_WEEK)) {
                calendar.add(Calendar.DATE, 1);
                continue;
            }
            String date = dateFormat.format(calendar.getTime().getTime());
            if (!WEEKENDS.contains(date)) {
                result.add(date);
            }
            calendar.add(Calendar.DATE, 1);
        }
        if (result.isEmpty()) {
            result.add(startDate);
        }
        return result;
    }

    public Collection<UserEstimateAndConsumedVo> getUserEstimateAndConsumedList(int deptId, String userName, String beginDate, String endDate) {
        Collection<Integer> deptIds = getDeptAndSubDeptIds(deptId);
        if (deptIds.isEmpty()) {
            return Collections.emptyList();
        }
        List<UserTaskEstimatedAndConsumedDto> userTasks = projectMapper.getUserTask(deptIds, userName, beginDate, endDate);
        Map<Integer, UserEstimateAndConsumedVo> userIdToVoMap = new HashMap<>();
        for (UserTaskEstimatedAndConsumedDto dto : userTasks) {
            UserEstimateAndConsumedVo vo = userIdToVoMap.computeIfAbsent(dto.getUserId(),
                    ignore -> {
                        UserEstimateAndConsumedVo estimateAndConsumedVo = new UserEstimateAndConsumedVo(dto.getUserId(), dto.getUserName(), dto.getRole());
                        initDateEstimateAndConsumed(estimateAndConsumedVo, beginDate, endDate);
                        return estimateAndConsumedVo;
                    });
            appendEstimate(vo, dto);
        }
        List<UserEffortDto> userEffortsDto = projectMapper.getUserEffortDetail(deptIds, userName, beginDate, endDate);
        for (UserEffortDto dto : userEffortsDto) {
            UserEstimateAndConsumedVo vo = userIdToVoMap.computeIfAbsent(dto.getUserId(),
                    ignore -> {
                        UserEstimateAndConsumedVo estimateAndConsumedVo = new UserEstimateAndConsumedVo(dto.getUserId(), dto.getUserName(), dto.getRole());
                        initDateEstimateAndConsumed(estimateAndConsumedVo, beginDate, endDate);
                        return estimateAndConsumedVo;
                    });
            vo.getDateConsumed().computeIfPresent(dto.getDate(), (ignored, v) -> v + dto.getConsumed());
        }
        // handle if estimate < 8 or > 12
        handleEstimateLessThanEight(userIdToVoMap);
        userIdToVoMap.values().forEach(v -> {
            v.setEstimate(BigDecimal.valueOf(v.getDateEstimate().values().stream().mapToDouble(Double::doubleValue).sum()).setScale(2, RoundingMode.CEILING).doubleValue());
            v.setConsumed(BigDecimal.valueOf(v.getDateConsumed().values().stream().mapToDouble(Double::doubleValue).sum()).setScale(2, RoundingMode.CEILING).doubleValue());
        });
        return userIdToVoMap.values();
    }

    private Collection<Integer> getDeptAndSubDeptIds(int deptId) {
        Set<Integer> result = new HashSet<>();
        result.add(deptId);
        addSubDeptIds(deptId, result);
        return result;
    }

    private void addSubDeptIds(Integer deptId, Set<Integer> result) {
        List<Integer> subDeptIds = projectMapper.getSubDeptIds(deptId);
        result.addAll(subDeptIds);
    }

    private void handleEstimateLessThanEight(Map<Integer, UserEstimateAndConsumedVo> userIdToVoMap) {
        Date current = new Date();
        DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
        userIdToVoMap.values().forEach(v -> {
            v.getDateEstimate().forEach((key, value) -> {
                try {
                    if (dateFormat.parse(key).before(current)) {
                        if (value < 8) {
                            v.getDateEstimate().put(key, 8.0);
                        } else if (value > 12) {
                            v.getDateEstimate().put(key, 12.0);
                        }
                    }
                } catch (ParseException e) {
                    throw new RuntimeException(e);
                }
            });
        });
    }

    public Collection<UserProjectConsumedVo> getUserProjectConsumedList(int userId, String beginDate, String endDate) {
        List<UserProjectConsumedDetailDto> consumedDetailDto = projectMapper.getUserProjectConsumedList(userId, beginDate, endDate);
        Map<Integer, UserProjectConsumedVo> projectIdToVoMap = new HashMap<>();
        for (UserProjectConsumedDetailDto dto : consumedDetailDto) {
            UserProjectConsumedVo vo = projectIdToVoMap.computeIfAbsent(dto.getProjectId(),
                    ignore -> {
                        UserProjectConsumedVo estimateAndConsumedVo = new UserProjectConsumedVo(dto.getProjectName(), dto.getProjectId(), dto.getBegin(), dto.getEnd());
                        initDateEstimateAndConsumed(estimateAndConsumedVo, beginDate, endDate);
                        return estimateAndConsumedVo;
                    });
            appendEstimate(vo, dto);
        }
        handleEstimateMoreThanTwelve(projectIdToVoMap);
        List<UserEffortDto> userEffortsDto = projectMapper.getUserProjectEffort(userId, beginDate, endDate);
        for (UserEffortDto dto : userEffortsDto) {
            UserProjectConsumedVo vo = projectIdToVoMap.computeIfAbsent(dto.getProjectId(),
                    ignore -> {
                        UserProjectConsumedVo userProjectConsumedVo = new UserProjectConsumedVo(dto.getProjectName(), dto.getProjectId(), dto.getProjectBegin(), dto.getProjectEnd());
                        initDateEstimateAndConsumed(userProjectConsumedVo, beginDate, endDate);
                        return userProjectConsumedVo;
                    });
            vo.getDateConsumed().computeIfPresent(dto.getDate(), (ignored, v) -> BigDecimal.valueOf(v + dto.getConsumed()).setScale(2, RoundingMode.CEILING).doubleValue());
        }
        projectIdToVoMap.values().forEach(v -> {
            v.setEstimate(BigDecimal.valueOf(v.getDateEstimate().values().stream().mapToDouble(Double::doubleValue).sum()).setScale(2, RoundingMode.CEILING).doubleValue());
            v.setConsumed(BigDecimal.valueOf(v.getDateConsumed().values().stream().mapToDouble(Double::doubleValue).sum()).setScale(2, RoundingMode.CEILING).doubleValue());
        });
        return projectIdToVoMap.values();
    }

    private void handleEstimateMoreThanTwelve(Map<Integer, UserProjectConsumedVo> projectIdToVoMap) {
        for (UserProjectConsumedVo vo : projectIdToVoMap.values()) {
            vo.getDateEstimate().forEach((key, value) -> {
                if (value > 12) {
                    vo.getDateEstimate().put(key, 12.0);
                }
            });
        }
    }

    private void handleEstimateMoreThanTwelveForProject(Map<Integer, ProjectTeamDetailVo> userIdToTeamDetailVoMap) {
        for (ProjectTeamDetailVo vo : userIdToTeamDetailVoMap.values()) {
            vo.getDateEstimate().forEach((key, value) -> {
                if (value > 12) {
                    vo.getDateEstimate().put(key, 12.0);
                }
            });
        }
    }

    public List<DateToProjectConsumedVo> getDateToProjectConsumedVoList(String parentProjectName, String beginDate, String endDate) {
        List<ProjectEstimateDto> dto = projectMapper.getProjectTaskConsumed(parentProjectName, beginDate, endDate);
        List<String> workingDays = getWorkingDays(beginDate, endDate);
        Map<String, Map<Integer, ProjectConsumedVo>> dateToProjectIdToConsumedVoMap = new LinkedHashMap<>();
        for (String each : workingDays) {
            dateToProjectIdToConsumedVoMap.put(each, new LinkedHashMap<>());
        }
        for (ProjectEstimateDto each : dto) {
            if (each.getEstimate() != 0.0 && null != each.getStarted() && !INVALID_TIME.equals(each.getStarted())) {
                List<String> days = getWorkingDays(each.getStarted(), each.getEnded());
                double avgConsumed = BigDecimal.valueOf(each.getEstimate()).divide(new BigDecimal(days.size()), RoundingMode.CEILING).setScale(1, RoundingMode.CEILING).doubleValue();
                for (String day : days) {
//                    dateToProjectIdToConsumedVoMap.computeIfPresent(day, (ignored, v) -> {
//                        ProjectConsumedVo projectConsumedVo = v.computeIfAbsent(each.getProjectId(), ignored2 -> new ProjectConsumedVo(each.getProjectId(), each.getProjectName()));
//                    });
//                    target.getDateEstimate().computeIfPresent(each, (ignored, v) -> v + avgConsumed);
                }
            }
//            if (source.getConsumed() != 0.0 && null != source.getRealStarted() && !INVALID_TIME.equals(source.getRealStarted())) {
//                List<String> workingDays = getWorkingDays(source.getRealStarted(), source.getRealEnded());
//                double avgConsumed = BigDecimal.valueOf(source.getConsumed()).divide(new BigDecimal(workingDays.size()), RoundingMode.CEILING).setScale(1, RoundingMode.CEILING).doubleValue();
//                for (String each : workingDays) {
//                    target.getDateConsumed().computeIfPresent(each, (ignored, v) -> v + avgConsumed);
//                }
//            }

//            dateToProjectIdToConsumedVoMap.computeIfPresent()
        }
        return null;
    }

    /**
     * 根据项目名称和部门获取项目里程碑
     *
     * @param projectName 项目名称
     * @param programId   项目集id
     * @return 返回项目里程碑
     */
    public List<ZtProjectDTO> getMilestoneByName(String projectName, Integer programId) {
        List<ZtProjectDTO> ztProjects = projectMapper.getProjectByName(projectName, programId);
        if (CollectionUtils.isEmpty(ztProjects)) {
            return Lists.newArrayList();
        }

        List<Integer> parentIds = ztProjects.stream().map(v -> v.getId()).collect(Collectors.toList());
        List<ZtProjectDTO> stages = projectMapper.getProjectChildrenByParent(parentIds, "stage");
        // 判断是否逾期
        stages.stream().filter(v -> ("doing".equals(v.getStatus()) || "wait".equals(v.getStatus()))).forEach(v -> {
            String[] dateNumbers = v.getEnd().split("-");
            long daysBetween = ChronoUnit.DAYS.between(
                    LocalDate.of(Integer.parseInt(dateNumbers[0]), Integer.parseInt(dateNumbers[1]),
                            Integer.parseInt(dateNumbers[2])), LocalDate.now());
            if (daysBetween > 0) {
                v.setIsDelay(Boolean.TRUE);
                v.setDelayDayCount(daysBetween);
            }
        });
        ztProjects.addAll(stages);
        ztProjects.stream().forEach(v -> {
            ztProjects.stream().forEach(t -> {
                if (v.getId().equals(t.getParent())) {
                    if (v.getChildren() == null) {
                        v.setChildren(Lists.newArrayList());
                    }
                    v.getChildren().add(t);
                }
            });
        });
        return ztProjects.stream().filter(v -> v.getParent() == null).collect(Collectors.toList());
    }

    /**
     * 根据父级id获取项目集
     *
     * @param parentId 父级id
     * @return 返回项目集
     */
    public List<ZtProjectDTO> getProgramByParentId(Integer parentId) {
        List<Integer> parentIds = new ArrayList<>();
        if (parentId != null) {
            parentIds.add(parentId);
        }
        List<ZtProjectDTO> ztProjects = projectMapper.getProjectChildrenByParent(parentIds, "program");
        Integer parentGrade = ztProjects.stream().map(v -> v.getGrade()).min(Integer::compareTo).get();

        ztProjects.stream().forEach(v -> {
            ztProjects.stream().forEach(t -> {
                if (v.getId().equals(t.getParent())) {
                    if (v.getChildren() == null) {
                        v.setChildren(Lists.newArrayList());
                    }
                    v.getChildren().add(t);
                }
            });
        });
        return ztProjects.stream().filter(v -> v.getGrade() == parentGrade).collect(Collectors.toList());
    }

    /**
     * 根据部门id获取用户
     *
     * @param deptId 部门id
     * @param name   用户姓名
     * @return 返回用户信息
     */
    public List<ZtUser> getUserByDept(Integer deptId, String name) {
        return getUserByDeptList(deptId == null ? null : Lists.newArrayList(deptId), name);
    }

    /**
     * 根据部门id集合获取用户
     *
     * @param deptIdList 部门id集合
     * @param name       用户姓名
     * @return 返回用户信息
     */
    public List<ZtUser> getUserByDeptList(List<Integer> deptIdList, String name) {
        return projectMapper.getUserByDeptList(deptIdList, name);
    }

    /**
     * 获取任务
     *
     * @param userAccountList
     * @param date
     * @return
     */
    public List<ZtTask> getTaskByFinishUser(List<String> userAccountList, String date) {
        return projectMapper.getTaskByFinishUser(userAccountList, date);
    }

    /**
     * 获取任务
     *
     * @param userAccount
     * @param date
     * @return
     */
    public List<ZtTask> getTaskByUser(String userAccount, String date) {
        return projectMapper.getTaskByUser(userAccount, date);
    }

    /**
     * 查询用户积分看板
     *
     * @param userId
     * @param deptId
     * @param date
     * @return
     */
    public List<ZtUserScoreVO> getUserScore(Integer userId, Integer deptId, String date) {
        String[] split = date.split("[^\\d]");
        Integer month = Integer.valueOf(split[1]);
        List<Integer> userIdList = Lists.newArrayList();
        List<String> userAccountList = Lists.newArrayList();
        if (userId == null) {
            List<ZtUser> userList = getUserByDept(deptId, null);
            userIdList.addAll(userList.stream().map(v -> v.getId()).collect(Collectors.toList()));
            userAccountList.addAll(userList.stream().map(v -> v.getAccount()).collect(Collectors.toList()));
        } else {
            userIdList.add(userId);
            ZtUser ztUser = ztUserDAO.getBaseMapper().selectById(userId);
            userAccountList.add(ztUser.getAccount());
        }
        List<ZtTask> taskList = getTaskByFinishUser(userAccountList, date.split("-")[0]);
        if (CollectionUtils.isEmpty(taskList)) {
            return null;
        }

        List<ZtUser> ztUsers = ztUserDAO.getBaseMapper().selectBatchIds(userIdList);
        Map<Integer, ZtUser> userMap = ztUsers.stream().collect(Collectors.toMap(v -> v.getId(), v -> v, (k1, k2) -> k1));
        List<ZtTask> currentMonthTaskList = taskList.stream().
                filter(v -> isCurrentMonth(v.getFinishedDate().toString().split(" ")[0], month)).collect(Collectors.toList());
        Map<String, List<ZtTask>> monthTaskMap = currentMonthTaskList.stream().collect(Collectors.groupingBy(v -> v.getFinishedBy()));
        Map<String, List<ZtTask>> yearTaskMap = taskList.stream().collect(Collectors.groupingBy(v -> v.getFinishedBy()));
        List<ZtUserScoreVO> userScoreVOS = userIdList.stream().map(v -> {
            ZtUserScoreVO userScoreVO = new ZtUserScoreVO();
            ZtUser ztUser = userMap.get(v);
            List<ZtTask> yearTaskList = yearTaskMap.get(ztUser.getAccount());
            List<ZtTask> monthTaskList = monthTaskMap.get(ztUser.getAccount());

            Integer yearScore = countScoreTotal(yearTaskList);
            Integer monthScore = countScoreTotal(monthTaskList);
            userScoreVO.setId(v);
            userScoreVO.setAccount(ztUser.getAccount());
            userScoreVO.setDept(ztUser.getDept());
            userScoreVO.setRealname(ztUser.getRealname());
            userScoreVO.setRole(ztUser.getRole().toUpperCase(Locale.ROOT));
            userScoreVO.setMonthScore(monthScore);
            userScoreVO.setMonthScoreTotal(monthScore + 80);
            userScoreVO.setYearScoreTotal(yearScore + 80 * month);
            return userScoreVO;
        }).collect(Collectors.toList());

        Integer monthScoreMax = 999999;
        Integer yearScoreMax = 999999;
        Integer monthRankMax = 0;
        Integer yearRankMax = 0;
        LinkedList<ZtUserScoreVO> monthUserScoreList = userScoreVOS.stream()
                .sorted(Comparator.comparing(ZtUserScoreVO::getMonthScoreTotal).reversed())
                .collect(Collectors.toCollection(LinkedList::new));
        for (int i = 0; i < monthUserScoreList.size(); i++) {
            ZtUserScoreVO userScoreVO = monthUserScoreList.get(i);
            if (monthScoreMax > userScoreVO.getMonthScoreTotal()) {
                monthScoreMax = userScoreVO.getMonthScoreTotal();
                monthRankMax++;
            }
            userScoreVO.setMonthRank(monthRankMax);
        }
        LinkedList<ZtUserScoreVO> yearUserScoreList = userScoreVOS.stream()
                .sorted(Comparator.comparing(ZtUserScoreVO::getYearScoreTotal)
                        .thenComparing(ZtUserScoreVO::getMonthScore).reversed())
                .collect(Collectors.toCollection(LinkedList::new));
        for (int i = 0; i < yearUserScoreList.size(); i++) {
            ZtUserScoreVO userScoreVO = yearUserScoreList.get(i);
            if (yearScoreMax > userScoreVO.getYearScoreTotal()) {
                yearScoreMax = userScoreVO.getYearScoreTotal();
                yearRankMax++;
            }
            userScoreVO.setYearRank(yearRankMax);
        }
        return yearUserScoreList;
    }

    /**
     * 获取用户的积分任务
     *
     * @param account 账号
     * @param date    日期
     * @return 返回任务
     */
    public List<ZtTask> getUserScoreTask(String account, String date) {
        if (ObjectUtils.isEmpty(account)) {
            return null;
        }
        List<ZtTask> taskList = getTaskByUser(account, date.split("-")[0]);
        List<ZtTask> taskResult = taskList.stream()
                .filter(v -> {
                    if (Status.Done_State.getValue().equalsIgnoreCase(v.getStatus().getValue())) {
                        return v.getFinishedDate().toString().startsWith(date.substring(0, 7));
                    }
                    return true;
                }).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(taskList)) {
            return null;
        }
        Set<String> userAccountList = taskResult.stream().flatMap(
                v -> Stream.of(v.getFinishedBy(), v.getAssignedTO())).collect(Collectors.toSet());

        List<ZtUser> userList = projectMapper.getUserByAccount(userAccountList.stream().collect(Collectors.toList()));
        Map<String, ZtUser> userMap = userList.stream().collect(Collectors.toMap(v -> v.getAccount(), v -> v, (k1, k2) -> k1));

        taskResult.stream().forEach(v -> {
            if (!ObjectUtils.isEmpty(v.getFinishedBy())) {
                v.setFinishedBy(userMap.get(v.getFinishedBy()).getRealname());
            }
            v.setAssignedTO(userMap.get(v.getAssignedTO()).getRealname());
        });
        return taskResult;
    }

    /**
     * 判断日期是否是指定月份
     *
     * @param date
     * @param month
     * @return 返回结果
     */
    private Boolean isCurrentMonth(String date, Integer month) {
        String[] split = date.split("[^\\d]");
        Integer targetMonth = Integer.valueOf(split[1]);
        return month == targetMonth;
    }

    /**
     * 计算任务总得分
     *
     * @param taskList
     * @return 返回结果
     */
    private Integer countScoreTotal(List<ZtTask> taskList) {
        if (CollectionUtils.isEmpty(taskList)) {
            return Integer.valueOf(0);
        }
        Integer scoreTotal = taskList.stream().map(v -> {
            String taskName = v.getName();
            String scoreTask = taskName.split("积分】")[0];
            if (ObjectUtils.isEmpty(scoreTask)) {
                return null;
            }
            return Integer.valueOf(scoreTask.substring(1));
        }).filter(v -> v != null).reduce(Integer::sum).orElse(0);
        return scoreTotal;
    }

    /**
     * 查询用户任务列表
     *
     * @param queryTaskReq 请求对象
     * @param scene        查询场景 0：组织维度 1：项目维度
     * @return 返回
     */
    public ZtTaskRespVO queryTaskListByCondition(queryTaskReq queryTaskReq, Integer scene) {
        ZtTaskRespVO ztTaskRespVO = new ZtTaskRespVO();
        long availableDuration = 0;
        // 当前部门下所有用户
        List<ZtUser> userList = ztUserDAO.queryUserList(queryTaskReq.getDeptId());
        Map<String, ZtUser> accountMap = userList.stream().collect(Collectors.toMap(ZtUser::getAccount, Function.identity(), (a, b) -> a));
        // 查询任务列表
        List<ZtTaskProjectVO> taskList = projectMapper.queryTaskListByCondition(queryTaskReq, accountMap.keySet());
        if (CollectionUtils.isNotEmpty(taskList)) {
            taskList.forEach(x -> {
                if (StringUtils.isBlank(x.getFinishedBy())) {
                    x.setFinishedBy(x.getAssignedTO());
                }
            });
            // 组织维度，查询时间条件范围内的实际工期
            Map<Integer, Map<String, Optional<Float>>> taskIdMap = Maps.newHashMap();
            if (Constant.QUERY_TASK_LIST_SCENE_WITH_DEPT.equals(scene) && StringUtils.isNotBlank(queryTaskReq.getStartTime()) && StringUtils.isNotBlank(queryTaskReq.getEndTime())) {
                List<Integer> taskIdList = taskList.stream().map(ZtTaskProjectVO::getId).collect(Collectors.toList());
                List<ZtEffort> taskEffortList = projectMapper.queryTaskConsumed(taskIdList, queryTaskReq.getStartTime(), queryTaskReq.getEndTime());
                taskIdMap = taskEffortList.stream().collect(Collectors.groupingBy(ZtEffort::getObjectID, Collectors.groupingBy(ZtEffort::getAccount, Collectors.mapping(ZtEffort::getConsumed, Collectors.reducing(Float::sum)))));
            }

            // 计算计划工期、实际工期、进度、工时偏差率
            MathContext mathContext = new MathContext(3, RoundingMode.HALF_UP);
            for (ZtTaskProjectVO x : taskList) {
                // 组织维度，需要根据时间范围计算计划工期和实际工期
                if (Constant.QUERY_TASK_LIST_SCENE_WITH_DEPT.equals(scene) && StringUtils.isNotBlank(queryTaskReq.getStartTime()) && StringUtils.isNotBlank(queryTaskReq.getEndTime())) {
                    calculateEstimate(x, queryTaskReq, taskIdMap);
                }
                // 计划工期四舍五入
                BigDecimal estimate = ObjectUtils.isEmpty(x.getEstimate()) || x.getEstimate().equals(new BigDecimal(0)) ? new BigDecimal(0) : x.getEstimate().setScale(1, RoundingMode.HALF_UP);
                x.setEstimate(estimate);

                // 实际工期四舍五入
                BigDecimal consumed = ObjectUtils.isEmpty(x.getConsumed()) || x.getConsumed().equals(new BigDecimal(0)) ? new BigDecimal(0) : x.getConsumed().setScale(1, RoundingMode.HALF_UP);
                x.setConsumed(consumed);

                // 任务类型
                x.setTypeName(KanBanStatus.fromValue(x.getType()));

                // 计算进度
                String process = calculateProcess(x.getId(), x.getConsumed(), x.getLeft(), mathContext);
                x.setTaskProgress(process);

                // 工时偏差率
                String workHourDeviation = calculateWorkHourDeviation(x.getEstimate(), x.getConsumed(), mathContext);
                x.setWorkHourDeviationRate(workHourDeviation);
            }

            // 组织维度，需要计算可用工期
            if (Constant.QUERY_TASK_LIST_SCENE_WITH_DEPT.equals(scene)) {
                availableDuration = calculateAvailablePeriod(queryTaskReq);
            }
        }

        // 查询bug列表
        List<ZtTaskProjectVO> bugList = projectMapper.queryBugListByCondition(queryTaskReq);
        if (CollectionUtils.isNotEmpty(bugList)) {
            taskList.addAll(bugList);
        }
        if (CollectionUtils.isEmpty(taskList)) {
            return ztTaskRespVO;
        }

        // 构建用户任务列表
        List<ZtTaskVO> userTaskList = buildUserTaskList(taskList, scene, availableDuration, accountMap);
        ztTaskRespVO.setTaskList(userTaskList);
        // 构建任务汇总模块
        ZtTaskSummaryVO taskSummaryVO = buildTaskSummary(userTaskList, scene, availableDuration);
        ztTaskRespVO.setTaskSummary(taskSummaryVO);
        return ztTaskRespVO;
    }

    /**
     * 构建任务列表及及汇总行
     *
     * @param taskList          任务列表
     * @param scene             查询场景 0：组织维度 1：项目维度
     * @param availableDuration 可用工期
     * @param accountMap        用户信息
     * @return 返回
     */
    private List<ZtTaskVO> buildUserTaskList(List<ZtTaskProjectVO> taskList, Integer scene, long availableDuration, Map<String, ZtUser> accountMap) {
        List<ZtTaskVO> userTaskList = Lists.newArrayList();
        // 以用户分组展示任务及bug
        Map<String, List<ZtTaskProjectVO>> accountTaskList = taskList.stream().collect(Collectors.groupingBy(ZtTaskProjectVO::getFinishedBy));
        for (Map.Entry<String, List<ZtTaskProjectVO>> accountEntry : accountTaskList.entrySet()) {
            String key = accountEntry.getKey();
            List<ZtTaskProjectVO> value = accountEntry.getValue();
            ZtTaskVO taskVO = new ZtTaskVO();
            taskVO.setAccount(key);
            taskVO.setRealname(ObjectUtils.isEmpty(accountMap.get(key)) ? "其他" : accountMap.get(key).getRealname());
            taskVO.setRole(ObjectUtils.isEmpty(accountMap.get(key)) ? "其他" : accountMap.get(key).getRole());
            taskVO.setUserTaskList(value);

            // 转换时间格式，便于前端展示
            value.forEach(x -> {
                x.setEstStartedStr(ObjectUtils.isEmpty(x.getEstStarted()) ? "" : DateUtils.parseLocalDateToString(x.getEstStarted(), DateUtils.DATE_FORMAT_TWO));
                x.setDeadlineStr(ObjectUtils.isEmpty(x.getDeadline()) ? "" : DateUtils.parseLocalDateToString(x.getDeadline(), DateUtils.DATE_FORMAT_TWO));
            });

            // 构建任务汇总行信息
            ZtTaskTotalVO taskTotalVO = new ZtTaskTotalVO();
            // 项目id列表
            List<String> projectIdList = value.stream().map(ZtTaskProjectVO::getProjectId).distinct().collect(Collectors.toList());
            taskTotalVO.setProjectIdList(projectIdList);

            // 项目名称列表
            List<String> projectNameList = value.stream().map(ZtTaskProjectVO::getProjectName).distinct().collect(Collectors.toList());
            taskTotalVO.setProjectNameList(projectNameList);

            // 汇总行实际工期
            BigDecimal totalConsumed = value.stream().map(ZtTaskProjectVO::getConsumed).filter(consumed -> !ObjectUtils.isEmpty(consumed)).reduce(BigDecimal.ZERO, BigDecimal::add);
            taskTotalVO.setConsumed(totalConsumed.equals(new BigDecimal(0)) ? new BigDecimal(0) : totalConsumed.setScale(1, RoundingMode.HALF_UP));

            // bug类型的任务不展示如下字段
            boolean flag = value.stream().anyMatch(x -> !KanBanStatus.bug.getValue().equals(x.getType()));
            if (flag) {
                // 最早计划开始时间
                LocalDate earliestEstStarted = value.stream().map(ZtTaskProjectVO::getEstStarted).filter(estStarted -> !ObjectUtils.isEmpty(estStarted)).min(LocalDate::compareTo).orElse(null);
                taskTotalVO.setEarliestEstStarted(earliestEstStarted);
                taskTotalVO.setEarliestEstStartedStr(ObjectUtils.isEmpty(earliestEstStarted) ? "" : DateUtils.parseLocalDateToString(earliestEstStarted, DateUtils.DATE_FORMAT_TWO));

                // 最晚计划完成时间
                LocalDate latestDeadline = value.stream().map(ZtTaskProjectVO::getDeadline).filter(deadline -> !ObjectUtils.isEmpty(deadline)).max(LocalDate::compareTo).orElse(null);
                taskTotalVO.setLatestDeadline(latestDeadline);
                taskTotalVO.setLatestDeadlineStr(ObjectUtils.isEmpty(latestDeadline) ? "" : DateUtils.parseLocalDateToString(latestDeadline, DateUtils.DATE_FORMAT_TWO));

                // 最早实际开始时间
                LocalDateTime earliestRealStarted = value.stream().map(ZtTaskProjectVO::getRealStarted).filter(x -> !ObjectUtils.isEmpty(x)).min(LocalDateTime::compareTo).orElse(null);
                taskTotalVO.setEarliestRealStarted(earliestRealStarted);

                // 最晚实际完成时间
                LocalDateTime latestFinishedDate = value.stream().map(ZtTaskProjectVO::getFinishedDate).filter(x -> !ObjectUtils.isEmpty(x)).max(LocalDateTime::compareTo).orElse(null);
                taskTotalVO.setLatestFinishedDate(latestFinishedDate);

                // 汇总行计划工期
                BigDecimal totalEstimate = value.stream().map(ZtTaskProjectVO::getEstimate).filter(estimate -> !ObjectUtils.isEmpty(estimate)).reduce(BigDecimal.ZERO, BigDecimal::add);
                taskTotalVO.setEstimate(totalEstimate.equals(new BigDecimal(0)) ? new BigDecimal(0) : totalEstimate.setScale(1, RoundingMode.HALF_UP));

                // 计算汇总行进度
                MathContext mathContext = new MathContext(3, RoundingMode.HALF_UP);
                double totalLeft = value.stream().filter(x -> !ObjectUtils.isEmpty(x.getLeft())).mapToDouble(ZtTaskProjectVO::getLeft).sum();
                String process = calculateProcess(null, totalConsumed, (float) totalLeft, mathContext);
                taskTotalVO.setTaskProgress(process);

                // 计算汇总行工时偏差率
                String workHourDeviation = calculateWorkHourDeviation(totalEstimate, totalConsumed, mathContext);
                taskTotalVO.setWorkHourDeviationRate(workHourDeviation);

                // 组织维度需要计算计利用率、实际利用率
                if (Constant.QUERY_TASK_LIST_SCENE_WITH_DEPT.equals(scene)) {
                    // 计算汇总行预计利用率
                    String expectedUtilizationRate = calculateExpectedUtilizationRate(totalEstimate, availableDuration, mathContext);
                    taskTotalVO.setExpectedUtilizationRate(expectedUtilizationRate);

                    // 计算汇总行实际利用率
                    String actualUtilizationRate = calculateActualUtilizationRate(totalConsumed, availableDuration, mathContext);
                    taskTotalVO.setActualUtilizationRate(actualUtilizationRate);
                }
            }
            taskVO.setTaskTotal(taskTotalVO);
            userTaskList.add(taskVO);
        }
        return userTaskList;
    }

    /**
     * 构建任务汇总模块
     *
     * @param userTaskList      任务列表
     * @param scene             查询场景 0：组织维度 1：项目维度
     * @param availableDuration 可用工期
     * @return 返回
     */
    private ZtTaskSummaryVO buildTaskSummary(List<ZtTaskVO> userTaskList, Integer scene, long availableDuration) {
        ZtTaskSummaryVO taskSummary = new ZtTaskSummaryVO();
        // 总人数
        taskSummary.setPersonNum(userTaskList.size());

        // PR人数
        long devNum = userTaskList.stream().filter(x -> KanBanStatus.DEV.getValue().equals(x.getRole())).count();
        taskSummary.setDevNum(devNum);

        // QA人数
        long qaNum = userTaskList.stream().filter(x -> KanBanStatus.QC.getValue().equals(x.getRole())).count();
        taskSummary.setQaNum(qaNum);

        // 计划工期
        BigDecimal totalEstimate = userTaskList.stream().map(x -> x.getTaskTotal().getEstimate()).filter(estimate -> !ObjectUtils.isEmpty(estimate)).reduce(BigDecimal.ZERO, BigDecimal::add);
        taskSummary.setTotalEstimate(totalEstimate.equals(new BigDecimal(0)) ? new BigDecimal(0) : totalEstimate.setScale(1, RoundingMode.HALF_UP));

        // 实际工期
        BigDecimal totalConsumed = userTaskList.stream().map(x -> x.getTaskTotal().getConsumed()).filter(consumed -> !ObjectUtils.isEmpty(consumed)).reduce(BigDecimal.ZERO, BigDecimal::add);
        taskSummary.setTotalConsumed(totalConsumed.equals(new BigDecimal(0)) ? new BigDecimal(0) : totalConsumed.setScale(1, RoundingMode.HALF_UP));

        // 计算汇总模块工时偏差率
        MathContext mathContext = new MathContext(3, RoundingMode.HALF_UP);
        String workHourDeviation = calculateWorkHourDeviation(totalEstimate, totalConsumed, mathContext);
        taskSummary.setTotalWorkHourDeviationRate(workHourDeviation);

        // 组织维度需要计算预计利用率、实际利用率
        if (Constant.QUERY_TASK_LIST_SCENE_WITH_DEPT.equals(scene)) {
            // 可用工期
            availableDuration = availableDuration * userTaskList.size();
            taskSummary.setAvailableDuration(availableDuration);

            // 计算汇总模块预计利用率
            String expectedUtilizationRate = calculateExpectedUtilizationRate(totalEstimate, availableDuration, mathContext);
            taskSummary.setTotalExpectedUtilizationRate(expectedUtilizationRate);

            // 计算汇总模块实际利用率
            String actualUtilizationRate = calculateActualUtilizationRate(totalConsumed, availableDuration, mathContext);
            taskSummary.setTotalActualUtilizationRate(actualUtilizationRate);
        }
        return taskSummary;
    }

    /**
     * 计算可用工期
     *
     * @param queryTaskReq 请求参数
     * @return 返回
     */
    private long calculateAvailablePeriod(queryTaskReq queryTaskReq) {
        // 可用工期为时间范围内的工作日
        long availableDuration = calculateAvailableDuration(LocalDate.parse(queryTaskReq.getStartTime()), LocalDate.parse(queryTaskReq.getEndTime()));
        return availableDuration * 8;
    }

    /**
     * 计算可用工期
     *
     * @param startDate 开始时间
     * @param endDate   截止时间
     * @return 返回
     */
    private long calculateAvailableDuration(LocalDate startDate, LocalDate endDate) {
        QueryWrapper<ZtDJHolidayDate> queryWrapper = new QueryWrapper<>();
        queryWrapper.between("date", startDate, endDate);
        List<ZtDJHolidayDate> holidayDateList = holidayDateMapper.selectList(queryWrapper);

        // 计算可用工期
        List<String> holidayList = holidayDateList.stream().filter(x -> Constant.DATE_TYPE_HOLIDAY.equals(x.getDateType())).map(ZtDJHolidayDate::getDate).collect(Collectors.toList());
        List<String> workingList = holidayDateList.stream().filter(x -> Constant.DATE_TYPE_WORKING.equals(x.getDateType())).map(ZtDJHolidayDate::getDate).collect(Collectors.toList());
        return DateUtils.calculateLegalWorkingDays(startDate, endDate, holidayList, workingList);
    }

    /**
     * 计算计划工期和实际工期
     *
     * @param x            任务
     * @param queryTaskReq 参数
     * @param taskIdMap    参数
     */
    private static void calculateEstimate(ZtTaskProjectVO x, queryTaskReq queryTaskReq, Map<Integer, Map<String, Optional<Float>>> taskIdMap) {
        // 实际工期
        if (MapUtils.isNotEmpty(taskIdMap) && MapUtils.isNotEmpty(taskIdMap.get(x.getId()))) {
            Optional<Float> optional = taskIdMap.get(x.getId()).get(x.getFinishedBy());
            float temp = ObjectUtils.isEmpty(optional) ? 0 : optional.orElse(0f);
            x.setConsumed(temp == 0 ? new BigDecimal(0) : new BigDecimal(String.valueOf(temp)));
        } else {
            x.setConsumed(new BigDecimal(0));
        }

        // 计划工期
        BigDecimal estimate = x.getEstimate();
        if (ObjectUtils.isEmpty(estimate) || estimate.equals(new BigDecimal(0)) || ObjectUtils.isEmpty(x.getEstStarted()) || ObjectUtils.isEmpty(x.getDeadline())) {
            x.setEstimate(new BigDecimal(0));
            return;
        }
        long coveragePeriod;
        long totalPeriod = DateUtils.calculateDateDifference(x.getEstStarted(), x.getDeadline());
        if (totalPeriod == 0) {
            x.setEstimate(new BigDecimal(0));
            return;
        }
        // 设置MathContext，指定小数点后保留三位，使用四舍五入模式
        MathContext mc = new MathContext(3, RoundingMode.HALF_UP);
        // 计划完成小于开始时间或计划开始大于结束时间
        if (x.getDeadline().isBefore(LocalDate.parse(queryTaskReq.getStartTime())) || x.getEstStarted().isAfter(LocalDate.parse(queryTaskReq.getEndTime()))) {
            estimate = new BigDecimal(0);
        }
        // 计划开始小于开始时间且计划完成小于等于结束时间
        if (x.getEstStarted().isBefore(LocalDate.parse(queryTaskReq.getStartTime())) && !x.getDeadline().isAfter(LocalDate.parse(queryTaskReq.getEndTime()))) {
            coveragePeriod = DateUtils.calculateDateDifference(LocalDate.parse(queryTaskReq.getStartTime()), x.getDeadline());
            estimate = new BigDecimal(coveragePeriod).divide(new BigDecimal(totalPeriod), mc).multiply(estimate);
        }
        // 计划开始大于等于开始时间且计划完成小于等于结束时间
        else if (!x.getEstStarted().isBefore(LocalDate.parse(queryTaskReq.getStartTime())) && !x.getDeadline().isAfter(LocalDate.parse(queryTaskReq.getEndTime()))) {
            estimate = x.getEstimate();
        }
        // 计划开始在开始时间和结束时间之间且计划完成大于结束时间
        else if (!x.getEstStarted().isBefore(LocalDate.parse(queryTaskReq.getStartTime())) && !x.getEstStarted().isAfter(LocalDate.parse(queryTaskReq.getEndTime()))
                && x.getDeadline().isAfter(LocalDate.parse(queryTaskReq.getEndTime()))) {
            coveragePeriod = DateUtils.calculateDateDifference(x.getEstStarted(), LocalDate.parse(queryTaskReq.getEndTime()));
            estimate = new BigDecimal(coveragePeriod).divide(new BigDecimal(totalPeriod), mc).multiply(estimate);
        }
        // 计划开始小于开始时间且计划完成大于结束时间
        else if (x.getEstStarted().isBefore(LocalDate.parse(queryTaskReq.getStartTime())) && x.getDeadline().isAfter(LocalDate.parse(queryTaskReq.getEndTime()))) {
            coveragePeriod = DateUtils.calculateDateDifference(LocalDate.parse(queryTaskReq.getStartTime()), LocalDate.parse(queryTaskReq.getEndTime()));
            estimate = new BigDecimal(coveragePeriod).divide(new BigDecimal(totalPeriod), mc).multiply(estimate);
        }
        x.setEstimate(estimate);
    }

    /**
     * 计算进度
     *
     * @param taskId 任务id
     * @param consumed 计划工期
     * @param left     剩余工期
     * @return 返回
     */
    private String calculateProcess(Integer taskId, BigDecimal consumed, Float left, MathContext mathContext) {
        try {
            if (consumed.equals(new BigDecimal("0")) || consumed.equals(new BigDecimal("0.0"))) {
                return "0%";
            }
            if ((consumed.add(new BigDecimal(left))).equals(new BigDecimal("0")) || (consumed.add(new BigDecimal(left))).equals(new BigDecimal("0.0"))) {
                return "";
            }
            BigDecimal process = consumed.divide(consumed.add(new BigDecimal(left)), mathContext);
            // 四舍五入，保留两位小数，并转为百分比格式，例：process=0.1364159，结果返回14%
            DecimalFormat df = new DecimalFormat("#%");
            return df.format(process);
        } catch (Exception e) {
            log.info("taskId:{}, new BigDecimal(left):{}, consumed.add(new BigDecimal(left)):{}", taskId, new BigDecimal(left), consumed.add(new BigDecimal(left)), e);
        }
        return "";
    }

    /**
     * 计算工时偏差率
     *
     * @param estimate 计划工期
     * @param consumed 实际工期
     * @return 返回
     */
    private String calculateWorkHourDeviation(BigDecimal estimate, BigDecimal consumed, MathContext mathContext) {
        if (estimate.equals(new BigDecimal("0")) || estimate.equals(new BigDecimal("0.0"))) {
            return "0%";
        }
        if (consumed.equals(new BigDecimal("0")) || consumed.equals(new BigDecimal("0.0"))) {
            return "";
        }
        BigDecimal workHourDeviation = estimate.divide(consumed, mathContext);
        // 四舍五入，保留两位小数，并转为百分比格式，例：workHourDeviation=0.1364159，结果返回14%
        DecimalFormat df = new DecimalFormat("#%");
        return df.format(workHourDeviation);
    }

    /**
     * 计算预计利用率
     *
     * @param estimate          计划工期
     * @param availableDuration 可用工期
     * @return 返回
     */
    private String calculateExpectedUtilizationRate(BigDecimal estimate, Long availableDuration, MathContext mathContext) {
        if (estimate.equals(new BigDecimal(0))) {
            return "0%";
        }
        if (availableDuration == 0) {
            return "";
        }
        BigDecimal expectedUtilizationRate = estimate.divide(new BigDecimal(availableDuration), mathContext);
        // 四舍五入，保留两位小数，并转为百分比格式，例：expectedUtilizationRate=0.1364159，结果返回14%
        DecimalFormat df = new DecimalFormat("#%");
        return df.format(expectedUtilizationRate);
    }

    /**
     * 计算实际利用率
     *
     * @param consumed          实际工期
     * @param availableDuration 可用工期
     * @return 返回
     */
    private String calculateActualUtilizationRate(BigDecimal consumed, Long availableDuration, MathContext mathContext) {
        if (consumed.equals(new BigDecimal(0))) {
            return "0%";
        }
        if (availableDuration == 0) {
            return "";
        }
        BigDecimal actualUtilizationRate = consumed.divide(new BigDecimal(availableDuration), mathContext);
        // 四舍五入，保留两位小数，并转为百分比格式，例：actualUtilizationRate=0.1364159，结果返回14%
        DecimalFormat df = new DecimalFormat("#%");
        return df.format(actualUtilizationRate);
    }
}
