package com.digiwin.athena.eventcenter.core.base;

import com.mongodb.client.result.DeleteResult;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.GenericTypeResolver;
import org.springframework.data.domain.Example;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.repository.MongoRepository;
import org.springframework.util.ClassUtils;

import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * srp包下由模型对象产生的业务层service的抽象业务类
 *
 * @author zhangzhi@digiwin.com
 * @date 2023/09/04
 */
public abstract class AbstractService<D extends MongoRepository, T extends BaseModel> {

    @Autowired
    protected D repository;

    @Autowired
    @Qualifier("eventcenterMongoTemplate")
    protected MongoTemplate template;

    protected final Class<T> entityClass = this.currentModelClass();

    protected Class<T> currentModelClass() {
        Class<?>[] typeArguments = GenericTypeResolver.resolveTypeArguments(ClassUtils.getUserClass(this.getClass()), AbstractService.class);
        return null == typeArguments ? null : (Class<T>) typeArguments[1];
    }

    public T save(T model) {
        return (T) repository.insert(model);
    }

    public List<T> save(List<T> models) {
        return repository.insert(models);
    }

    public T editById(T model) {
        return (T) repository.save(model);
    }

    public List<T> editById(List<T> models) {
        return repository.saveAll(models);
    }

    public T selectOneByCondition(Example<T> example) {
        Optional<T> result = repository.findOne(example);
        return result.orElse(null);
    }

    public T selectOneByCondition(Criteria criteria) {
        return template.findOne(new Query(criteria), entityClass);
    }

    public List<T> listByCondition(Example<T> example) {
        return repository.findAll(example);
    }

    public List<T> listByCriteria(Criteria criteria) {
        criteria.and("tenantId").is("获取租户字段值");
        return template.find(new Query(criteria), entityClass);
    }

    public List<T> listByCriteria(Query query) {
        return template.find(query, entityClass);
    }

    public List<T> findAll() {
        return repository.findAll();
    }

    public long countByCondition(Example<T> example) {
        return repository.count(example);
    }

    public boolean exists(Example<T> example) {
        return repository.exists(example);
    }

    public void delete(T model) {
        repository.delete(model);
    }

    public void deleteAll(Iterable<? extends T> entities) {
        repository.deleteAll(entities);
    }

    public boolean deleteByProperty(String propertyName, String propertyValue) {
        Criteria criteria = new Criteria().and(propertyName).is(propertyValue);
        DeleteResult result = template.remove(new Query(criteria), entityClass);
        return result.getDeletedCount() > 0;
    }

    public boolean deleteByProperty(Map<String, String> property) {
        Criteria criteria = new Criteria();
        property.forEach((key, value) -> criteria.and(key).is(value));

        DeleteResult result = template.remove(new Query(criteria), entityClass);
        return result.getDeletedCount() > 0;
    }

    public long countByCriteria(Criteria criteria) {
        return template.count(new Query(criteria), entityClass);
    }

    public long countByCriteria(Query query) {
        return template.count(query, entityClass);
    }

    /**
     * 分页
     */
    public <E> Page<E> page(Integer pageNum, Integer pageSize, Criteria criteria, Class<E> tClass) {
        Query query = new Query(criteria);
        long total = this.template.count(query, this.entityClass);

        query.skip((long) pageSize * (pageNum - 1)).limit(pageSize);
        List<T> list = this.template.find(query, this.entityClass);

        return new Page<>(pageNum, pageSize, total, list, tClass);
    }
}
