package com.digiwin.athena.eventcenter.service.facade.impl;

import com.digiwin.app.container.exceptions.DWRuntimeException;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.eventcenter.core.constant.Constant;
import com.digiwin.athena.eventcenter.pojo.bo.km.EventResponseBo;
import com.digiwin.athena.eventcenter.pojo.dto.trigger.TriggerEventDto;
import com.digiwin.athena.eventcenter.repository.model.trigger.EventTriggerLogModel;
import com.digiwin.athena.eventcenter.service.client.KmClient;
import com.digiwin.athena.eventcenter.service.facade.IEventTriggerFacadeService;
import com.digiwin.athena.eventcenter.service.publish.EventPublishService;
import com.digiwin.athena.eventcenter.service.srp.component.IEventTriggerLogService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import java.util.Objects;

/**
 * @author huwei
 */
@Service
@Slf4j
public class EventTriggerFacadeServiceImpl implements IEventTriggerFacadeService {

    @Autowired
    @Qualifier("eventKmClient")
    private KmClient kmClient;

    @Autowired
    private IEventTriggerLogService eventLogService;

    @Autowired
    private EventPublishService eventPublishService;

    /**
     * 根据事件id查询km触发事件处理
     */
    @Override
    public void triggerEvent(TriggerEventDto dto) {
        // 记录日志
        String eventLogId = saveEventLog(dto);
        // 获取事件定义
        EventResponseBo eventResponseBo = getEventById(dto);
        // 发起流程
        eventPublishService.invokeProcessAsync(dto, eventLogId, eventResponseBo, getDWServiceContext());
    }

    private String saveEventLog(TriggerEventDto dto) {
        //记录日志
        EventTriggerLogModel eventTriggerLogModel = new EventTriggerLogModel(dto);
        eventLogService.save(eventTriggerLogModel);
        return eventTriggerLogModel.getId();
    }

    private EventResponseBo getEventById(TriggerEventDto dto) {
        // 根据事件id查询km获取事件定义
        EventResponseBo eventResponseBo = kmClient.getEventById(dto.getEventId());
        //判断返回参数
        if (Objects.isNull(eventResponseBo) || Objects.isNull(eventResponseBo.getResponse())) {
            log.error("未获取事件[{}]的定义和订阅信息！", dto.getEventId());
            throw new DWRuntimeException("P.eventcenter.500.1001", "未获取事件[" + dto.getEventId() + "]的定义和订阅信息！");
        }
        //判断返回参数
        if (!Constant.SUCCESS_INTEGER.equals(eventResponseBo.getStatus())) {
            log.error("获取事件[{}]出错，错误信息：[{}]！", dto.getEventId(), eventResponseBo.getStatusDescription());
            throw new DWRuntimeException("P.eventcenter.500.1002", "获取事件[" + dto.getEventId() + "]出错！");
        }
        if (!Constant.ENABLED.equals(eventResponseBo.getResponse().getEventEnable())) {
            log.error("事件[{}]状态:[{}]不可用", dto.getEventId(), eventResponseBo.getResponse().getEventEnable());
            throw new DWRuntimeException("P.eventcenter.500.1003", "事件[" + dto.getEventId() + "]状态不可用!");
        }
        if (CollectionUtils.isEmpty(eventResponseBo.getResponse().getSubscribes())) {
            log.error("订阅事件配置集合(subscribes)为空，发起流程失败");
            throw new DWRuntimeException("P.eventcenter.500.1004", "事件[" + dto.getEventId() + "]订阅事件配置集合不能为空!");
        }
        return eventResponseBo;
    }

    private DWServiceContext getDWServiceContext() {
        try {
            return DWServiceContext.getContext().deepClone();
        } catch (Exception e) {
            log.error("DWServiceContext数据获取异常", e);
            throw new RuntimeException(e);
        }
    }

}