package com.digiwin.athena.eventcenter.service.publish;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.eventcenter.core.constant.Constant;
import com.digiwin.athena.eventcenter.core.util.JsonUtil;
import com.digiwin.athena.eventcenter.core.util.StringUtil;
import com.digiwin.athena.eventcenter.pojo.bo.km.EventResponseBo;
import com.digiwin.athena.eventcenter.pojo.bo.km.EventSubscribeBo;
import com.digiwin.athena.eventcenter.pojo.bo.publish.InvokeProcessBodyBo;
import com.digiwin.athena.eventcenter.pojo.dto.trigger.TriggerEventDto;
import com.digiwin.athena.eventcenter.repository.model.consumer.EventSubscribeLogModel;
import com.digiwin.athena.eventcenter.service.client.dispatch.IDispatchClient;
import com.digiwin.athena.eventcenter.service.srp.component.IEventSubscribeLogService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;

@Slf4j
@Service("eventPublishService")
public class EventPublishService {
    private final IEventSubscribeLogService eventSubscribeLogService;
    private final Map<String, IDispatchClient> dispatchClientMap;

    public EventPublishService(IEventSubscribeLogService eventSubscribeLogService, Map<String, IDispatchClient> dispatchClientMap) {
        this.eventSubscribeLogService = eventSubscribeLogService;
        this.dispatchClientMap = dispatchClientMap;
    }

    /**
     * 发起流程
     */
    @Async("eventExecutor")
    public void invokeProcessAsync(TriggerEventDto dto, String eventLogId, EventResponseBo eventResponseBo, DWServiceContext dwServiceContext) {
        DWServiceContext.setContext(dwServiceContext);
        try {
            List<EventSubscribeBo> subscribes = eventResponseBo.getResponse().getSubscribes();
            for (EventSubscribeBo subscribeBo : subscribes) {
                if (!Constant.SUBSCRIBE_TYPE_WORKFLOW.equals(subscribeBo.getSubscribeType())) {
                    log.error("暂不支持非workflow订阅:[{}]", subscribeBo.getSubscribeType());
                    saveEventConsumeLog(Constant.STATUS_FAILURE, "暂不支持非workflow订阅", dto, eventLogId, subscribeBo);
                    continue;
                }
                if (!Constant.DISPATCH_TYPE_HTTP.equals(subscribeBo.getDispatchType())) {
                    log.error("暂不支持非http派发方式:[{}]", subscribeBo.getDispatchType());
                    saveEventConsumeLog(Constant.STATUS_FAILURE, "暂不支持非http派发方式", dto, eventLogId, subscribeBo);
                    continue;
                }
                if (!Constant.PROCESSOR_TYPE_PROCESS.equals(subscribeBo.getProcessorType())) {
                    log.error("暂不支持发起项目:[{}]", subscribeBo.getProcessorType());
                    saveEventConsumeLog(Constant.STATUS_FAILURE, "暂不支持发起项目", dto, eventLogId, subscribeBo);
                    continue;
                }
                if (subscribeBo.getProcessorConfig() == null || subscribeBo.getProcessorConfig().getProcessId() == null) {
                    log.error("流程id不能为空");
                    saveEventConsumeLog(Constant.STATUS_FAILURE, "流程id不能为空", dto, eventLogId, subscribeBo);
                    continue;
                }
                JSONObject initiator = dto.getEventHeader().getInitiator();
                boolean initiatorFlag = initiator != null && initiator.getString("initiatorId") != null;
                EventSubscribeBo.ProcessorConfigBo processorConfig = subscribeBo.getProcessorConfig();
                String variableStr = processorConfig.getTriggerForm() != null ? processorConfig.getTriggerForm().getString("variable") : null;
                JSONObject variable = StringUtil.isNotBlank(variableStr) ? JsonUtil.getObject(variableStr) : new JSONObject();
                if (dto.getEventHeader().getSourceIds() != null) {
                    variable.put("sourceIds", dto.getEventHeader().getSourceIds());
                }
                InvokeProcessBodyBo bodyBo = InvokeProcessBodyBo.builder()
                        .processId(subscribeBo.getProcessorConfig().getProcessId())
                        .processEOC(dto.getEventHeader().getEocMap())
                        .initiatorId(initiatorFlag ? initiator.getString("initiatorId") : Constant.INITIATOR_DEFAULT)
                        .initiatorNo(initiatorFlag ? initiator.getString("initiatorNo") : Constant.INITIATOR_DEFAULT)
                        .initiatorName(initiatorFlag ? initiator.getString("initiatorName") : Constant.INITIATOR_DEFAULT)
                        .emergency(50)
                        .variables(variable)
                        .dispatchData(dto.getEventBody())
                        .build();
                try {
                    IDispatchClient dispatchClient = dispatchClientMap.get("eventDispatchBy" + StringUtil.firstUpper(subscribeBo.getDispatchType()));
                    dispatchClient.invokeProcess(bodyBo);
                    saveEventConsumeLog(Constant.STATUS_SUCCESS, "", dto, eventLogId, subscribeBo);
                } catch (Exception e) {
                    log.error("流程发起异常", e);
                    saveEventConsumeLog(Constant.STATUS_FAILURE, "流程发起异常", dto, eventLogId, subscribeBo);
                }
            }
        } finally {
            DWServiceContext.getContext().remove();
        }
    }

    private void saveEventConsumeLog(Integer status, String message, TriggerEventDto dto, String eventLogId,
                                     EventSubscribeBo subscribeBo) {
        eventSubscribeLogService.save(new EventSubscribeLogModel(status, message, dto, eventLogId, subscribeBo));
    }

}