package com.digiwin.athena.eventcenter.core.base;

import org.springframework.data.domain.Example;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;

import java.util.List;
import java.util.Map;

/**
 * 标记service接口，用于处理service层面通配处理逻辑
 *
 * @author zhangzhi@digiwin.com
 * @date 2023/09/04
 */
public interface IBaseService<T> {
    /**
     * 保存单个模型对象
     *
     * @param model
     * @return
     */
    T save(T model);

    /**
     * 保存批量模型对象
     *
     * @param models
     * @return
     */
    List<T> save(List<T> models);

    /**
     * 根据主键ID查询符合条件记录
     *
     * @param model
     * @return
     */
    T editById(T model);

    /**
     * 根据传入的模型对象主键ID更新模型对象
     *
     * @param models
     * @return
     */
    List<T> editById(List<T> models);

    /**
     * 根据传入条件查询符合的单条记录
     *
     * @param example
     * @return
     */
    T selectOneByCondition(Example<T> example);

    /**
     * 根据传入条件查询符合的单条记录
     *
     * @param criteria
     * @return
     */
    T selectOneByCondition(Criteria criteria);

    /**
     * 根据传入条件查询符合的列表记录
     *
     * @param example
     * @return
     */
    List<T> listByCondition(Example<T> example);

    /**
     * 自定义条件查询
     *
     * @param criteria
     * @return
     */
    List<T> listByCriteria(Criteria criteria);

    /**
     * 自定义条件查询
     * @param query
     * @return
     */
    List<T> listByCriteria(Query query);

    /**
     * 查询所有记录
     *
     * @return
     */
    List<T> findAll();

    /**
     * 根据条件统计存在的记录数量
     *
     * @param example
     * @return
     */
    long countByCondition(Example<T> example);

    /**
     * 根据条件判断记录是否存在
     *
     * @param example
     * @return
     */
    boolean exists(Example<T> example);

    /**
     * 删除记录
     *
     * @param model
     * @return
     */
    void delete(T model);

    /**
     * 根据单个属性删除匹配记录
     *
     * @param propertyName
     * @param propertyValue
     * @return
     */
    boolean deleteByProperty(String propertyName, String propertyValue);

    /**
     * 根据属性 key/value 集合删除匹配记录
     *
     * @param property
     * @return
     */
    boolean deleteByProperty(Map<String, String> property);

    /**
     * 批量删除多条
     *
     * @param entities
     */
    void deleteAll(Iterable<? extends T> entities);

    /**
     * 自定义条件查询
     *
     * @param criteria
     * @return
     */
    long countByCriteria(Criteria criteria);

    /**
     * 自定义条件查询
     * @param query
     * @return
     */
    long countByCriteria(Query query);

    /**
     * 分页查询
     *
     * @param pageNo   当前页
     * @param pageSize 每页条数
     * @param <E>      转换泛型
     * @return data
     */
    <E> Page<E> page(Integer pageNo, Integer pageSize, Criteria criteria, Class<E> tClass);
}
