package com.digiwin.athena.executionengine.trans.components;

import com.digiwin.athena.executionengine.constant.TransConstant;
import com.digiwin.athena.executionengine.exception.BusinessException;
import com.digiwin.athena.executionengine.model.trans.DealResult;
import com.digiwin.athena.executionengine.model.trans.StepElement;
import com.digiwin.athena.executionengine.model.trans.components.CalculateRuleDto;
import com.digiwin.athena.executionengine.trans.Step;
import com.digiwin.athena.executionengine.trans.TransAbstractStep;
import com.digiwin.athena.executionengine.util.DateUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;

import static com.digiwin.athena.executionengine.constant.CommonConstant.THIRTY_TWO;

/**
 * @description: 计算组件 version2
 * @author: fenglei
 * @date: 2021-1-5
 */
@Component("calculate")
public class CalculateStep extends TransAbstractStep {
    private static final Logger LOGGER = LoggerFactory.getLogger(CalculateStep.class);
    //一天绝对毫秒值
    private static final long ONE_DAY_MILLISECOND = 24 * 60 * 60 * 1000L;

    @Override
    public boolean defineCheck(StepElement stepElement) {
        List<CalculateRuleDto> rules = stepElement.getCalculateRuleDtoList();
        boolean checkResult = false;
        for (CalculateRuleDto rule : rules) {
            if ("WeekOfYearOfDate".equals(rule.getCalcType())) {
                checkResult = checkWeekOfYearOfDate(rule);
            } else if ("Add".equals(rule.getCalcType())) {
                checkResult = checkAdd(rule);
            } else if ("Abs".equals(rule.getCalcType())) {
                checkResult = checkAbs(rule);
            } else if ("Minus".equals(rule.getCalcType())) {
                checkResult = checkMinus(rule);
            } else if ("Multiply".equals(rule.getCalcType())) {
                checkResult = checkMultiply(rule);
            } else if ("Divide".equals(rule.getCalcType())) {
                checkResult = checkDivide(rule);
            } else if ("DateDiffInDays".equals(rule.getCalcType())) {
                checkResult = checkDateDiffInDays(rule);
            } else if ("NumberOfIntervalByDay".equals(rule.getCalcType())) {
                checkResult = checkNumberOfIntervalByDay(rule);
            } else if ("Precision".equals(rule.getCalcType())) {
                checkResult = checkPrecision(rule);
            } else if ("FormatDate".equals(rule.getCalcType())) {
                checkResult = checkFormatDate(rule);
            } else if ("DateFormat".equals(rule.getCalcType())) {
                checkResult = checkDateFormat(rule);
            } else if ("Rooting".equals(rule.getCalcType())) {
                checkResult = checkRootingDate(rule);
            } else if ("Power".equals(rule.getCalcType())) {
                checkResult = checkPowerDate(rule);
            } else if ("GrowthRate".equals(rule.getCalcType())) {
                checkResult = checkGrowthRateDate(rule);
            } else {
                LOGGER.error("未预料的算法:{}", rule.getCalcType());
                checkResult = false;
            }

            if (!checkResult) {
                break;
            }
        }
        return checkResult;
    }

    private boolean checkFormatDate(CalculateRuleDto rule) {
        return !(rule.getFieldA() == null || rule.getNewField() == null
                || rule.getValueType() == null || rule.getFormatA() == null);
    }

    private boolean checkDateFormat(CalculateRuleDto rule) {
        return !(rule.getFieldA() == null || rule.getNewField() == null
                || rule.getValueType() == null || rule.getFormatA() == null);
    }

    private boolean checkRootingDate(CalculateRuleDto rule) {
        return !(rule.getFieldA() == null || rule.getFieldB() == null || rule.getNewField() == null
                || rule.getValueType() == null || rule.getRetain() == null);
    }

    private boolean checkPowerDate(CalculateRuleDto rule) {
        return !(rule.getFieldA() == null || rule.getFieldB() == null || rule.getNewField() == null
                || rule.getValueType() == null || rule.getRetain() == null);
    }

    private boolean checkGrowthRateDate(CalculateRuleDto rule) {
        return !(rule.getFieldA() == null || rule.getFieldB() == null || rule.getNewField() == null
                || rule.getValueType() == null || rule.getRetain() == null);
    }

    /**
     * 检查：绝对值参数
     *
     * @param rule
     * @return
     */
    private boolean checkAbs(CalculateRuleDto rule) {
        return !(rule.getFieldA() == null || rule.getNewField() == null
                || rule.getValueType() == null || rule.getRetain() == null);
    }

    /**
     * precision类型定义检查
     *
     * @param rule
     * @return
     */
    private boolean checkPrecision(CalculateRuleDto rule) {
        if (rule.getFieldA() == null || rule.getPrecisionType() == null || rule.getPrecision() == null
                || rule.getNewField() == null || rule.getValueType() == null || rule.getRetain() == null) {
            LOGGER.error("计算器step Precision 定义错误！Trans执行 {} 失败");
            return false;
        }
        return true;
    }

    @Override
    public DealResult doDealData(Step step) {
        DealResult dataResult = new DealResult();
        try {
            //获取数据，并校验
            Object inputData = getCurrentData(step.getPrevStepNameList());
            if (!(inputData instanceof List)) {
                LOGGER.warn("计算组件只能接受表状数据，非展平数据需要先执行展平节点！Trans执行 {} 失败，数据结构错误！",
                        step.getName());
                dataResult.setDataNullFail();
                return dataResult;
            }
            List<Map<String, Object>> dataList = (List<Map<String, Object>>) inputData;


            for (Map<String, Object> data : dataList) {
                calculate(data, step.getStepElement());
            }

            dataResult.setSuccess(dataList);
            return dataResult;
        } catch (Exception e) {
            LOGGER.error("执行计算组件出现异常", e);
            throw new BusinessException(e.getMessage());
        }
    }

    /**
     * 《**如果增加新的计算支持，需要同步更新TransConstant中的CALCULATE_METHOD常量重的计算类型**》
     *
     * @param data        数据行的map结构，执行结果会根据retain的配置决定是否更新进该结构
     * @param stepElement
     * @return 返回执行算法的结果，该结果并不是作为计算节点输出的数据
     * @throws Exception
     */
    private void calculate(Map<String, Object> data, StepElement stepElement) throws ParseException {
        boolean result;
        //记录每行数据在计算过程中的每个步骤的结果值，该结果可能会被用于后续步骤的输入
        Map<String, Object> extraData = new HashMap<>();
        List<CalculateRuleDto> rules = stepElement.getCalculateRuleDtoList();
        for (CalculateRuleDto rule : rules) {
            if ("WeekOfYearOfDate".equals(rule.getCalcType())) {
                result = calcWeekOfYearOfDate(data, extraData, rule);
            } else if ("Add".equals(rule.getCalcType())) {
                result = calcAdd(data, extraData, rule);
            } else if ("Abs".equals(rule.getCalcType())) {
                result = calcAbs(data, extraData, rule);
            } else if ("Minus".equals(rule.getCalcType())) {
                result = calcMinus(data, extraData, rule);
            } else if ("Multiply".equals(rule.getCalcType())) {
                result = calcMultiply(data, extraData, rule);
            } else if ("Divide".equals(rule.getCalcType())) {
                result = calcDivide(data, extraData, rule);
            } else if ("DateDiffInDays".equals(rule.getCalcType())) {
                result = calcDateDiffInDays(data, extraData, rule);
            } else if ("NumberOfIntervalByDay".equals(rule.getCalcType())) {
                result = calcNumberOfIntervalByDay(data, extraData, rule);
            } else if ("Precision".equals(rule.getCalcType())) {
                result = calcPrecision(data, extraData, rule);
            } else if ("FormatDate".equals(rule.getCalcType())) {
                result = calcFormatDate(data, extraData, rule);
            } else if ("DateFormat".equals(rule.getCalcType())) {
                result = calcDateFormat(data, extraData, rule);
            } else if ("Rooting".equals(rule.getCalcType())) {
                result = rooting(data, extraData, rule);
            } else if ("Power".equals(rule.getCalcType())) {
                result = power(data, extraData, rule);
            } else if ("GrowthRate".equals(rule.getCalcType())) {
                result = growthRate(data, extraData, rule);
            } else {
                LOGGER.error("未预料的算法:{}", rule.getCalcType());
                return;
            }

            if (!result) {
                LOGGER.error("计算时发生错误，计算中断！");
                return;
            }
        }
    }

    private boolean calcDateFormat(Map<String, Object> data, Map<String, Object> extraData, CalculateRuleDto rule) {
        if (StringUtils.isEmpty(String.valueOf(data.get(rule.getFieldA())))) {
            return true;
        }


        if ("true".equals(rule.getRetain())) {
            data.put(rule.getNewField(), data.get(rule.getFieldA()));
        }
        extraData.put(rule.getNewField(), data.get(rule.getFieldA()));
        return true;
    }

    private boolean calcFormatDate(Map<String, Object> data, Map<String, Object> extraData,
                                   CalculateRuleDto rule) {
        String fieldA = getData(data, extraData, rule.getFieldA(), rule.getFieldAType());
        if (StringUtils.isEmpty(fieldA)) {
            return true;
        }
        String inputFormat = analyseDatePattern(fieldA);
        String outputFormat = convertOutputFormat(rule.getFormatA(), rule.getSuffix());

        String date = DateUtils.convertDateFormat(fieldA, inputFormat, outputFormat);

        if ("true".equals(rule.getRetain())) {
            data.put(rule.getNewField(), date);
        }
        extraData.put(rule.getNewField(), date);
        return true;
    }

    /**
     * 日期格式转换
     *
     * @param formatA
     * @param suffix
     * @return
     */
    private String convertOutputFormat(String formatA, String suffix) {
        if (StringUtils.isEmpty(suffix)) {
            return formatA;
        }
        if (StringUtils.isNotEmpty(getTransDataManager().getOutputFormat())) {
            return getTransDataManager().getOutputFormat();
        }
        formatA = formatA.replaceFirst("yyyy", "yyyy年");
        formatA = formatA.replaceFirst("MM", "MM月");
        formatA = formatA.replaceFirst("dd", "dd日");
        getTransDataManager().setOutputFormat(formatA);
        return formatA;
    }

    private String analyseDatePattern(String date) {
        if (StringUtils.isNotEmpty(getTransDataManager().getInputFormat())) {
            return getTransDataManager().getInputFormat();
        }
        String pattern = DateUtils.analyseFormat(date);
        getTransDataManager().setInputFormat(pattern);
        return pattern;
    }

    /**
     * 精度计算
     *
     * @param data
     * @param extraData
     * @param rule
     * @return
     */
    private boolean calcPrecision(Map<String, Object> data, Map<String, Object> extraData, CalculateRuleDto rule) {
        String fieldA = getData(data, extraData, rule.getFieldA(), rule.getFieldAType());
        if (StringUtils.isEmpty(fieldA)) {
            LOGGER.error("执行计算组件Precision步骤时错误，fieldA没有取到数据, data: {}", data);
            return false;
        }
        try {
            BigDecimal bigDecimal;
            switch (rule.getPrecisionType()) {
                case TransConstant.CEIL:
                    bigDecimal = new BigDecimal(fieldA).setScale(Integer.parseInt(rule.getPrecision()), BigDecimal.ROUND_CEILING);
                    break;
                case TransConstant.FLOOR:
                    bigDecimal = new BigDecimal(fieldA).setScale(Integer.parseInt(rule.getPrecision()), BigDecimal.ROUND_FLOOR);
                    break;
                case TransConstant.ROUNDING:
                    bigDecimal = new BigDecimal(fieldA).setScale(Integer.parseInt(rule.getPrecision()), BigDecimal.ROUND_HALF_UP);
                    break;
                case TransConstant.ROUND_UP:
                    bigDecimal = new BigDecimal(fieldA).setScale(Integer.parseInt(rule.getPrecision()), BigDecimal.ROUND_UP);
                    break;
                case TransConstant.ROUND_DOWN:
                    bigDecimal = new BigDecimal(fieldA).setScale(Integer.parseInt(rule.getPrecision()), BigDecimal.ROUND_DOWN);
                    break;
                default:
                    LOGGER.error("未匹配到的precisionType, precisionType: {}", rule.getPrecisionType());
                    return false;
            }

            BigDecimal value = new BigDecimal(bigDecimal.toPlainString());
            if ("true".equals(rule.getRetain())) {
                data.put(rule.getNewField(), getValueByType(value, rule.getValueType()));
            }
            extraData.put(rule.getNewField(), value);
        } catch (NumberFormatException e) {
            LOGGER.error("执行计算组件Precision步骤时错误，fieldA为非number类型, data: {}", data);
            return false;
        }
        return true;
    }

    /**
     * 计算目标日期与当前日期的间隔处于第几个周期
     *
     * @param data
     * @param extraData
     * @param rule
     * @return
     */
    private boolean calcNumberOfIntervalByDay(Map<String, Object> data, Map<String, Object> extraData, CalculateRuleDto rule) throws ParseException {
        String fieldA = getData(data, extraData, rule.getFieldA(), rule.getFieldAType());
        String fieldB = getData(data, extraData, rule.getFieldB(), rule.getFieldBType());
        if (StringUtils.isEmpty(fieldA)) {
            LOGGER.error("执行计算组件Interval步骤时错误，fieldA没有取到数据, data: {}", data);
            return false;
        }
        if (StringUtils.isEmpty(fieldB)) {
            LOGGER.error("执行计算组件Interval步骤时错误，fieldB没有取到数据, data: {}", data);
            return false;
        }

        long fieldALongTime = getFieldLongTime(fieldA, rule.getFormatA());
        long fieldBLongTime = getFieldLongTime(fieldB, rule.getFormatB());
        long difference = Math.abs(fieldALongTime - fieldBLongTime);//得到的两个日期的毫秒数大小
        double day = (double) difference / ONE_DAY_MILLISECOND + 1;
        int period = (int) Math.ceil(day / rule.getIntervalDays());

        if ("true".equals(rule.getRetain())) {
            data.put(rule.getNewField(), getValueByType(new BigDecimal(period), rule.getValueType()));
        }
        extraData.put(rule.getNewField(), period);
        return true;
    }

    /**
     * 获取日期格式的FieldA的longtime
     *
     * @param time
     * @param format
     * @return
     */
    private long getFieldLongTime(String time, String format) throws ParseException {
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        Date date = sdf.parse(time);
        Calendar calendar = sdf.getCalendar();
        calendar.setTime(date);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        return calendar.getTimeInMillis();
    }

    /**
     * 计算：两个参数的加法
     *
     * @param data
     * @param extraData
     * @param rule
     */
    private boolean calcAdd(Map<String, Object> data, Map<String, Object> extraData,
                            CalculateRuleDto rule) {
        String fieldA = getData(data, extraData, rule.getFieldA(), rule.getFieldAType());
        String fieldB = getData(data, extraData, rule.getFieldB(), rule.getFieldBType());

        if (StringUtils.isEmpty(fieldA)) {
            fieldA = "0";
        }

        if (StringUtils.isEmpty(fieldB)) {
            fieldB = "0";
        }

        BigDecimal valueA = new BigDecimal(fieldA);
        BigDecimal valueB = new BigDecimal(fieldB);
        BigDecimal value = valueA.add(valueB);
        value = getResultValue(rule.getPrecision(), value, getRoundingMode(rule.getPrecisionType()));
        if ("true".equals(rule.getRetain())) {
            data.put(rule.getNewField(), getValueByType(value, rule.getValueType()));
        }
        extraData.put(rule.getNewField(), value);

        return true;
    }

    /**
     * 计算：绝对值处理
     *
     * @param data
     * @param extraData
     * @param rule
     */
    private boolean calcAbs(Map<String, Object> data, Map<String, Object> extraData,
                            CalculateRuleDto rule) {
        String fieldA = getData(data, extraData, rule.getFieldA(), rule.getFieldAType());

        if (StringUtils.isEmpty(fieldA)) {
            fieldA = "0";
        }

        BigDecimal valueA = new BigDecimal(fieldA);
        BigDecimal value = valueA.abs();
        value = getResultValue(rule.getPrecision(), value, getRoundingMode(rule.getPrecisionType()));
        if ("true".equals(rule.getRetain())) {
            data.put(rule.getNewField(), getValueByType(value, rule.getValueType()));
        }
        extraData.put(rule.getNewField(), value);

        return true;
    }

    /**
     * 检查：两个参数的加法
     *
     * @param rule
     * @return
     */
    private boolean checkAdd(CalculateRuleDto rule) {
        return !(rule.getFieldA() == null || rule.getFieldB() == null
                || rule.getNewField() == null || rule.getValueType() == null || rule.getRetain() == null);
    }

    /**
     * 计算：两个参数的减法
     *
     * @param data
     * @param extraData
     * @param rule
     */
    private boolean calcMinus(Map<String, Object> data, Map<String, Object> extraData,
                              CalculateRuleDto rule) {
        String fieldA = getData(data, extraData, rule.getFieldA(), rule.getFieldAType());
        String fieldB = getData(data, extraData, rule.getFieldB(), rule.getFieldBType());

        if (StringUtils.isEmpty(fieldA)) {
            fieldA = "0";
        }

        if (StringUtils.isEmpty(fieldB)) {
            fieldB = "0";
        }

        BigDecimal valueA = new BigDecimal(fieldA);
        BigDecimal valueB = new BigDecimal(fieldB);
        BigDecimal value = valueA.subtract(valueB);
        value = getResultValue(rule.getPrecision(), value, getRoundingMode(rule.getPrecisionType()));
        if ("true".equals(rule.getRetain())) {
            data.put(rule.getNewField(), getValueByType(value, rule.getValueType()));
        }
        extraData.put(rule.getNewField(), value);

        return true;
    }

    /**
     * 检查：两个参数的减法
     *
     * @param rule
     * @return
     */
    private boolean checkMinus(CalculateRuleDto rule) {
        return !(rule.getFieldA() == null || rule.getFieldB() == null
                || rule.getNewField() == null || rule.getValueType() == null || rule.getRetain() == null);
    }

    /**
     * 计算：两个参数的乘法
     *
     * @param data
     * @param extraData
     * @param rule
     */
    private boolean calcMultiply(Map<String, Object> data, Map<String, Object> extraData,
                                 CalculateRuleDto rule) {
        String fieldA = getData(data, extraData, rule.getFieldA(), rule.getFieldAType());
        String fieldB = getData(data, extraData, rule.getFieldB(), rule.getFieldBType());

        if (StringUtils.isEmpty(fieldA)) {
            fieldA = "0";
        }

        if (StringUtils.isEmpty(fieldB)) {
            fieldB = "0";
        }

        BigDecimal valueA = new BigDecimal(fieldA);
        BigDecimal valueB = new BigDecimal(fieldB);
        BigDecimal value = valueA.multiply(valueB);
        value = getResultValue(rule.getPrecision(), value, getRoundingMode(rule.getPrecisionType()));

        if ("true".equals(rule.getRetain())) {
            data.put(rule.getNewField(), getValueByType(value, rule.getValueType()));
        }
        extraData.put(rule.getNewField(), value);

        return true;
    }

    /**
     * 检查：两个参数的乘法
     *
     * @param rule
     * @return
     */
    private boolean checkMultiply(CalculateRuleDto rule) {
        return !(rule.getFieldA() == null || rule.getFieldB() == null
                || rule.getNewField() == null || rule.getValueType() == null || rule.getRetain() == null);
    }

    /**
     * 计算：两个参数的除法
     *
     * @param data
     * @param extraData
     * @param rule
     */
    private boolean calcDivide(Map<String, Object> data, Map<String, Object> extraData,
                               CalculateRuleDto rule) {
        String fieldA = getData(data, extraData, rule.getFieldA(), rule.getFieldAType());
        String fieldB = getData(data, extraData, rule.getFieldB(), rule.getFieldBType());

        if (StringUtils.isEmpty(fieldA)) {
            fieldA = "0";
        }

        if (StringUtils.isEmpty(fieldB)) {
            fieldB = "0";
        }

        BigDecimal valueA = new BigDecimal(fieldA);
        BigDecimal valueB = new BigDecimal(fieldB);

        BigDecimal value = new BigDecimal(0);
        /*分母为0的默认处理*/
        if (valueB.compareTo(BigDecimal.ZERO) == 0) {
            if (rule.getExceptionHandling() == null || "divideDefault".equals(rule.getExceptionHandling())) {
                value = valueA.compareTo(BigDecimal.ZERO) == 0 ? new BigDecimal(0) : new BigDecimal(1);
            } else if ("DividedZeroToZero".equals(rule.getExceptionHandling())) {
                //新增n/0=0逻辑 禅道：194526
                value = new BigDecimal(0);
            }
        } else {
            value = valueA.divide(valueB, getPrecision(rule.getPrecision()), getRoundingMode(rule.getPrecisionType()));
            value = new BigDecimal(value.stripTrailingZeros().toPlainString());
        }

        if ("true".equals(rule.getRetain())) {
            data.put(rule.getNewField(), getValueByType(value, rule.getValueType()));
        }
        extraData.put(rule.getNewField(), value);

        return true;
    }

    /**
     * 检查：两个参数的除法
     *
     * @param rule
     * @return
     */
    private boolean checkDivide(CalculateRuleDto rule) {
        /*
        return !(rule.getFieldA() == null || rule.getFieldB() == null || rule.getPrecision() == null || Integer.parseInt(rule.getPrecision()) < 0
                || rule.getNewField() == null || rule.getValueType() == null || rule.getRetain() == null);
         */
        return !(rule.getFieldA() == null || rule.getFieldB() == null || rule.getNewField() == null
                || rule.getValueType() == null || rule.getRetain() == null);
    }

    /**
     * 计算：两个日期的间隔天数
     *
     * @param data
     * @param extraData
     * @param rule
     */
    private boolean calcDateDiffInDays(Map<String, Object> data, Map<String, Object> extraData,
                                       CalculateRuleDto rule) {
        String fieldA = getData(data, extraData, rule.getFieldA(), rule.getFieldAType());
        String fieldB = getData(data, extraData, rule.getFieldB(), rule.getFieldBType());

        if (fieldA == null || fieldA.isEmpty()) {
            LOGGER.error("执行计算组件DateDiffInDays步骤时，fieldA没有取到数据, data: {}", data);
            return false;
        }

        if (fieldB == null || fieldB.isEmpty()) {
            LOGGER.error("执行计算组件DateDiffInDays步骤时，fieldB没有取到数据, data: {}", data);
            return false;
        }

        BigDecimal value;
        try {
            value = DateUtils.getDateDiffInDays(rule.getFormatA(), rule.getFormatB(), fieldA, fieldB);
        } catch (Exception e) {
            LOGGER.error("执行计算组件DateDiffInDays步骤时出现异常, data: {}, extraData: {}", data, extraData);
            return false;
        }

        if ("true".equals(rule.getRetain())) {
            data.put(rule.getNewField(), getValueByType(value, rule.getValueType()));
        }
        extraData.put(rule.getNewField(), value);

        return true;
    }

    /**
     * 计算：nc次方
     *
     * @param data
     * @param extraData
     * @param rule
     */
    private boolean power(Map<String, Object> data, Map<String, Object> extraData,
                          CalculateRuleDto rule) {
        String fieldA = getData(data, extraData, rule.getFieldA(), rule.getFieldAType());
        String fieldB = getData(data, extraData, rule.getFieldB(), rule.getFieldBType());

        if (StringUtils.isEmpty(fieldA)) {
            fieldA = "0";
        }

        if (StringUtils.isEmpty(fieldB) || Integer.parseInt(fieldB) < 0) {
            fieldB = "0";
        }
        BigDecimal valueA = new BigDecimal(fieldA);
        int n = Integer.parseInt(fieldB);

        BigDecimal value = valueA.pow(n);

        value = getResultValue(rule.getPrecision(), value, getRoundingMode(rule.getPrecisionType()));

        if ("true".equals(rule.getRetain())) {
            data.put(rule.getNewField(), getValueByType(value, rule.getValueType()));
        }
        extraData.put(rule.getNewField(), value);

        return true;
    }

    /**
     * 计算：n次根
     *
     * @param data
     * @param extraData
     * @param rule
     */
    private boolean rooting(Map<String, Object> data, Map<String, Object> extraData,
                            CalculateRuleDto rule) {
        String fieldA = getData(data, extraData, rule.getFieldA(), rule.getFieldAType());
        String fieldB = getData(data, extraData, rule.getFieldB(), rule.getFieldBType());

        if (StringUtils.isEmpty(fieldA)) {
            fieldA = "0";
        }

        if (StringUtils.isEmpty(fieldB)) {
            fieldB = "0";
        }

        BigDecimal valueA = new BigDecimal(fieldA);
        BigDecimal value = new BigDecimal(0);
        /*底数A,当取值<0时，取exceptionHandling：baseLessThanZero（底数小于零）*/
        if (Integer.parseInt(fieldA) < 0) {
            if (rule.getExceptionHandling() == null || "baseLessThanZero".equals(rule.getExceptionHandling())) {
                value = new BigDecimal(0);
            }
        } else if (Integer.parseInt(fieldB) < 0) {
            if (rule.getExceptionHandling() == null || "radicandLessEqualZero".equals(rule.getExceptionHandling())) {
                value = new BigDecimal(0);
            }
        } else {
            int n = Integer.parseInt(fieldB);
            value = BigDecimal.valueOf(Math.pow(valueA.doubleValue(), 1.0 / n));
            value = getResultValue(rule.getPrecision(), value, getRoundingMode(rule.getPrecisionType()));
        }

        if ("true".equals(rule.getRetain())) {
            data.put(rule.getNewField(), getValueByType(value, rule.getValueType()));
        }
        extraData.put(rule.getNewField(), value);

        return true;
    }

    /**
     * 计算：增长率
     *
     * @param data
     * @param extraData
     * @param rule
     */
    private boolean growthRate(Map<String, Object> data, Map<String, Object> extraData,
                               CalculateRuleDto rule) {
        String fieldA = getData(data, extraData, rule.getFieldA(), rule.getFieldAType());
        String fieldB = getData(data, extraData, rule.getFieldB(), rule.getFieldBType());

        if (StringUtils.isEmpty(fieldA)) {
            fieldA = "0";
        }

        if (StringUtils.isEmpty(fieldB)) {
            fieldB = "0";
        }

        BigDecimal valueA = new BigDecimal(fieldA);
        BigDecimal valueB = new BigDecimal(fieldB);
        BigDecimal value = new BigDecimal(0);
        if (valueB.compareTo(BigDecimal.ZERO) == 0) {
            if (rule.getExceptionHandling() == null || "divideDefault".equals(rule.getExceptionHandling())) {
                value = valueA.compareTo(BigDecimal.ZERO) == 0 ? new BigDecimal(0) : new BigDecimal(1);
            }
        } else {
            //增长率 = (A - B) ÷ B × 100%
            value = valueA.subtract(valueB).divide(valueB.abs(), getPrecision(rule.getPrecision()), getRoundingMode(rule.getPrecisionType()));
            value = new BigDecimal(value.stripTrailingZeros().toPlainString());
        }

        if ("true".equals(rule.getRetain())) {
            data.put(rule.getNewField(), getValueByType(value, rule.getValueType()));
        }
        extraData.put(rule.getNewField(), value);

        return true;
    }

    public static BigDecimal getResultValue(String precision, BigDecimal value, RoundingMode roundingMode) {
        Integer precisionValue = getPrecision(precision);
        value = value.setScale(precisionValue, roundingMode);
        value = new BigDecimal(value.stripTrailingZeros().toPlainString());
        return value;
    }

    /**
     * 计算精度且设置默认值处理
     *
     * @param precision
     * @return
     */
    public static Integer getPrecision(String precision) {
        int i;
        if (precision == null || !precision.matches("\\d+")) {
            i = THIRTY_TWO; // 默认值
        } else {
            i = Integer.parseInt(precision);
        }
        return i;
    }

    /**
     * 转换精度类型
     *
     * @param precisionType
     * @return
     */
    public static RoundingMode getRoundingMode(String precisionType) {
        if (null == precisionType) {
            return RoundingMode.valueOf(BigDecimal.ROUND_HALF_UP);
        }
        RoundingMode roundingMode;
        switch (precisionType) {
            case TransConstant.CEIL:
                roundingMode = RoundingMode.valueOf(BigDecimal.ROUND_CEILING);
                break;
            case TransConstant.FLOOR:
                roundingMode = RoundingMode.valueOf(BigDecimal.ROUND_FLOOR);
                break;
            case TransConstant.ROUND_UP:
                roundingMode = RoundingMode.valueOf(BigDecimal.ROUND_UP);
                break;
            case TransConstant.ROUND_DOWN:
                roundingMode = RoundingMode.valueOf(BigDecimal.ROUND_DOWN);
                break;
            case TransConstant.ROUNDING:
            default:
                roundingMode = RoundingMode.valueOf(BigDecimal.ROUND_HALF_UP);
                break;
        }
        return roundingMode;
    }

    /**
     * 检查：两个日期的间隔天数
     *
     * @param rule
     * @return
     */
    private boolean checkDateDiffInDays(CalculateRuleDto rule) {
        return !(rule.getFieldA() == null || rule.getFieldB() == null || rule.getFormatA() == null || rule.getFormatB() == null
                || rule.getNewField() == null || rule.getValueType() == null || rule.getRetain() == null);
    }

    /**
     * 检查计算周期偏移的定义
     *
     * @param rule
     * @return
     */
    private boolean checkNumberOfIntervalByDay(CalculateRuleDto rule) {
        return !(rule.getFieldA() == null || rule.getFormatA() == null || rule.getFieldAType() == null
                || rule.getFieldB() == null || rule.getFormatB() == null || rule.getFieldBType() == null
                || rule.getNewField() == null || rule.getValueType() == null || rule.getRetain() == null
                || rule.getIntervalDays() == 0);
    }

    /**
     * 计算：根据日期计算自然年中属于第几周
     *
     * @param data
     * @param extraData
     * @param rule
     */
    private boolean calcWeekOfYearOfDate(Map<String, Object> data, Map<String, Object> extraData,
                                         CalculateRuleDto rule) {
        try {
            String fieldA = getData(data, extraData, rule.getFieldA(), rule.getFieldAType());
            if (StringUtils.isEmpty(fieldA)) {
                LOGGER.error("执行计算组件WeekOfYearOfDate步骤时错误，fieldA没有取到数据, data: {}", data);
                return false;
            }

            //计算时距
            BigDecimal value = new BigDecimal(DateUtils.getWeekOfYear(fieldA, rule.getFormatA()));

            if ("true".equals(rule.getRetain())) {
                data.put(rule.getNewField(), getValueByType(value, rule.getValueType()));
            }
            extraData.put(rule.getNewField(), value);
        } catch (Exception e) {
            LOGGER.error("执行计算组件WeekOfYearOfDate步骤时出现异常, data: {}, extraData: {}", data, extraData);
        }

        return true;
    }

    /**
     * 检查：根据日期计算自然年中属于第几周
     *
     * @param rule
     * @return
     */
    private boolean checkWeekOfYearOfDate(CalculateRuleDto rule) {
        return !(rule.getFieldA() == null || rule.getFormatA() == null || rule.getFormatA().isEmpty()
                || rule.getNewField() == null || rule.getValueType() == null || rule.getRetain() == null);
    }


    private String getData(Map<String, Object> data, Map<String, Object> extraData, String fieldName, String fieldType) {
        if (StringUtils.isEmpty(fieldType)) {
            fieldType = TransConstant.SOURCE_TYPE_COLUMN;
        }
        if (TransConstant.SOURCE_TYPE_CONSTANT.equalsIgnoreCase(fieldType)) {
            return fieldName;
        }
        if (TransConstant.SOURCE_TYPE_COLUMN.equalsIgnoreCase(fieldType)) {
            return getData(data, extraData, fieldName);
        }
        return null;
    }

    private String getData(Map<String, Object> data, Map<String, Object> extraData, String fieldName) {
        Object value;

        value = data.get(fieldName);
        if (value != null) {
            return value.toString();
        }

        value = extraData.get(fieldName);
        if (value != null) {
            return value.toString();
        }

        return null;
    }

    private Object getValueByType(BigDecimal value, String valueType) {
        if (value == null) {
            LOGGER.error("计算结果转换时错误，计算结果为null!");
            return null;
        }

        //TODO 迭代16 故事单1073需要后续处理
        if ("String".equals(valueType) || "string".equals(valueType)) {
            return value.toString();
        } else if ("Number".equals(valueType) || "number".equals(valueType)) {
            return value;
        } else {
            LOGGER.error("执行计算组件add步骤时错误，未预料的valueType: {}", valueType);
            return null;
        }
    }

}