package com.digiwin.athena.executionengine.service.facade.mapping.data;

import com.digiwin.athena.executionengine.constant.CommonConstant;
import com.digiwin.athena.executionengine.core.holder.ReadContextHolder;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;


/**
 * @description: 持有一个数据源的字段信息对象集合，并提供对这个字段信息对象集合的操作，
 * 该集合包含来自一个数据源（JSON）的所有字段对象（DataNode）
 * 持有字段的上下关系、层次分区信息、数据形态认知信息等，但不保存数据
 * @author: fenglei
 * @date: 2020-11-9
 */
public class DataMappingHandler {

    private ReadContextHolder readContextHolder;

    /**
     * json中所有节点的集合，DataNode包含运行时下标，通过hashMap深拷贝到不同的运行实例。
     * --未来这里可以将DataNode中的运行时数据单独存放，而其他固化数据可以放在单例中维护。
     * key：nodeId
     * value：dataNode结构
     */
    private HashMap<String, DataNode> mappingDataCollection;

    /**
     * 查询索引，记录paramPath和nodeId之间一对一的关系
     * 在复制实例时，不需要对索引进行赋值！
     * key: paramPath
     * key: nodeId
     */
    private Map<String, String> idxPathToNode;

    /**
     * 判断映射关系是否在数据上存在
     *
     * @param path
     * @return
     */
    public boolean pathIsExistsInData(String path) {
        return idxPathToNode.containsKey(path);
    }

    public DataMappingHandler(HashMap<String, DataNode> mappingDataCollection, ReadContextHolder readContextHolder) {
        this.mappingDataCollection = mappingDataCollection;
        if(readContextHolder == null){
            readContextHolder = new ReadContextHolder();
        }
        this.readContextHolder = readContextHolder;
        idxPathToNode = new HashMap<>();
        if (!mappingDataCollection.isEmpty()) {
            for (HashMap.Entry<String, DataNode> mappingData : mappingDataCollection.entrySet()) {
                idxPathToNode.put(mappingData.getValue().getParamPath(), mappingData.getKey());
            }
        }
    }

    protected HashMap<String, DataNode> getDataCollection() {
        return mappingDataCollection;
    }

    /**
     * 根据更新paramPath的查询索引
     *
     * @param dataNode
     */
    private void setIdxPathToNode(DataNode dataNode) {
        if (dataNode != null && !idxPathToNode.containsKey(dataNode.getParamPath())) {
            idxPathToNode.put(dataNode.getParamPath(), dataNode.getNodeId());
        }
    }

    /**
     * 将paramPath转换成JsonRoute
     *
     * @param paramPath 参数路径字符串 $.A.B.C.D
     * @return
     */
    private List<String> convertToJsonRoute(String paramPath) {
        if (StringUtils.isEmpty(paramPath)) {
            return null;
        }

        List<String> jsonRoute = new ArrayList<>();
        String path = paramPath;
        int pos = path.lastIndexOf(".");

        while (pos > 0) {
            jsonRoute.add(idxPathToNode.get(path));
            path = path.substring(0, pos);
            pos = path.lastIndexOf(".");
        }

        jsonRoute.add(idxPathToNode.get(path));

        //做个逆序
        Collections.reverse(jsonRoute);
        return jsonRoute;
    }

    /**
     * 创建一个节点，该节点只创建了nodeId
     *
     * @return
     */
    public DataNode createNode(boolean isRootNode, String nodeName, String parentNodeId, String lastCollectionNode,
                               int totalCnt, boolean isObject, boolean isArray, boolean isArrayAsObject,
                               String pathTemplate, String paramPath) {

        String name = nodeName;
        if (isRootNode) {
            name = ".ROOT";
        }

        DataNode dataNode = new DataNode(name, parentNodeId, lastCollectionNode, totalCnt, isObject,
                isArray, isArrayAsObject, pathTemplate, paramPath, null);
        //立刻更新索引
        setIdxPathToNode(dataNode);
        mappingDataCollection.put(dataNode.getNodeId(), dataNode);

        if (!isRootNode) {
            dataNode.setJsonRoute(convertToJsonRoute(paramPath));
        }
        return dataNode;
    }

    /**
     * 遍历数据时构建含有实际意义的对象
     *
     * @return
     */
    public DataNode createNode(String nodeName, Object jsonDataPart, DataNode parentNode,
                               String subFieldParamPath, Object jsonDataWhole) {
        boolean isArray = false;
        boolean isObject = false;

        //判断并设置 isArray 和 array中含有Object A[]/A[{},{}...]
        if (jsonDataPart instanceof List) {
            isArray = true;
            List subCollection = (List) jsonDataPart;
            if (subCollection.size() > 0) {
                if (subCollection.get(0) instanceof Map) {
                    isObject = true;
                }
            }
        }

        //判断并设置 isObject A{}
        if (jsonDataPart instanceof Map) {
            isObject = true;
        }

        //取该节点在整个response json中的数据量
        String pathTemplate = getPathTemplate(parentNode) + "." + nodeName;
        //String pathForLength = pathTemplate + ".length()";
        if (isArray) {
            pathTemplate = pathTemplate + "[*]";
        }

        ////通过JsonPath取size   这里用jsonPath取全局的数据量是取不到的
        //int totalCnt = (Integer) JsonResolverUtils.getJsonPathVal(jsonDataWhole, pathForLength);
        int totalCnt = 0;

        //取父节点id和上层中最近的一个集合节点
        String parentDataNodeId = getDataNodeId(parentNode);
        String lastCollectionNode = isArray(parentNode)
                ? getDataNodeId(parentNode)
                : getLastCollectionNode(parentNode);


        //创建一个新的节点，并得到该节点实例
        DataNode dataNode = createNode(false, nodeName, parentDataNodeId,
                lastCollectionNode, totalCnt, isObject, isArray, false, pathTemplate,
                subFieldParamPath);
        mappingDataCollection.put(dataNode.getNodeId(), dataNode);
        return dataNode;
    }

    /**
     * 根据当前节点的是否list来计算当前节点在全局数据中的数量
     *
     * @param dataNode
     * @param jsonDataWhole  整个json数据结构
     * @param key            保证同一个数据只生成一个jsonpath的ReadContext对象
     * @param jsonLengthPath $.A[*].B[*].C
     * @return
     */
    public int calcZoneSize(DataNode dataNode, Object jsonDataWhole, String key, String jsonLengthPath) {

        int zoneSize = 1;
        if (dataNode.isArray()) {
            ///$.A[*].B[*].C.length 求取数据量
            String concretePathForLength = jsonLengthPath + ".length()";
            zoneSize = (Integer) readContextHolder.getData(key, jsonDataWhole, concretePathForLength);
        }

        return zoneSize;
    }

    /**
     * 设置当前路径下的该节点的实际数据量
     * 例如: $.A.B.C.D, dataNode描述D节点。
     *
     * @param concreteJsonPath   当前dataNode的实例jsonPath ($.A[0].B[2].C[1].D)
     * @param parentConcretePath 上一层的实例jsonPath ($.A[0].B[2].C[1])
     * @param dataNode           计算的则是该节点的数据量 , (D in $.A.B.C.D)
     */
    public void setZone(String concreteJsonPath, String parentConcretePath, DataNode dataNode, int zoneSize) {

        if (!isExistZone(dataNode, parentConcretePath)) {
            // zone的结构是父层的实例路径对应的该节点数据量，
            // 所以这里用 parentConcretePath 对应 concreteJsonPath 取出的size
            // 更新到 concreteJsonPath 对应的dataNode
            addZone(dataNode, parentConcretePath, zoneSize);
        }
    }

    /**
     * 追加totalCnt的数量
     */
    public void appendTotalCnt(DataNode dataNode, int count) {
        dataNode.addTotalCnt(count);
    }

    /**
     * 根据nodeId取节点对象
     *
     * @param nodeId
     * @return
     */
    public DataNode getDataNode(String nodeId) {
        return mappingDataCollection.get(nodeId);
    }

    /**
     * 根据nodeId取节点对象
     *
     * @param dataNode
     * @return
     */
    public String getDataNodeId(DataNode dataNode) {
        if (dataNode == null) {
            return null;
        }
        return dataNode.getNodeId();
    }

    /**
     * 根据nodeId取节点对象
     *
     * @param dataNode
     * @return
     */
    public String getDataNodeName(DataNode dataNode) {
        if (dataNode == null) {
            return null;
        }
        return dataNode.getNodeName();
    }

    /**
     * 根据nodeId取节点对象
     *
     * @param paramPath
     * @return
     */
    public DataNode getDataNodeByParamPath(String paramPath) {
        String nodeId = idxPathToNode.get(paramPath);
        if (nodeId == null || nodeId.isEmpty()) {
            return null;
        }
        return mappingDataCollection.get(nodeId);
    }

    /**
     * 根据node实例获取paramPath
     *
     * @param dataNode
     * @return
     */
    public String getParamPath(DataNode dataNode) {
        return dataNode.getParamPath();
    }

    /**
     * 根据nodeId获取对应的paramPath
     *
     * @param nodeId
     * @return 如果没有找到返回null
     */
    public String getParamPath(String nodeId) {
        if (!mappingDataCollection.containsKey(nodeId)) {
            return null;
        }
        return mappingDataCollection.get(nodeId).getParamPath();
    }

    /**
     * 根据node实例获取paramName
     *
     * @param dataNode
     * @return
     */
    public String getNodeName(DataNode dataNode) {
        return dataNode.getNodeName();
    }

    /**
     * 根据nodeId获取对应的paramName
     *
     * @param nodeId
     * @return 如果没有找到返回null
     */
    public String getNodeName(String nodeId) {
        if (!mappingDataCollection.containsKey(nodeId)) {
            return null;
        }
        return mappingDataCollection.get(nodeId).getNodeName();
    }

    /**
     * 根据node实例获取pathTemplate
     *
     * @param dataNode
     * @return
     */
    public String getPathTemplate(DataNode dataNode) {
        return dataNode.getPathTemplate();
    }

    /**
     * 根据nodeId获取对应的pathTemplate
     *
     * @param nodeId
     * @return 如果没有找到返回null
     */
    public String getPathTemplate(String nodeId) {
        if (!mappingDataCollection.containsKey(nodeId)) {
            return null;
        }
        return mappingDataCollection.get(nodeId).getPathTemplate();
    }

    /**
     * 根据node实例获取pathTemplate
     *
     * @param paramPath
     * @return
     */
    public String getPathTemplateByPath(String paramPath) {
        DataNode dataNode = getDataNodeByParamPath(paramPath);
        return dataNode == null ? null : dataNode.getPathTemplate();
    }

    /**
     * 设定节点是一个空map结构
     */
    public void setEmptyMapNode(String nodeId) {
        getDataNode(nodeId).setEmptyMap();
    }

    /**
     * 设定节点是一个空map结构
     */
    public void setEmptyMapNode(DataNode dataNode) {
        dataNode.setEmptyMap();
    }

    /**
     * 取所有下一层级的字段
     *
     * @param dataNode 指定的dataNode
     * @return 返回字段的nodeId，如果没有找到返回null
     */
    public List<String> getChildrenId(DataNode dataNode) {
        if (dataNode == null || dataNode.getChildren() == null) {
            return null;
        }

        return dataNode.getChildren();
    }

    /**
     * 取所有下一层级的字段
     *
     * @param nodeId 指定的nodeId
     * @return 返回字段的nodeId，如果没有找到返回null
     */
    public List<String> getChildrenId(String nodeId) {
        return getChildrenId(mappingDataCollection.get(nodeId));
    }

    /**
     * 取当前节点父节点的所在pos位置
     *
     * @return
     */
    private int getLastCollectionZonePos(DataNode node) {
        //如果是顶层节点，直接返回顶层节点的下标0
        if (CommonConstant.ROOT_NODE.equals(node.getNodeName())) {
            return 0;
        }

        DataNode parentNode = getDataNode(node.getParentNodeId());
        return parentNode.getZonePos();
    }

    /**
     * 递归向父层创建jsonPath的实例，自己如果是list结构就取ZonePos创建[?]的内容
     *
     * @param node
     * @return
     */
    private String getJsonPathInst(DataNode node) {
        String parentNodeId = node.getParentNodeId();
        String jsonPathInst = "";
        if (!CommonConstant.ROOT_NODE.equals(node.getNodeName())) {
            jsonPathInst = getJsonPathInst(mappingDataCollection.get(parentNodeId));
            jsonPathInst = jsonPathInst + "." + node.getNodeName();
        } else {
            jsonPathInst = "$";
        }

        if (node.isArray()) {
            jsonPathInst = jsonPathInst + "[" + node.getZonePos() + "]";
        }

        return jsonPathInst;
    }

    /**
     * 取当前字段所在数据集合的总数量
     *
     * @param node 当前节点
     * @return
     */
    private int getZoneSize(DataNode node) {
        Map<String, Integer> zone = node.getZone();
        String jsonPathInst = getJsonPathInst(node);

        ///当前的node（paramPath）是A.B.C.D，jsonPathInst是：$.A.B.C.D
        ///但是这时node中的zone key：$.A.B.C
        int pos = jsonPathInst.lastIndexOf(".");
        ///pos<=0 说明jsonPathInst一定是根节点了
        if (pos > 0) {
            jsonPathInst = jsonPathInst.substring(0, pos);
        } else {
            //这里比较特殊，如果当前的节点是.ROOT，那么它的zone的key，就是$
            jsonPathInst = "$";
        }

        System.out.println(jsonPathInst);
        Integer zoneSize = zone.get(jsonPathInst);
        return zoneSize == null ? 0 : zoneSize.intValue();
    }

    /**
     * 根据paramPath取size
     * paramPath: $.A.A3   =>  $[0].A[0].A3.size，
     * 根据paramPath获取具体路径，并根据具体jsonpath路径，获取路径下该节点的数据量
     *
     * @param paramPath
     * @return
     */
    public int getZoneDataSize(String paramPath) {
        DataNode dataNode = getDataNodeByParamPath(paramPath);
        int size = 0;
        if (dataNode != null) {
            if (CommonConstant.ROOT_NODE.equals(dataNode.getNodeName())) {
                size = dataNode.getZone().get("$");
            } else {
                String zoneKey = getConcreteJsonPath(getParentNode(dataNode));
                size = getSizeInZone(dataNode, zoneKey);
            }

        }
        return size;
    }

    /**
     * 特殊的做法  ----->API
     * 根据固定的jsonPath取对应路径下该节点的数据量大小
     *
     * @param paramPath paramPath用于找出DataNode
     * @param fixedPath 实例jsonPath，处理后用于匹配zoneKey
     * @return
     */
    public int getZoneDataSize(String paramPath, String fixedPath) {
        int size = 0;
        if (paramPath == null || paramPath.isEmpty() || fixedPath == null || fixedPath.isEmpty()) {
            return size;
        }

        DataNode dataNode = getDataNodeByParamPath(paramPath);
        if (dataNode != null) {
            if (CommonConstant.ROOT_NODE.equals(dataNode.getNodeName())) {
                size = dataNode.getZone().get("$");
            } else {
                String zoneKey = fixedPath.substring(0, fixedPath.lastIndexOf("."));
                size = getSizeInZone(dataNode, zoneKey);
            }

        }
        return size;
    }

    /**
     * 根据paramPath取size
     * paramPath: $.A.A3   =>  $[*].A[*].A3.size
     * 根据paramPath，取得属于这个该paramPath关系的所有具体路径下的数据量
     *
     * @param paramPath
     * @param fixedPath 固定的路径，在取整个size的时候，需要指定一个范围，
     *                  取size会在整个数据节点下进行统计。
     *                  fixedPath = $[0]
     *                  paramPath = $.A.A3
     *                  那么，得到sum（ $[0].A[0].A3,
     *                  $[0].A[1].A3,
     *                  ...
     *                  $[0].A[x].A3）
     *                  如果该字段传"", 则统计所有
     * @return
     */
    public int getPathDataSize(String paramPath, String fixedPath) {
        DataNode dataNode = getDataNodeByParamPath(paramPath);
        int size = 0;
        if (dataNode == null) {
            return size;
        }

        if (fixedPath != null && "".equals(fixedPath)) {
            size = dataNode.getTotalCnt();
        } else {
            if (dataNode.getZone() != null && dataNode.getZone().size() != 0) {
                for (Map.Entry<String, Integer> zoneSize : dataNode.getZone().entrySet()) {
                    if (zoneSize.getKey().contains(fixedPath)) {
                        Integer sizeValue = zoneSize.getValue();
                        size += (sizeValue == null ? 0 : sizeValue.intValue());
                    }
                }
            }
        }
        return size;
    }

    /**
     * 根据paramPath取size
     * paramPath: $.A.A3   =>  $[*].A[*].A3.size
     * 根据paramPath，取得属于这个该paramPath关系的所有具体路径下的数据量
     *
     * @param paramPath 根绝paramPath找其上一个父节点，根据GlobalPos确定当前参数的jsonPath，
     *                  调用getPathDataSize(String paramPath, String fixedPath)方法
     *                  paramPath = $.A.A3
     *                  那么，得到sum（ $[0].A[0].A3,
     *                  $[0].A[1].A3,
     *                  ...
     *                  $[0].A[x].A3）
     *                  如果该字段传"", 则统计所有
     *                  注：如果传入的path本身就是$,那么返回的就是$的size
     * @return
     */
    public int getPathDataSize(String paramPath) {
        DataNode dataNode = getDataNodeByParamPath(paramPath);
        if (dataNode == null) {
            return 0;
        }

        if (CommonConstant.ROOT_NODE.equals(dataNode.getNodeName())) {
            return dataNode.getTotalCnt();
        }

        DataNode parentNode = getParentNode(dataNode);
        String fixedPath = concatJsonPath(parentNode);
        return getPathDataSize(paramPath, fixedPath);
    }

    /**
     * 根据paramPath获取集合层级数 ---> API
     *
     * @param paramPath
     * @return
     */
    public int countCollectionInPath(String paramPath) {
        DataNode dataNode = getDataNodeByParamPath(paramPath);
        return countCollectionInPath(dataNode);
    }

    /**
     * 根据paramPath获取集合层级数
     *
     * @param dataNode
     * @return
     */
    private int countCollectionInPath(DataNode dataNode) {
        if (dataNode != null && dataNode.getLastCollectionNode() != null) {
            return countCollectionInPath(getDataNode(dataNode.getLastCollectionNode())) + 1;
        } else {
            return 0;
        }
    }

    /**
     * 根据paramPath获取集合层paramPath, ---> API
     *
     * @param paramPath
     * @return
     */
    public List<String> getCollectionParamPathList(String paramPath) {
        DataNode dataNode = getDataNodeByParamPath(paramPath);
        List<String> pathList = getCollectionParamPathList(dataNode);

        Collections.sort(pathList);
        return pathList;
    }

    /**
     * 根据paramPath获取集合层paramPath
     *
     * @param dataNode
     * @return
     */
    private List<String> getCollectionParamPathList(DataNode dataNode) {
        List<String> paramPathList = new ArrayList<>();
        if (dataNode != null && dataNode.getLastCollectionNode() != null) {
            DataNode parentNode = getDataNode(dataNode.getLastCollectionNode());
            paramPathList.add(parentNode.getParamPath());
            paramPathList.addAll(getCollectionParamPathList(parentNode));
        }
        return paramPathList;
    }

    /**
     * 根据paramPath判断是否是参数是否是集合
     * 输入A.B.C，给出C是否是集合结构
     *
     * @param paramPath
     * @return
     */
    public boolean isCollectionParam(String paramPath) {
        return getDataNodeByParamPath(paramPath) == null ? true : getDataNodeByParamPath(paramPath).isArray();
    }

    /**
     * 根据paramPath判断是否是含有对象的集合
     * 注意：使用该方法需要先验证paramPath在数据中是否存在！外部校验dataNode是否为空
     * 输入A.B.C，给出C是否是由对象组成的集合结构
     *
     * @param paramPath
     * @return
     */
    public boolean isObjectCollectionParam(String paramPath) {
        DataNode dataNode = getDataNodeByParamPath(paramPath);
        return dataNode.isArray() && dataNode.isObject();
    }

    /**
     * 检查当前节点是否有值，没有值继续递归
     *
     * @param node
     * @return
     */
    private int isExistZone(DataNode node) {
        int lastCollectionZonePos = -1;
        String zoneKey = getConcreteJsonPath(node);
        if (!node.getZone().containsKey(zoneKey)) {
            System.out.println("----> 当前节点不存在，重新指定:" + zoneKey);
            lastCollectionZonePos = nextPos(mappingDataCollection.get(node.getLastCollectionNode()));
        }

        return lastCollectionZonePos;
    }

    /**
     * array数据取下一个有效记录的下标，
     * 当前集合中如果无法获取更多值，将重新计算下个值所在的位置，并更新涉及到的上层节点的数据下标。
     *
     * @param node
     * @return 返回一个int，用于记录当前数据的下标
     */
    private int nextPos(DataNode node) {

        // 获取当前节点所在zone的实例总数
        /////////fenglei：原先的想法是，通过最后一个集合层级，取其zone的key，然后再拼接后面的路径，可以提高效率
        ///////////////// 现在就是在递归遍历父层次的当前取值下标。
        int zoneSize = getZoneSize(node);

        // 获取当前节点在当前zone中所处的索引
        int zonePos = node.getZonePos();

        // 试算并决定是否递归
        // 如果是对象，没有就直接找上层中的集合
        if (node.isArray()) {
            if (zonePos + 1 < zoneSize) {
                //zone里面能取到数据，可以直接取
                int tmp = zonePos;
                zonePos++;
                System.out.println("数据充足，node.Name:" + node.getNodeName() + " pos from "
                        + tmp + " to " + zonePos);
                node.setZonePos(zonePos);
                return zonePos;
            } else {
                ///默认，上面已经没有集合层了，默认：现在整个数据层次链路已经无法找到所需要的叶子数据。
                int lastCollectionZonePos = -1;

                //当前zone中已经没有值可以取了，找父层中的集合层移动下标
                if (node.getLastCollectionNode() != null) {
                    System.out.println("即将进入：" + mappingDataCollection.get(node.getLastCollectionNode()).getNodeName());
                    lastCollectionZonePos = nextPos(mappingDataCollection.get(node.getLastCollectionNode()));
                    //父集合层做next，返回其下标为0，或者大于0，说明父集合层中完成了移动下标的处理。本层要重置zonePos
                    if (lastCollectionZonePos >= 0) {
                        int tmp = zonePos;
                        zonePos = 0;

                        System.out.println("nextPos: node.Name:" + node.getNodeName() + " pos from "
                                + tmp + " to " + zonePos);
                        node.setZonePos(zonePos);
                    }
                }
                return lastCollectionZonePos;
            }
        } else {
            //这里不会存在递归中，只会在递归开始，递归开始前，可能是一个对象、一个字段、或者一个集合
            System.out.println("即将进入：" + mappingDataCollection.get(node.getLastCollectionNode()).getNodeName());
            return nextPos(mappingDataCollection.get(node.getLastCollectionNode()));
        }
    }

    /**
     * 对外提供的步进接口，传入paramPath，将对该的数据进行步进。
     * 无论当前节点是单值，还是map，还是list，步进将递增dataNode的zonePos，并在zone的pos尽头，递归上层的内容
     *
     * @param paramPath 参数路径 A.B.C.D.E
     * @param limitPath 允许往上层步进的范围。
     * @return true: 步进成功， false: 全部用完，无法继续步进
     */
    public boolean next(String paramPath, String limitPath) {
        if (nextPos(getDataNodeByParamPath(paramPath)) < 0) {
            return false;
        }
        return true;
    }

    /**
     * 取当前节点实例的JsonPath
     *
     * @param paramPath 指定的paramPath A.B.C.D
     * @return 返回字段的nodeId，如果没有找到返回null
     */
    public String getConcreteJson(String paramPath) {
        DataNode dataNode = getDataNodeByParamPath(paramPath);
        return getConcreteJsonPath(dataNode);
    }

    /**
     * 取当前节点实例的JsonPath
     *
     * @param node 指定的dataNode
     * @return 返回字段的nodeId，如果没有找到返回null
     */
    public String getConcreteJsonPath(DataNode node) {

        if (node == null || node.getChildren() == null) {
            return null;
        }

        String concretePath = node.getPathTemplate();
        List<String> nodeIdList = node.getJsonRoute();
        int pos = 0;
        DataNode nodeInRoute;
        //将jsonPath模板中的*替换成实例数据的下标
        for (String nodeId : nodeIdList) {
            nodeInRoute = getDataNode(nodeId);
            if (!nodeInRoute.isArray()) {
                continue;
            }

            pos = getZonePos(nodeInRoute);

            //替换掉第一次出现的*, 正则表达
            concretePath = concretePath.replaceFirst("\\*", String.valueOf(pos));
        }

        return concretePath;
    }

    /**
     * 取当前节点实例的JsonPath
     *
     * @param currentNodeId 指定的nodeId
     * @return 返回字段的nodeId，如果没有找到返回null
     */
    public String getConcreteJsonPath(String currentNodeId) {
        return getConcreteJsonPath(mappingDataCollection.get(currentNodeId));
    }

    public int getZonePos(DataNode dataNode) {
        return dataNode.getZonePos();
    }

    /**
     * 根据当前节点取其上层中最近的集合节点
     *
     * @param dataNode
     * @return
     */
    public String getLastCollectionNode(DataNode dataNode) {
        return dataNode.getLastCollectionNode();
    }

    /**
     * 根据当前节点取其上层中最近的集合节点
     *
     * @param paramPath
     * @return
     */
    public String getLastCollectionNodeParamPath(String paramPath) {
        DataNode node = getDataNodeByParamPath(paramPath);
        if (node == null) {
            return null;
        }
        DataNode lastCollectionNode = getDataNode(node.getLastCollectionNode());
        if (lastCollectionNode == null) {
            return null;
        }
        return lastCollectionNode.getParamPath();
    }

    /**
     * 根据当前节点取是否array
     *
     * @param dataNode
     * @return
     */
    public boolean isArray(DataNode dataNode) {
        return dataNode.isArray();
    }

    /**
     * 根据当前节点path判断是否array
     *
     * @param paramPath
     * @return
     */
    public boolean isArray(String paramPath) {
        return getDataNodeByParamPath(paramPath).isArray();
    }

    public boolean isExistZone(DataNode dataNode, String zonePath) {
        Map<String, Integer> zone = dataNode.getZone();
        return zone.containsKey(zonePath);
    }

    public void addZone(DataNode dataNode, String zonePath, int zoneSize) {
        Integer size = new Integer(zoneSize);
        dataNode.addZone(zonePath, size);
    }

    public void addChildNode(DataNode parentNode, DataNode childNode) {
        parentNode.addChildren(childNode.getNodeId());
    }

    /**
     * 根据dataNode和该节点当前的实例路径，获取其在zone中的大小
     * dataNode: 节点D: A.B.C.D
     * zonePath: $[0].A[0].B[2].C[1]  ->  获得D这个节点的在当前路径下的数据集大小
     *
     * @param dataNode
     * @param zonePath
     * @return
     */
    public int getSizeInZone(DataNode dataNode, String zonePath) {
        Integer size = dataNode.getZone().get(zonePath);
        return size == null ? 0 : size.intValue();
    }

    /**
     * 同public int getSizeInZone(DataNode dataNode, String zonePath);
     *
     * @param nodeId   当前的DataNode
     * @param zonePath 这里需要用其父层的jsonPath实例路径去match
     * @return
     */
    public int getSizeInZone(String nodeId, String zonePath) {
        DataNode dataNode = mappingDataCollection.get(nodeId);
        return getSizeInZone(dataNode, zonePath);
    }

    /**
     * 根据入参nodeId的节点，取其父节点dataNode对象
     *
     * @param dataNode
     * @return
     */
    public DataNode getParentNode(DataNode dataNode) {
        String parentNodeId = dataNode.getParentNodeId();
        if (parentNodeId == null) {
            return null;
        }
        DataNode parentDataNode = mappingDataCollection.get(parentNodeId);
        return parentDataNode;
    }

    /**
     * 根据入参nodeId的节点，取其父节点dataNode对象
     *
     * @param nodeId
     * @return
     */
    public DataNode getParentNode(String nodeId) {
        return getParentNode(mappingDataCollection.get(nodeId));
    }

    /**
     * 当该数据节点是一个集合的时候，设定DataNode当前数据映射中该节点取数据的下标
     * <注意>该方法只设置了传入的节点的pos，并不会自动将其子集合节点pos重置！！</注意>
     *
     * @param paramPath
     * @param pos
     */
    public void setParentNodePosition(String paramPath, int pos) {
        DataNode dataNode = getParentNode(getDataNodeByParamPath(paramPath).getNodeId());
        if (dataNode != null && dataNode.isArray()) {
            dataNode.setGlobalPos(pos);
        }
    }

    /**
     * 当该数据节点是一个集合的时候，设定DataNode当前数据映射中该节点取数据的下标
     * <注意>该方法只设置了传入的节点的pos，并不会自动将其子集合节点pos重置！！</注意>
     *
     * @param paramPath
     * @param pos
     */
    public void setNodePosition(String paramPath, int pos) {
        DataNode dataNode = getDataNodeByParamPath(paramPath);
        if (dataNode != null && dataNode.isArray()) {
            dataNode.setGlobalPos(pos);
        }
    }

    public String getDataJsonPath(String paramPath) {
        DataNode dataNode = getDataNodeByParamPath(paramPath);
        if (dataNode == null) {
            return null;
        }
        return concatJsonPath(dataNode);
    }

    private String concatJsonPath(DataNode dataNode) {

        String jsonPath;
        if (CommonConstant.ROOT_NODE.equals(dataNode.getNodeName())) {
            jsonPath = "$";
        } else {
            jsonPath = concatJsonPath(getParentNode(dataNode.getNodeId()));
            jsonPath = jsonPath + "." + dataNode.getNodeName();
        }

        ///来自参数映射的要求
        // 如果dataNode是集合并且也是对象，说明这是一个path节点
        // 如果dataNode仅仅是一个集合不是对象，说明这是一个简单集合节点，属于数据节点，
        //     映射中是要通过jsonPath获取该简答集合的结构和数据
        if (dataNode.isArray() && dataNode.isObject()) {
            jsonPath = jsonPath + "[" + dataNode.getGlobalPos() + "]";
        }

        return jsonPath;
    }

    /**
     * 获取所有叶子节点的paramPath
     *
     * @return
     */
    public Set<String> getLeafParamPath() {
        if (MapUtils.isEmpty(mappingDataCollection)) {
            return new HashSet<>();
        }
        Set<String> allLeafParamPathSet = new LinkedHashSet<>();
        for (Map.Entry<String, DataNode> entry : mappingDataCollection.entrySet()) {
            DataNode dataNode = entry.getValue();
            if (isLeafNode(dataNode)) {
                allLeafParamPathSet.add(dataNode.getParamPath());
            }
        }
        return allLeafParamPathSet;
    }

    /**
     * 判断是否为叶子节点
     *
     * @param node
     * @return
     */
    protected boolean isLeafNode(DataNode node) {
        return node != null && !node.isObject();
    }

    /**
     * 判断是否为叶子节点
     *
     * @param paramPath
     * @return
     */
    public boolean isLeafNode(String paramPath) {
        return isLeafNode(getDataNodeByParamPath(paramPath));
    }

    /**
     * 根据子dataNode 获取parent DataNode 的paramPath
     *
     * @param paramPath
     * @return
     */
    public String getParentParamPath(String paramPath) {
        DataNode node = getDataNodeByParamPath(paramPath);
        if (node == null) {
            return null;
        }
        if (StringUtils.isEmpty(node.getParentNodeId())) {
            return null;
        }
        return getDataNode(node.getParentNodeId()).getParamPath();
    }

    /**
     * 根据paramPath 获取dataNode 的zone信息
     *
     * @param paramPath
     * @return
     */
    public Map<String, Integer> getZoneByParamPath(String paramPath) {
        return getDataNodeByParamPath(paramPath).getZone();
    }

    /**
     * 获取totalcount值
     *
     * @param paramPath
     * @return
     */
    public int getTotalCountByParamPath(String paramPath) {
        return getDataNodeByParamPath(paramPath).getTotalCnt();
    }

    /**
     * 是否为根节点
     *
     * @param paramPath
     * @return
     */
    public boolean isRootNode(String paramPath) {
        return CommonConstant.ROOT_NODE.equalsIgnoreCase(getDataNodeByParamPath(paramPath).getNodeName());
    }

    /**
     * 取所有下一层级的paramPath
     *
     * @param paramPath
     * @return
     */
    public List<String> getChildrenParamPath(String paramPath) {
        List<String> nodeIds = getChildrenId(getDataNodeByParamPath(paramPath));
        if (CollectionUtils.isEmpty(nodeIds)) {
            return Lists.newArrayList();
        }
        List<String> paramPathList = new ArrayList<>();
        for (String nodeId : nodeIds) {
            DataNode node = getDataNode(nodeId);
            if (node != null) {
                paramPathList.add(node.getParamPath());
            }
        }
        return paramPathList;
    }


}
